package Direct::RetargetingConditions;

use Mouse;
use Direct::Modern;

use Settings;

use Direct::Model::RetargetingCondition;
use Direct::Model::AbSegmentCondition;
use Direct::Model::BrandSafetyCondition;
use Direct::Model::RetargetingCondition::Manager;

use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::I18n;
use Yandex::HashUtils qw/hash_merge/;
use Yandex::ListUtils qw/xisect/;
use List::MoreUtils qw/any all/;

use JavaIntapi::GenerateObjectIds;

has 'items' => (is => 'ro', isa => 'ArrayRef[Direct::Model::RetargetingCondition]');
has 'total' => (is => 'ro', isa => 'Int');
has 'data'  => (is => 'ro', isa => 'HashRef', init_arg => undef, lazy => 1, default => sub { +{}; });

around BUILDARGS => sub { my ($orig, $class) = (shift, shift); $class->$orig(@_ == 1 ? (items => $_[0]) : @_) };

=head2 manager_class
=head2 manager
=cut

sub manager_class { 'Direct::Model::RetargetingCondition::Manager' }
sub manager { $_[0]->manager_class->new(items => $_[0]->items) }

=head2 WEB_FIELD_NAMES

Название полей в web интерфейсе для расшифровки результатов валидации.

=cut

sub WEB_FIELD_NAMES {(
    condition_name => {field => sprintf('"%s"', iget('Название условия подбора аудитории'))},
    condition_desc => {field => sprintf('"%s"', iget('Описание условия подбора аудитории'))},
    goals => {elements_of_field => iget('целей в правиле условия подбора аудитории')}
)}

my %KEYS_MAP = (id => 'ret_cond_id', ClientID => 'client_id');

=head2 get_by($key, $vals, %options)

По заданному критерию возвращает instance с выбранными условиями ретаргетинга.

Параметры:
    $key -> по какому ключу выбирать условия ретаргетинга: id, client_id

    $vals -> (Int|ArrayRef[Int]); список идентификаторов
    %options:
        limit/offset    -> параметры для постраничной выборки
        total_count     -> при использовании limit/offset также вычислять общее количество элементов
        with_deleted    -> выбирать также удаленные условия
        with_is_used    -> вычислить, используется ли условие
        filter          -> (HashRef); дополнительный фильтр
        fields          -> (ArrayRef); список полей модели, которые выбирать из БД
        type           -> (ArrayRef[Str]); "классический" ретаргетинг (по-умолчанию) ["retargeting"], интересы ["interest"], или все ["retargeting", "interest"]

    Дополнительный фильтр:
        Применимы любые поля из таблицы с условиями ретаргетинга. Алиасы таблиц обязательны.

=cut

sub get_by {
    my ($class, $key, $vals, %options) = @_;

    return $class->new(items => []) if !defined $vals || (ref($vals) eq 'ARRAY' && !@$vals);

    my @shard;
    my %where = %{$options{filter} // {}};
    $key = $KEYS_MAP{$key} if exists $KEYS_MAP{$key};
    if ($key eq 'ret_cond_id') {
        @shard = (ret_cond_id => $vals);
        $where{'rc.ret_cond_id'} = SHARD_IDS;
    } elsif ($key eq 'client_id') {
        @shard = (ClientID => $vals);
        $where{'rc.ClientID'} = SHARD_IDS;
    } else {
        croak "Unknown get_by key: `$key`";
    }

    my $with_is_accessible = 1;
    if (my @fields = @{$options{fields} // []}) {
        $options{fields} = [grep { $_ ne 'is_accessible' } @fields];
        $with_is_accessible = @{$options{fields}} != @fields;
    }

    my (@select_columns, @from_tables);

    push @select_columns,
        Direct::Model::RetargetingCondition->get_db_columns(retargeting_conditions => 'rc', prefix => '', fields => $options{fields});

    push @from_tables,
        'retargeting_conditions rc';

    $where{'rc.is_deleted'} = 0 if !$options{with_deleted};
    ($where{'rc.properties__scheck'} //= {})->{autoretargeting} = 0;

    state $_types = [ qw/retargeting interest/ ];
    $options{type} //= ['retargeting'];
    if (@{xisect($options{type}, $_types)} == @$_types) { # если нужны и ретаргетинги и интересы
        delete $where{'rc.properties__scheck'}->{interest} if defined $where{'rc.properties__scheck'};
    } else {
        ($where{'rc.properties__scheck'} //= {})->{interest} = (any {$_ eq 'interest'} @{$options{type}}) ? 1 : 0;
    }

    unless ($where{retargeting_conditions_type}) {
        $where{retargeting_conditions_type} = ['interests','metrika_goals','dmp','ab_segments','shortcuts'];
    }

    my $calc_found_rows = $options{limit} && $options{total_count} ? 'SQL_CALC_FOUND_ROWS' : '';
    my $ret_cond_rows = get_all_sql(PPC(@shard), [
        sprintf("SELECT $calc_found_rows %s FROM %s", join(', ', @select_columns), join(' ', @from_tables)),
        where => \%where,
        'ORDER BY rc.ret_cond_id',
        ($options{limit} ? (limit => $options{limit}, $options{offset} ? (offset => $options{offset}) : ()) : ()),
    ]);

    my $found_rows = $calc_found_rows ? select_found_rows(PPC(@shard)) : undef;
    my $self = $class->new(items => [], $calc_found_rows ? (total => $found_rows) : ());

    return $self if !@$ret_cond_rows;

    if ($with_is_accessible) {
        my @ret_cond_ids = map { $_->{ret_cond_id} } @$ret_cond_rows;
        my $not_accessible_by = get_hash_sql(PPC(ret_cond_id => \@ret_cond_ids), [
            "SELECT DISTINCT ret_cond_id FROM retargeting_goals", WHERE => {ret_cond_id => SHARD_IDS, is_accessible => 0}
        ]);

        $_->{is_accessible} = !exists $not_accessible_by->{ $_->{ret_cond_id} } for @$ret_cond_rows;
    }

    if ($options{with_is_used}) {
        my @ret_cond_ids = map { $_->{ret_cond_id} } @$ret_cond_rows;
        my $is_used_by = get_hash_sql(PPC(ret_cond_id => \@ret_cond_ids), [
            "(SELECT bret.ret_cond_id, 1 FROM bids_retargeting bret JOIN phrases phr ON phr.pid = bret.pid JOIN campaigns camp ON camp.cid = phr.cid",
            WHERE => {'bret.ret_cond_id' => SHARD_IDS, 'camp.statusEmpty' => 'No'}, ")",
            "UNION",
            "(SELECT bperf.ret_cond_id, 1 FROM bids_performance bperf JOIN phrases phr ON phr.pid = bperf.pid JOIN campaigns camp ON camp.cid = phr.cid",
            WHERE => {'bperf.ret_cond_id' => SHARD_IDS, 'camp.statusEmpty' => 'No', 'bperf.is_deleted' => 0}, ")",
            "UNION",
            "(SELECT mbret.ret_cond_id, 1 FROM mediaplan_bids_retargeting mbret",
            WHERE => {'mbret.ret_cond_id' => SHARD_IDS}, ")",
        ]);

        @ret_cond_ids = grep { !$is_used_by->{$_} } @ret_cond_ids; # Отфильтруем те, что не используются в группах выше
        hash_merge $is_used_by, get_hash_sql(PPC(ret_cond_id => \@ret_cond_ids), [
            "SELECT rmv.ret_cond_id, 1 FROM retargeting_multiplier_values rmv", WHERE => {'rmv.ret_cond_id' => SHARD_IDS},
        ]);

        $_->{is_used} = !!$is_used_by->{ $_->{ret_cond_id} } for @$ret_cond_rows;
    }

    my $_cache;
    for my $row (@$ret_cond_rows) {
        if (defined $row->{retargeting_conditions_type} && $row->{retargeting_conditions_type} eq 'ab_segments') {
            push @{$self->items}, Direct::Model::AbSegmentCondition->from_db_hash($row, \$_cache);
        } elsif (defined $row->{retargeting_conditions_type} && $row->{retargeting_conditions_type} eq 'brandsafety') {
            push @{$self->items}, Direct::Model::BrandSafetyCondition->from_db_hash($row, \$_cache)
        } else {
            push @{$self->items}, Direct::Model::RetargetingCondition->from_db_hash($row, \$_cache);
        }

    }

    return $self;
}

=head2 items_by($key)

Возвращает структуру с условиями ретаргетинга, вида:
    $key //eq 'id' => {$ret_cond1->id => $ret_cond1, $ret_cond2->id => $ret_cond2, ...};

=cut

sub items_by {
    my ($self, $key) = @_;

    $key //= 'id';
    $key = $KEYS_MAP{$key} if exists $KEYS_MAP{$key};

    my %result;
    if ($key eq 'ret_cond_id') {
        $result{$_->id} = $_ for @{$self->items};
    } elsif ($key eq 'client_id') {
        push @{$result{ $_->client_id }}, $_ for @{$self->items};
    } else {
        croak "Unknown items_by key: `$key`";
    }

    return \%result;
}

=head2 prepare_create

Подготовка списка условий ретаргетинга к созданию.

=cut

sub prepare_create {
    my ($self) = @_;

    for my $ret_cond (@{$self->items}) {
        $ret_cond->last_change('now');
    }

    return $self;
}

=head2 create

Создание списков условий ретаргетинга в БД.

=cut

sub create {
    my ($self) = @_;

    my %ids_by_client_id;
    $ids_by_client_id{$_->client_id}++ for @{$self->items};
    $ids_by_client_id{$_} = JavaIntapi::GenerateObjectIds->new(object_type => "retargeting_condition",
            count => $ids_by_client_id{$_}, client_id => $_)->call() for keys %ids_by_client_id; 
    $_->id(shift @{$ids_by_client_id{$_->client_id}}) for @{$self->items};

    $self->prepare_create();
    $self->manager->create();

    return;
}

=head2 prepare_update

Подготовка списка условий ретаргетинга к обновлению.

=cut

sub prepare_update {
    my ($self) = @_;

    for my $ret_cond (@{$self->items}) {
        if ($ret_cond->is_condition_changed || !$ret_cond->is_accessible) {
            $ret_cond->do_bs_sync_adgroups(1);
            $ret_cond->do_bs_sync_multipliers(1);
            $ret_cond->do_update_adgroups_last_change(1);

            $ret_cond->is_condition_changed(1); # Для пересохранения целей (is_accessible установится в `true`)
        }

        if ($ret_cond->is_condition_name_changed || $ret_cond->is_condition_desc_changed) {
            # Special for API
            $ret_cond->do_update_adgroups_last_change(1);
        }

        $ret_cond->last_change('now') if $ret_cond->is_changed;
    }

    return $self;
}

=head2 update

Обновление списка условий ретаргетинга в БД.

=cut

sub update {
    my ($self) = @_;

    $self->prepare_update();
    $self->manager->update();

    return;
}

=head2 prepare_delete

Подготовка списка условий ретаргетинга к удалению.

=cut

sub prepare_delete {
    my ($self) = @_;

    for my $ret_cond (@{$self->items}) {
        $ret_cond->is_changed(0);
        $ret_cond->is_deleted(1);
        $ret_cond->last_change('now');
        $ret_cond->do_clear_goals(1);
    }

    return $self;
}

=head2 delete

Удаление списка условий ретаргетинга в БД.

=cut

sub delete {
    my ($self) = @_;

    $self->prepare_delete();
    $self->manager->update();

    return;
}

1;
