package Direct::Retargetings;

use Mouse;

use Direct::Modern;

use Settings;

use Direct::Errors::Messages;
use Direct::Model::Retargeting;
use Direct::Model::TargetInterest;
use Direct::Model::Retargeting::Manager;
use Direct::TargetingCategories;
use Direct::RetargetingConditionIntAPIClient qw/
    create_retargeting_condition
    update_retargeting_condition
    validate_create_retargeting_condition
    validate_update_retargeting_condition
/;
use PhrasePrice qw/validate_cpm_price/;


use Yandex::DBShards;
use Yandex::DBTools;
use Yandex::I18n;
use Yandex::HashUtils qw/hash_cut/;
use List::Util qw/first/;
use Yandex::ListUtils qw/xisect/;
use List::MoreUtils qw/any uniq all/;

use LogTools qw//;
use MailNotification qw/mass_mail_notification/;
use ShardingTools qw/choose_shard_param/;
use Primitives qw//;
use PrimitivesIds qw/get_clientid/;
use JavaIntapi::GenerateObjectIds;

our @EXPORT = qw/
    validate_socdem_retargetings_for_cpm_adgroup
    save_socdem_retargetings_for_cpm_adgroup
/;

# Максимальное число ретаргетингов в группе охватного продукта
our $MAX_RETARGETINGS_IN_CPM_ADGROUP = 1;

has 'items' => (is => 'ro', isa => 'ArrayRef[Direct::Model::Retargeting]');
has 'total' => (is => 'ro', isa => 'Int');
has 'data'  => (is => 'ro', isa => 'HashRef', init_arg => undef, lazy => 1, default => sub { +{}; });

around BUILDARGS => sub { my ($orig, $class) = (shift, shift); $class->$orig(@_ == 1 ? (items => $_[0]) : @_) };

=head2 manager_class
=head2 manager
=cut

sub manager_class { 'Direct::Model::Retargeting::Manager' }
sub manager { $_[0]->manager_class->new(items => $_[0]->items) }

=head2 WEB_FIELD_NAMES

Название полей в web интерфейсе для расшифровки результатов валидации.

=cut

sub WEB_FIELD_NAMES {(
    price_context       => {field => sprintf('"%s"', iget('Цена на сети'))},
    autobudget_priority => {field => sprintf('"%s"', iget('Приоритет автобюджета'))},
)}

my %KEYS_MAP = (cid => 'campaign_id', gid => 'adgroup_id', pid => 'adgroup_id', id => 'ret_id');

=head2 get_by($key, $vals, %options)

По заданному критерию возвращает instance с выбранными списками ретаргетинга

Параметры:
    $key -> по какому ключу выбирать ретаргетинг: id/ret_cond_id/adgroup_id/campaign_id

    $vals -> (Int|ArrayRef[Int]); список идентификаторов
    %options:
        limit/offset   -> параметры для постраничной выборки
        total_count    -> при использовании limit/offset также вычислять общее количество элементов
        filter         -> (HashRef); дополнительный фильтр
        with_ret_conds -> выбирать данные вместе с условием ретаргетинга
        type           -> (ArrayRef[Str]); "классический" ретаргетинг (по-умолчанию) ["retargeting"], интересы ["interest"], или все ["retargeting", "interest"]

    Дополнительный фильтр:
        Применимы любые поля из таблицы со списками ретаргетинга. Алиасы таблиц обязательны.

=cut

sub get_by {
    my ($class, $key, $vals, %options) = @_;

    return $class->new(items => []) if !defined $vals || (ref($vals) eq 'ARRAY' && !@$vals);

    my $straight_join = "STRAIGHT_JOIN";

    my @shard;
    my %where = %{$options{filter} // {}};
    $key = $KEYS_MAP{$key} if exists $KEYS_MAP{$key};
    if ($key eq 'adgroup_id') {
        @shard = (pid => $vals);
        $where{'br.pid'} = SHARD_IDS;
    } elsif ($key eq 'campaign_id') {
        @shard = (cid => $vals);
        $where{'g.cid'} = SHARD_IDS;
        $straight_join = "";
    } elsif ($key eq 'ret_cond_id') {
        @shard = (ret_cond_id => $vals);
        $where{'br.ret_cond_id'} = SHARD_IDS;
    } elsif ($key eq 'ret_id') {
        @shard = $options{shard} ? (shard => $options{shard}) : choose_shard_param(\%options, [qw/ClientID/]);
        $where{'br.ret_id'} = $vals;
    } else {
        croak "Unknown get_by key: `$key`";
    }

    my $target_category_by_import_id;
    my $need_interests = 0;
    state $_types = [ qw/retargeting interest/ ];

    $options{type} //= ['retargeting'];
    ($where{'rc.properties__scheck'} //= {})->{autoretargeting} = 0;
    if (@{xisect($options{type}, $_types)} == @$_types) { # если нужны и ретаргетинги и интересы
        delete $where{'rc.properties__scheck'}->{interest} if defined $where{'rc.properties__scheck'};
        $need_interests = 1;
    } else {
        $need_interests = ($where{'rc.properties__scheck'} //= {})->{interest} = (any {$_ eq 'interest'} @{$options{type}}) ? 1 : 0;
    }
    if ($need_interests) {
        $target_category_by_import_id = Direct::TargetingCategories->get_rmp_interests->items_by('import_id');
        $options{with_ret_conds} = 1;
    }

    my (@select_columns, @from_tables);

    # всегда используем get_db_columns от Retargeting, т.к. колонки единые у обеих моделей (Retargeting и TargetInterest)
    push @select_columns,
        Direct::Model::Retargeting->get_db_columns(bids_retargeting => 'br', prefix => 'br_'),
        'g.cid AS br_campaign_id';

    push @from_tables,
        'bids_retargeting br',
        'JOIN phrases g ON (g.pid = br.pid)',
        'JOIN retargeting_conditions rc ON (rc.ret_cond_id = br.ret_cond_id)';

    if ($options{with_ret_conds}) {
        push @select_columns, Direct::Model::RetargetingCondition->get_db_columns(retargeting_conditions => 'rc', prefix => 'rc_');
    }

    if (any { /^c\./ } keys %where) {
        push @from_tables, 'JOIN campaigns c ON (c.cid = g.cid)';
    }

    my $calc_found_rows = $options{limit} && $options{total_count} ? 'SQL_CALC_FOUND_ROWS' : '';
    my $ret_rows = get_all_sql(PPC(@shard), [
        sprintf("SELECT $straight_join $calc_found_rows %s FROM %s", join(', ', @select_columns), join(' ', @from_tables)),
        where => \%where,
        'ORDER BY br.ret_id',
        ($options{limit} ? (limit => $options{limit}, $options{offset} ? (offset => $options{offset}) : ()) : ()),
        
    ]);

    my $found_rows = $calc_found_rows ? select_found_rows(PPC(@shard)) : undef;
    my $self = $class->new(items => [], $calc_found_rows ? (total => $found_rows) : ());

    my $cache;
    for my $row (@$ret_rows) {
        if ($options{with_ret_conds}) {
            my $retargeting_condition = Direct::Model::RetargetingCondition->from_db_hash($row, \$cache, prefix => 'rc_');
            if (any {$_ eq 'interest'} @{ $retargeting_condition->properties }) {
                my $target_interest = Direct::Model::TargetInterest->from_db_hash($row, \$cache, prefix => 'br_');
                $target_interest->ret_cond($retargeting_condition);
                my $target_category = $target_category_by_import_id->{$target_interest->ret_cond->condition->[0]->goals->[0]->goal_id};
                $target_interest->target_category_id($target_category->id);
                $target_interest->category_name($target_category->name);
                push @{$self->items}, $target_interest;
            } else {
                my $retargeting = Direct::Model::Retargeting->from_db_hash($row, \$cache, prefix => 'br_');
                $retargeting->ret_cond($retargeting_condition);
                push @{$self->items}, $retargeting;
            }
        } else {
            push @{$self->items}, Direct::Model::Retargeting->from_db_hash($row, \$cache, prefix => 'br_');
        }
    }

    return $self;
}

=head2 items_by($key)

Возвращает структуру со списками ретаргетинга, вида:
    $key //eq 'id' => {$retargeting1->id => $retargeting1, $retargeting2->id => $retargeting2, ...};

=cut

sub items_by {
    my ($self, $key) = @_;

    $key //= 'id';
    $key = $KEYS_MAP{$key} if exists $KEYS_MAP{$key};

    my %result;
    if ($key eq 'ret_id') {
        $result{$_->id} = $_ for @{$self->items};
    } elsif ($key eq 'adgroup_id') {
        push @{$result{ $_->adgroup_id }}, $_ for @{$self->items};
    } elsif ($key eq 'campaign_id') {
        push @{$result{ $_->campaign_id }}, $_ for @{$self->items};
    } elsif ($key eq 'ret_cond_id') {
        push @{$result{ $_->ret_cond_id }}, $_ for @{$self->items};
    } else {
        croak "Unknown items_by key: `$key`";
    }

    return \%result;
}

=head2 prepare_create

Подготовка списков ретаргетинга к созданию.

=cut

sub prepare_create {
    my ($self) = @_;

    for my $retargeting (@{$self->items}) {
        $retargeting->last_change('now');
        $retargeting->status_bs_synced('No');

        # Если ретаргетинг добавляется в группу не-черновик, то сбросим статус синхронизации на группе
        if ($retargeting->adgroup->status_moderate ne 'New') {
            $retargeting->do_bs_sync_adgroup(1);
        }

        $retargeting->do_update_adgroup_last_change(1);
    }

    return $self;
}

=head2 create

Создание списков ретаргетинга в БД.

=cut

sub create {
    my ($self) = @_;

    # Выдача идентификаторов
    my $ids = JavaIntapi::GenerateObjectIds->new(object_type => 'retargeting',
            count => scalar(@{$self->items}))->call();
    $_->id(shift @$ids) for @{$self->items};

    $self->prepare_create();
    $self->prepare_logging('create');
    $self->manager->create();
    $self->do_logging();

    return;
}

=head2 prepare_update

Подготовка списков ретаргетинга к обновлению.

=cut

sub prepare_update {
    my ($self) = @_;

    for my $retargeting (@{$self->items}) {
        # Время последнего обновления ретаргетинга
        if (any { $retargeting->$_ } map { "is_${_}_changed" } qw/
            ret_cond_id price_context autobudget_priority is_suspended
        /) {
            $retargeting->last_change('now');
        }

        # Статус синхронизации группы (сбрасываем при условии, что группа не черновик)
        if (
            (any { $retargeting->$_ } map { "is_${_}_changed" } qw/ret_cond_id is_suspended/) &&
            $retargeting->adgroup->status_moderate ne 'New'
        ) {
            $retargeting->do_bs_sync_adgroup(1);
        }

        # Время последнего изменения группы
        if (any { $retargeting->$_ } map { "is_${_}_changed" } qw/ret_cond_id is_suspended/) {
            $retargeting->do_update_adgroup_last_change(1);
        }

        # При изменении цен/приоритета_автобюджета ретаргетинг отправляется отдельным транспортом
        if (any { $retargeting->$_ } map { "is_${_}_changed" } qw/price_context autobudget_priority/) {
            $retargeting->status_bs_synced('No');
        }

    }

    return $self;
}

=head2 update

Обновление списков ретаргетинга в БД.

Параметры:
    %options:
        log_price__type -> тип логирования для цен

=cut

sub update {
    my ($self, %options) = @_;

    $self->prepare_update();
    $self->prepare_logging('update', %{hash_cut \%options, qw/log_price__type/});
    $self->manager->update();
    $self->do_logging();

    return;
}

=head2 prepare_delete

Подготовка списков ретаргетинга к удалению.

=cut

sub prepare_delete {
    my ($self) = @_;

    for my $retargeting (@{$self->items}) {
        $retargeting->do_update_adgroup_last_change(1);

        if ($retargeting->adgroup->status_moderate ne 'New') {
            $retargeting->do_bs_sync_adgroup(1);
        }
    }

    return $self;
}

=head2 delete

Удаление списков ретаргетинга в БД.

=cut

sub delete {
    my ($self) = @_;

    $self->prepare_delete();
    $self->prepare_logging('delete');
    $self->manager->delete();
    $self->do_logging();

    return;
}

=head2 prepare_logging($action, %params)
=head2 do_logging

Методы для логирования событий (действий).

Параметры:
    $action -> выполненное действие: create/update/delete
    %params:
        log_price__type -> тип логирования для цен
        uid             -> uid пользователя, над которым выполняется операция (по умолчанию берется из %LogTools::context)

=cut

sub prepare_logging {
    my ($self, $action, %params) = @_;

    my %log_context = %LogTools::context;
    my $uid = $params{uid} || $log_context{uid};

    croak "undefined uid" if !defined $uid;

    my @adgroup_ids = uniq(map { $_->adgroup_id } @{$self->items});
    my $adgroup_id2campaign = get_hashes_hash_sql(PPC(pid => \@adgroup_ids), [q{
        SELECT g.pid, g.cid, IFNULL(c.currency, 'YND_FIXED') AS currency FROM phrases g JOIN campaigns c ON (g.cid = c.cid)
    }, where => {
        'g.pid' => SHARD_IDS,
    }]);

    my $main_bid_by_gid = Primitives::get_main_banner_ids_by_pids(\@adgroup_ids);

    for my $retargeting (@{$self->items}) {
        my $main_banner_id = $main_bid_by_gid->{$retargeting->adgroup_id};
        my %logprice_tmpl = (
            %{$adgroup_id2campaign->{$retargeting->adgroup_id}}, # cid, pid, currency
            id => $retargeting->id,
            price => 0,
            price_ctx => $retargeting->has_price_context ? $retargeting->price_context : 0,
        );

        if ($action eq 'create') {
            push @{$self->data->{log_price}}, {%logprice_tmpl, type => $params{log_price__type} // 'ret_add'};

            push @{$self->data->{notifications}}, {
                object     => 'adgroup',
                event_type => 'b_retargeting',
                object_id  => $retargeting->adgroup_id,
                old_text   => '',
                new_text   => $retargeting->ret_cond_id,
                uid        => $uid,
            } if $main_banner_id;
        }
        elsif ($action eq 'update') {
            if ($retargeting->is_price_context_changed) {
                push @{$self->data->{log_price}}, {%logprice_tmpl, type => $params{log_price__type} // 'ret_update'};

                push @{$self->data->{notifications}}, {
                    object      => 'phrase',
                    event_type  => 'ph_price_ctx',
                    object_id   => $main_banner_id,
                    old_text    => $retargeting->old->price_context,
                    new_text    => $retargeting->price_context,
                    uid         => $uid,
                } if $main_banner_id;
            }

            push @{$self->data->{notifications}}, {
                object     => 'adgroup',
                event_type => 'b_retargeting',
                object_id  => $retargeting->adgroup_id,
                old_text   => $retargeting->old->ret_cond_id,
                new_text   => $retargeting->ret_cond_id,
                uid        => $uid,
            } if $main_banner_id && $retargeting->is_ret_cond_id_changed;
        }
        elsif ($action eq 'delete') {
            push @{$self->data->{log_price}}, {%logprice_tmpl, type => $params{log_price__type} // 'delete2'};

            push @{$self->data->{notifications}}, {
                object     => 'adgroup',
                event_type => 'ret_delete',
                object_id  => $retargeting->adgroup_id,
                old_text   => $retargeting->ret_cond_id,
                new_text   => '',
                uid        => $uid,
            } if $main_banner_id;
        }
        else {
            croak "Unknown action: $action";
        }
    }

    return;
}

sub do_logging {
    my ($self) = @_;

    LogTools::log_price($self->data->{log_price}) if $self->data->{log_price};
    mass_mail_notification($self->data->{notifications}) if $self->data->{notifications};

    return;
}

=head2 validate_socdem_retargetings_for_cpm_adgroup($user_rets, $campaign)

Выполняет валидацию (без сохранения) соцдема вызовом ручки Int API.
Возвращает результат валидации.

=cut

sub validate_socdem_retargetings_for_cpm_adgroup {
    my ($user_rets, $campaign) = @_;

    my $vr_main = Direct::ValidationResult->new();

    if (scalar @$user_rets > $MAX_RETARGETINGS_IN_CPM_ADGROUP) {
        $vr_main->add_generic(error_LimitExceeded(
            iget("Количество условий подбора аудитории в группе объявлений должно быть не более %d", $MAX_RETARGETINGS_IN_CPM_ADGROUP)
        ));
    }
    return $vr_main unless $vr_main->is_valid;

    my $client_id = get_clientid(cid => $campaign->{cid});
    foreach my $ret (@$user_rets) {
        my $vr = $vr_main->next;
        my $push_error;

        # Проверяем новое значение ставки, если она указана
        if (defined $ret->{price_context}) {
            if ($campaign->{strategy}->{is_autobudget}) {
                $vr->add(price_context => error_InvalidField(
                        iget("Выбранная стратегия не предполагает ручной установки ставок")));
            } else {
                my $price_error = validate_cpm_price($ret->{price_context}, $campaign->{currency});
                if ($price_error) {
                    $vr->add(price_context => error_InvalidField($price_error));
                }
            }
        }

        if ($ret->{ret_cond_id}) {
            $push_error = Direct::RetargetingConditionIntAPIClient::validate_update_retargeting_condition(
                $client_id, $ret);
        } else {
            $push_error = Direct::RetargetingConditionIntAPIClient::validate_create_retargeting_condition(
                $client_id, $ret);
        }
        add_converted_errors($vr, $push_error);
    }

    return $vr_main;
}

=head2 add_converted_errors($vr, $push_error)

Конвертирует ошибки валидации, возвращённые из WEB API, в формат для фронтенда
и добавляет их к validationResult

=cut

sub add_converted_errors {
    my ($vr, $push_error) = @_;

    return if !$push_error;
    if (defined $push_error->{server_error}) {
        $vr->add_generic(error_OperationFailed());
        return;
    }
    return if !defined $push_error->{validation_result} || !defined $push_error->{validation_result}->{errors};

    foreach my $error (@{$push_error->{validation_result}->{errors}}) {
        my $defect = new Direct::Defect(
            type => 'error',
            code => 6000,  # Здесь всегда 6000 как наиболее подходящее по смыслу
            text => $error->{text},  # Текст ошибки в той же локали, что и текущая
            description => $error->{description},  # Подробное описание ошибки в той же локали, что и текущая
            name => $error->{code}  # На эти коды завязывается фронт
        );

        if ($error->{path} eq 'name') {
            $vr->add('condition_name' => $defect);
        } elsif ($error->{path} eq 'conditions') {
            $vr->add('groups' => $defect);
        } else {
            $vr->add_generic($defect);
        }
    }
}

=head2 save_socdem_retargetings_for_cpm_adgroup($user_rets, $prepared_rets, $campaign)

Сохраняет соцдем ретаргетинги через ручку Int API.
ID созданных ретаргетингов будут записаны в $user_rets и $prepared_rets по ключу ret_cond_id

=cut

sub save_socdem_retargetings_for_cpm_adgroup {
    my ($user_rets, $prepared_rets, $campaign) = @_;

    my $client_id = get_clientid(cid => $campaign->{cid});
    foreach my $ret (@$user_rets) {
        if ($ret->{ret_cond_id}) {
            my $push_error = Direct::RetargetingConditionIntAPIClient::update_retargeting_condition($client_id, $ret);
            if ($push_error && defined $push_error->{validation_result} && defined $push_error->{validation_result}->{errors}) {
                die "Validation errors returned when saving socdem retargetings";
            }
        } else {
            my ($push_result, $push_error) = Direct::RetargetingConditionIntAPIClient::create_retargeting_condition($client_id, $ret);
            if ($push_error && defined $push_error->{validation_result} && defined $push_error->{validation_result}->{errors}) {
                die "Validation errors returned when saving socdem retargetings";
            } else {
                $ret->{ret_cond_id} = $push_result;
                foreach my $prepared_ret (@$prepared_rets) {
                    $prepared_ret->{ret_cond_id} ||= $push_result;
                }
            }
        }
    }
}

1;
