package Direct::StaticFilesHash;
use strict;
use warnings;
use feature qw(state);

=head1 NAME

Direct::StaticFilesHash

=head1 SYNOPSIS

    use Direct::StaticFilesHash;
    print Data::Dumper::Dumper( get_static_file_hashsums() ); # возвращает hashref, возможно, пустой

=head1 DESCRIPTION

Модуль нужен для того, чтобы при релизе менялись имена статических файлов. Если их не менять,
браузеры могут использовать для отображения старую статику из кэша, которая может плохо работать
с новым html сайта после релиза. (Для истории: разработали этот модуль после случая с гигантскими
буквами "запустите генератор продаж" на главной странице.)

Как это работает все в целом: после сборки bem запускается скрипт hash_data2_files.pl, который
считает хэш-сумму от статики (calculate_static_files_hash) и записывает в data2/hashsums.json
(save_static_file_hashsums).

Внутри процесса apache при компиляции этого модуля содержимое data2/hashsum загружается в переменную
и делается доступным для шаблонов в vars->{static_file_hashsums} (для этого Direct::Template вызывает
get_static_file_hashsums). bemtree/bemhtml исходят из этого значения, когда формируют ссылки на статику.

=cut

use base 'Exporter';
our @EXPORT = qw(
    get_static_file_hashsums
    get_static_exp_files_hash
);

use Yandex::LiveFile::JSON;

use Digest::MD5 qw( md5_hex );
use File::Slurp qw( read_file write_file );
use File::Spec;
use JSON;
use Try::Tiny;

use Settings;

my %FILE_UNIONS = (
    'desktop:ie' => qr{^data3/desktop\.bundles/direct/_direct\.ie.*\.css$},
);

my $DNA_HASHSUMS_FILE = "$Settings::DNA_ROOT/build/hashsums.json";

=head1 SUBROUTINES/METHODS

=head2 get_static_files_hash

=cut

sub get_static_file_hashsums {
    my (%opt) = @_;
    my $source = $opt{source} // 'data3';
    my $HASHSUM_FILE_PATH = "$source/hashsums.json";

    state $static_file_hashsums = {};

    if (!defined $static_file_hashsums->{$source}) {
        if ( -f "$Settings::ROOT/$HASHSUM_FILE_PATH" ) {
            try {
                my $file_content = read_file("$Settings::ROOT/$HASHSUM_FILE_PATH");
                $static_file_hashsums->{$source} = decode_json($file_content);
            } catch {
                warn "error reading static file hashsums: $_";
                $static_file_hashsums->{$source} = {};
            };
        } else {
            warn "could not load static file hashsums: $HASHSUM_FILE_PATH doesn\'t exist";
            $static_file_hashsums->{$source} = {};
        }
    }

    return {%{$static_file_hashsums->{$source}}, %{dna_hashsums()}};
}

=head2 get_static_exp_files_hash

=cut

sub get_static_exp_files_hash {
    my ($exp, %opt) = @_;
    my $source = $opt{source} // 'data3';
    my $file_path = "$source/" . $exp . "_hashsums.json";

    state $static_file_exp_hashsums = {};

    if (!defined $static_file_exp_hashsums->{$file_path}) {
        if ( -f "$Settings::ROOT/$file_path" ) {
            try {
                my $file_content = read_file("$Settings::ROOT/$file_path");
                $static_file_exp_hashsums->{$file_path} = decode_json($file_content);
            } catch {
                warn "error reading static file hashsums: $_";
                $static_file_exp_hashsums->{$file_path} = {};
            };
        } else {
            warn "could not load static file hashsums: " . $file_path . "doesn\'t exist";
            $static_file_exp_hashsums->{$file_path} = {};
        }
    }

    return {%{$static_file_exp_hashsums->{$file_path}}, %{dna_exp_hashsums($exp)}};
}

=head2 dna_hashsums

=cut

sub dna_hashsums {
    return {} unless -f $DNA_HASHSUMS_FILE;

    state $live_file;
    $live_file //= Yandex::LiveFile::JSON->new(filename => $DNA_HASHSUMS_FILE);
    
    return $live_file->data();
}

=head2 dna_exp_hashsums

=cut

sub dna_exp_hashsums {
    my ($exp) = @_;
    my $file_path = "$Settings::DNA_ROOT/build/" . $exp . "_hashsums.json";

    return {} unless -f $file_path;

    state $live_files = {};
    $live_files->{$file_path} //= Yandex::LiveFile::JSON->new(filename => $file_path);

    return $live_files->{$file_path}->data();
}

=head2 save_static_file_hashsums

=cut

sub save_static_file_hashsums {
    my $STATIC_FILES_PATHS = shift;
    my $HASHSUM_FILE_PATH = shift;

    my %hashsums;

    foreach my $filename ( map { glob("$Settings::ROOT/$_/*" ) } @$STATIC_FILES_PATHS ) {
        my $relpath = File::Spec->abs2rel( $filename, $Settings::ROOT );
        my $hashsum = md5_hex( read_file($filename) );
        $hashsums{$relpath} = $hashsum;
    }

    while ( my ( $union_name, $re ) = each %FILE_UNIONS ) {
        my @union_hashsums = map { "$hashsums{$_} $_" } grep { $_ =~ /$re/ } sort keys %hashsums;
        $hashsums{"union:$union_name"} = md5_hex(@union_hashsums);
    }

    write_file( "$Settings::ROOT/$HASHSUM_FILE_PATH", { atomic => 1 }, encode_json(\%hashsums) );
    return;
}

1;
