package Direct::Storage;

=pod

    $Id$

=head1 NAME

Direct::Storage - работа с МДС в Директе

=head1 SYNOPSYS

    use Direct::Storage;

    my $storage = Direct::Storage->new();
    my $mds_url = $feed_file->url;
    $storage->delete_file('direct-files', ClientID => $client_id, filename => $file_hash);

=head1 DESCRIPTION

Модуль для работы с МДС в Директе
Представляет из себя интерфейс к хранилищу файлов

Сохраняет файл в МДС
Полученный от МДС ключ и другие метаданные помещает в таблицу ppc.mds_metadata

Извне читать из mds_metadata можно, но не рекомендуется
Писать в mds_metadata из-за пределов этого модуля запрещается


Про имена файлов:
С Direct::Storage можно работать в двух режимах:
1. Обращаемся к файлу по хешу от его содержимого
2. Фиксированные имена файлов

Код для первого случая:
$storage->save($type, $content, ClientID => $clid);

Для второго случая:
$storage->save($type, $content, ClientID => $clid, filename => 'fixed name for this file');

Можно или нет для данного $type использовать имена - определяется в $Direct::Storage::Types::MDS_FILE_TYPES{$type}{custom_name}
или $Direct::Storage::Types::MDS_FILE_TYPES{$type}{hashed_name}

Действует правило, что нельзя сохранить два разных файла под одним именем. Правило действует в пределах $client_id/$type

Про ClientID:
Если файл (тип файлов) является не пользовательским, а обще-директовским, его можно сохранять не указывая ClientID:
$storage->save($type, $content); # filename считаем по хешу, ClientID - отсутствует
Для этих файлов метаданныхе хранятся в ppcdict

Можно ли для данного $type не указывать ClientID - определяется $Direct::Storage::Types::MDS_FILE_TYPES{$type}{empty_client_id}

Для get():
filename => '...' указывается всегда. Для файлов с custom_name это имя, заданное при сохранении; для прочих файлов - сгенерированное имя.
ClientID => $nnn - для файлов, у которых не выставлен empty_client_id

=head1 METHODS

=cut

use Direct::Modern;

use Settings;

use Yandex::MDS;
use Yandex::DBTools;
use Yandex::DBShards qw/get_new_id/;
use Yandex::Validate qw/is_valid_id/;
use Direct::Storage::File;
use Yandex::LiveFile;
use HashingTools qw/md5_base64ya/;
use Yandex::Validate qw/is_valid_int/;
use Direct::Storage::Types;
use Carp;
use PrimitivesIds qw/get_clientid/;
use EnvTools qw/is_beta/;
use Yandex::Log;
use Yandex::Trace;

=head2 $MDS_AUTHORIZATION_FILE

Файл, в котором хранится авторизационный ключ для МДС

=cut

our $MDS_AUTHORIZATION_FILE ||= '/etc/direct-tokens/mds-auth.txt';

=head2 new

my $storage = Direct::Storage->new(%opt)

%opt:
    * Опции для Yandex::MDS
        get_host, put_host -- хосты для чтения и записи
        authorization -- ключ авторизациии
        namespace -- неймспейс

    Указать хост для MDS можно:
     - с помощью опций get_host и put_host
     - с помощью глобальной переменной $Yandex::MDS::MDS_GET_HOST (и MDS_PUT_HOST)

=cut

sub new
{
    my $class = shift;
    state $auth_file = Yandex::LiveFile->new(filename => $MDS_AUTHORIZATION_FILE);
    my (%opt) = @_;
    my $auth = $opt{authorization} // $auth_file->data;
    # TODO: http_parallel_request подвержен http response splitting, нужно удалять \n из значений заголовков
    $auth =~ s!\s+!!g;
    my $self = {
        mds => Yandex::MDS->new(
            ($opt{get_host} ? (get_host => $opt{get_host}) : () ),
            ($opt{put_host} ? (put_host => $opt{put_host}) : () ),
            authorization => $auth,
            namespace => $opt{namespace} // $Settings::MDS_NAMESPACE,
        ),
    };
    $self->{mds_host} = $self->{mds}->get_host;
    return bless $self, $class;
}

=head2 save

my $feed_file = $storage->save($type, $data, ClientID => $client_id, filename => $filename, content_filename => '/tmp/uploaded_file' );

Загрузка в МДС и сохранение метаданных

Можно передать undef в качестве $data и имя файла в content_filename;
передавать определённое $data и content_filename одновременно нельзя.

Внутри MDS сохраняется по пути $type/$client_id/$filename

$filename должен быть уникален в пределах $type/$client_id

Если filename не указан - используется md5_base64ya($data)

Возвращает объект Direct::Storage::File
(см. описание в get_file)

=cut

sub save
{
    my ($self, $type, $content, %opt) = @_;

    $self->_get_clientid(\%opt);

    $self->_validate_path_parts($type, $opt{ClientID});

    my $client_id = delete $opt{ClientID};

    my $filename = delete $opt{filename};

    my $size = delete $opt{size};

    my $content_filename = delete $opt{content_filename};

    if (%opt) {
        croak "unknown options: ".join(' ', keys %opt);
    }

    if ( defined $content && defined $content_filename ) {
        croak "data and content_filename are mutually exclusive";
    }

    if ( ! defined $content && ! defined $content_filename ) {
        croak "Must pass one of data or content_filename";
    }

    my $content_hash;
    if (mds_check_type_trait($type, 'hashed_name') && $filename) { # если выполняется первое, но не второе — нестандартно — упадёт с ошибкой позже
        $content_hash = $filename;
    } elsif (Encode::is_utf8(ref $content ? $$content : $content)) {
        $content_hash = md5_base64ya(Encode::encode_utf8(ref $content ? $$content : $content));
    } else {
        $content_hash = md5_base64ya(ref $content ? $$content : $content);
    }

    if ($filename) {
        if (!(mds_check_type_trait($type, 'custom_name') || mds_check_type_trait($type, 'hashed_name'))) {
            croak "custom filenames are not allowed for type '$type'";
        }
    } else {
        if (mds_check_type_trait($type, 'custom_name') || mds_check_type_trait($type, 'hashed_name')) {
            croak "filename required for type '$type'";
        }
        $filename = $content_hash;
    }

    my $db = $self->_db($type, $client_id);
    if (my $file = $self->get_file($type, ClientID => $client_id, filename => $filename, _hash => $content_hash)) {
        if (mds_check_type_trait($type, 'temporary')) {
            # файл существует, обновляем время сохранения файла, чтобы не удалить во время чистки устаревших файлов
            do_update_table($db, 'mds_metadata', { 'create_time__dont_quote' => 'NOW()' }, where => { id => $file->_id });
        }
        return $file;
    }

    # проверку что $client_id может быть пустым сделали раньше, в get_file (-> в _validate_path_parts)
    $client_id //= 1; # значение 1 - фиктивное, для файлов с empty_client_id 
    
    my $path = join "/", ($type, $client_id, $content_hash);

    my $key;
    if ( defined $content ) {
        $key = $self->{mds}->upload($path, $content);;
    } elsif ( defined $content_filename ) {
        $key = $self->{mds}->upload_file( $content_filename, $path );
    } else {
        die "One of data or content_filename must be present";
    }

    my %metadata = (
        filename => $content_hash,
        file_imprint => $content_hash,
        storage_host => $self->{mds_host},
        type => $type,
        mds_key => $key,
        ClientID => $client_id,
        size => $size // length(Encode::encode('utf8', $content)),
    );
    my $mds_id = get_new_id('mds_id');
    die "could not get new mds_id" unless $mds_id;
    $metadata{id} = $mds_id;
    # а теперь этот id может поменяться из-за on duplicate key update
    my $insert_result = do_insert_into_table($db, 'mds_metadata', \%metadata, on_duplicate_key_update => 1, key => 'id');
    if ($insert_result) { # do_insert_into_table обещает возвращать id добавленной/обновленной записи, но по факту возвращает 0 при вставке без autoincrement
        $metadata{id} = $mds_id = $insert_result;
    }
    if (mds_check_type_trait($type, 'custom_name')) {
        # удаляем предыдущие записи с таким именем
        do_sql($db, [
            "DELETE mcn
            FROM mds_metadata mm
            JOIN mds_custom_names mcn on mcn.mds_id = mm.id",
            where => {
                'mm.type' => $type,
                'mm.ClientID' => $client_id,
                'mcn.filename' => $filename,
            }
        ]);
        do_replace_into_table($db, 'mds_custom_names', { mds_id => $mds_id, filename => $filename });
        $metadata{filename} = $filename;
    }
    $metadata{_id} = $mds_id;
    # _debug({ save => [$type, { filename => $filename, ClientID => $client_id }], meta => \%metadata });
    return Direct::Storage::File->new({
        %metadata,
        _mds_key => $metadata{mds_key},
        _mds => $self->{mds},
        _storage_host => $self->{mds_host},
    });
}

=head2 get_file

my $info = $storage->get_file($type, ClientID => $client_id, filename => $filename, %opt);

Получить файл

Возвращает объект типа Direct::Storage::File, с полями:

$info->size
$info->ClientID
$info->type
$info->create_time
$info->filename
$info->content # заполняется лениво при первом обращении

Если файла нет - undef

%opt:
    fetch_content => 1 -- сразу же подгрузить содержимое файла

=cut

sub get_file
{
    my ($self, $type, %opt) = @_;

    $self->_get_clientid(\%opt);
    
    $self->_validate_path_parts($type, $opt{ClientID});

    my $client_id = $opt{ClientID} // 1; # see save()
    my $filename = $opt{filename};

    my $meta;
    if (mds_check_type_trait($type, 'custom_name')) {
        $meta = $self->_meta_for_custom_name($type, $client_id, $filename, $opt{_hash});
    } else {
        $meta = get_one_line_sql($self->_db($type, $client_id),
            "select id as _id, mds_key as _mds_key, size, create_time, storage_host as _storage_host, file_imprint as filename
            from mds_metadata
            where type = ? and ClientID = ? and file_imprint = ?",
            $type, $client_id, $filename
        );
    }

    return undef unless $meta;

    my $file = Direct::Storage::File->new({
        %$meta,
        ClientID => $client_id,
        type => $type,
        _mds => $self->{mds},
    });
    # _debug({ get_file => [$type, \%opt], id => $file->_id });
    if ($opt{fetch_content}) {
        $file->content();
    }
    return $file;
}

=head2 _meta_for_custom_name

Получить описание файла из базы в случае, когда у файла есть специальное имя

$hash передаётся при сохранении файла, в таком случае мы ищем файл по имени и хешу

=cut

sub _meta_for_custom_name
{
    my ($self, $type, $client_id, $filename, $hash) = @_;
    my $meta = get_one_line_sql($self->_db($type, $client_id), [
        "select mm.id as _id, mm.mds_key as _mds_key, size, create_time, storage_host as _storage_host, mcn.filename,
        mm.file_imprint as file_hash
        from mds_metadata mm join mds_custom_names mcn on mm.id = mcn.mds_id
        ",
        where => {
            'mm.type' => $type,
            'mm.ClientID' => $client_id,
            'mcn.filename' => $filename,
            ( $hash ? ( 'mm.file_imprint' => $hash ) : () ),
        },
    ]);

    return $meta;
}

=head2 delete_file

$storage->delete_file($type, ClientID => $client_id, filename => $filename);

Удалить файл
если такого файла нет - умирает

=cut

sub delete_file
{
    my ($self, $type, %opt) = @_;
    
    $self->_get_clientid(\%opt);

    my $client_id = $opt{ClientID};
    my $filename = $opt{filename};

    my $file = $opt{file} // $self->get_file($type, ClientID => $client_id, filename => $filename);

    unless ($file) {
        croak "can not delete: file not found: $type/$client_id/$filename";
    }
    # _debug({ delete_file => [$type, \%opt], id => $file->_id });

    if (mds_check_type_trait($type, 'custom_name')) {
        do_delete_from_table($self->_db($type, $client_id), 'mds_custom_names', where => {
            mds_id => $file->_id,
            filename => $filename,
        });
        return;
    }
    # _debug({ delete_data => {%$file} });
    
    # ситуация, когда есть данные в хранилище, но мы о них не знаем, лучше
    # чем когда мы знаем о данных, которых нет в хранилище
    # поэтому сначала delete в базе, потом в хранилище
    
    do_delete_from_table($self->_db($type, $client_id), 'mds_metadata', where => {
        id => $file->_id,
    });
    $self->{mds}->delete($file->_mds_key);
}

=head2 _validate_path_parts

Проверить $type и $client_id

$type должен быть в %Direct::Storage::Types::MDS_FILE_TYPES

$client_id должен быть валидным ид

В случае если валидация не прошла - умирает

=cut

sub _validate_path_parts
{
    my ($self, $type, $client_id) = @_;
    $type //= '';

    unless (is_valid_mds_type($type)) {
        croak "unknown file type '$type'";
    }

    if (mds_check_type_trait($type, 'empty_client_id')) {
        # empty_client_id => 1 -- Никогда нельзя указывать ClientID, при указании - ошибка
        if (defined $client_id) {
            croak "ClientID=$client_id not allowed for type '$type'";
        }
        else {
            # не проверяем is_valid_id($client_id)
            return 1;
        }
    } else {
        # empty_client_id => 0 -- Обязательно нужно указывать ClientID, без него - ошибка (умираем)
        if (!defined $client_id) {
            croak "empty ClientID not allowed for type '$type'";
        }
    }
    
    $client_id //= '';

    unless (is_valid_id($client_id)) {
        croak "invalid client_id '$client_id'";
    }
}


=head2 remove_old_files

my $removed = $storage->remove_old_files($shard, 'type_name', 24 * 60 * 60, %opt);

%opt:
    limit => N - удалять не более чем N записей
    storage_host => 'storage-int.mdst.yandex.net' — удалять только записи, ссылающиеся на конкретный хост Стораджа

Функция удаляет файлы старше указанного возраста в секундах из ppc.mds_metadata и Стораджа
Возвращает массив хешей, описывающих удаленные файлы:
{
    id => - mds_metadata.id
    shard => номер шарда
    ClientID => 
    filename =>
    mds_key =>
    create_time =>
}

=cut

sub remove_old_files
{
    my ($self, $shard, $type, $max_age, %opt) = @_;
    my $limit = delete $opt{limit};
    my $storage_host = delete $opt{storage_host};
    my $profile = Yandex::Trace::new_profile('storage:remove_old_files');
    
    unless (is_valid_mds_type($type)) {
        croak "invalid type '$type'";
    }
    unless (is_valid_int($max_age, 0)) {
        croak "invalid max age '$max_age'";
    }

    my $type_has_custom_name = mds_check_type_trait($type, 'custom_name');

    my $db = mds_check_type_trait($type, 'empty_client_id') ? PPCDICT : PPC(shard => $shard);

    if (%opt) {
        croak "invalid options: ".(join ", ", keys %opt);
    }

    my @removed;
    my $to_remove = get_all_sql($db, [
        "select id, $shard as shard, ClientID, file_imprint, mds_key, create_time from mds_metadata",
        where => {
            type => $type,
            create_time__lt__dont_quote => "now() - interval $max_age second",
            ($storage_host ? (storage_host => $storage_host) : ()),
        },
        ( $limit ? ("LIMIT $limit") : () )
    ]);

    for my $row (@$to_remove) {
        if (mds_check_type_trait($type, 'custom_name')) {
            do_delete_from_table($db, 'mds_custom_names', where => { mds_id => $row->{id} });
        }
        my $is_removed = do_delete_from_table($db, 'mds_metadata', where => {
            id => $row->{id},
            type => $type,
            create_time__lt__dont_quote => "now() - interval $max_age second",
        });
        
        if ($is_removed) {
            eval {
                $self->{mds}->delete($row->{mds_key});
            };
            push @removed, $row;
        }

    }
    return \@removed;
}

=head2 remove_dangling_files($db, $type)

Удалить файлы, у которых должно быть, но отсутствует имя файла
$db - база данных (PPCDICT или PPC(shard => $shard))
$type - тип файлов

возвращает массив хешей - удаленные записи
%opt:
    ClientID => N -- только для указанного клиента
    limit => N -- удалять не более чем N файлов

=cut

sub remove_dangling_files
{
    my ($self, $db, $type, %opt) = @_;
    my $REMOVE_AGE = $Direct::Storage::Types::MDS_FILE_TYPES{$type}{cleanup_time} // 7 * 24 * 60 * 60;
    my $LIMIT = $opt{limit} // 50_000;
    my $to_delete = get_all_sql($db, ["
        select mm.id, mm.mds_key, count(mcn.id) as cnt from mds_metadata mm
        left join mds_custom_names mcn on mcn.mds_id = mm.id",
        where => {
            'mm.type' => $type,
            'create_time__lt__dont_quote' => "now() - interval $REMOVE_AGE second",
            ( $opt{ClientID} ? ( 'mm.ClientID' => $opt{ClientID} ) : () ),
        },
        "group by mm.id",
        having => {
            cnt => 0,
        },
        "order by mm.id",
        limit => $LIMIT,
    ]);
    my @result;
    for my $file (@$to_delete) {
        # если здесь не получится удалить из mds - ничего страшного, попробуем еще раз позже
        eval {
            $self->{mds}->delete($file->{mds_key});
            do_delete_from_table($db, 'mds_metadata', where => { id => $file->{id} });
            1;
        };
        if ($@) {
            warn "Delete error, mds_key = ".$file->{mds_key}.": ".$@;
            next;
        }
        push @result, $file;
    }
    return \@result;
}

=head2 find_all

Найти все файлы по заданному условию
Параметры именованные:
    ClientID => $client_id -- id клиента, чьи файлы мы хотим найти

Возвращает массив [ Direct::Storage::File ]

=cut

sub find_all
{
    my ($self, %opt) = @_;
    my $client_id = $opt{ClientID};

    my $query = "select id as _id, mds_key as _mds_key, size, create_time, storage_host as _storage_host, file_imprint as filename,
        ClientID, type
        from mds_metadata
        where ClientID = ?";
    my $rows = get_all_sql(PPC(ClientID => $client_id), $query, $client_id);
    push @$rows, @{get_all_sql(PPCDICT, $query, $client_id)};
    return [ map { Direct::Storage::File->new({ %$_, _mds => $self->{mds} }) } @$rows ];
}

=head2 _db($type, $ClientID)

Получить указатель на базу (в терминах Yandex::DBTools) для данного типа и client_id

=cut

sub _db
{
    my ($self, $type, $ClientID) = @_;
    if (mds_check_type_trait($type, 'empty_client_id')) {
        return PPCDICT;
    }
    return PPC(ClientID => $ClientID);
}

=head2 _get_clientid

Если в $opt не указан ClientID, но указан uid - заполняем в $opt ClientID

=cut

sub _get_clientid
{
    my ($self, $opt) = @_;
    return if exists $opt->{ClientID};
    return unless $opt->{uid};
    $opt->{ClientID} = get_clientid(uid => delete $opt->{uid});
}

=head2 _debug

=cut

sub _debug
{
    state $is_beta = is_beta();
    return unless $is_beta;
    state $log = Yandex::Log->new(log_file_name => 'storage.log');
    $log->out(@_);
}

1;
