package Direct::Storage::Types;

=head1 DESCRIPTION

Модуль с описанием типов файлов в хранилище

=cut

use Direct::Modern;
use base qw/Exporter/;

our @EXPORT = qw/
    is_valid_mds_type
    mds_check_access
    mds_get_mime_type
    mds_preprocess_filename
    mds_check_type_trait
/;

=head2 %MDS_FILE_TYPES

file_type => {
    check_access => sub {...},
    mime => sub {...},
    preprocess => sub {...},
    empty_client_id => 1|0,
    custom_name => 1|0,
    hashed_name => 1|0,
}

Ни один из ключей в описании не является обязательным, но mime и preprocess без check_access не имеют смысла

check_access - ссылка на функцию, которая получает "путь к файлу" ($client_id, $type, $filename, $orig_filename)
Должна вернуть логическое значение - можно ли отдавать этот файл из веб-интерфейса (1 -- можно, 0 -- нельзя), или нужно ответить 403
$orig_filename - имя файла до обработки функцией preprocess
Если отсутствует - подразумевается, что доступ к файлам запрещен

mime - функция, которая по "пути" (см check_access) должна вернуть правильный mime-type
Возвращает строку с mime, если строка пустая, или если ключ mime не определен - используется application/octet-stream

preprocess - функция, которая получает имя файла, и возвращает новое имя файла.
По-умолчанию - используется исходное имя. На случай, если имя, полученное от
клиента, не совпадает с именем в базе/mds

empty_client_id - можно ли сохранять файлы этого типа, не указывая ClientID
(по-умолчанию true)

custom_name - можно ли при сохранении указывать filename, не совпадающий с хешом от содержимого
hashed_name — как custom_name, только переданное имя также будет записано вместо хеша. Позволяет отдельно это имя не хранить

temporary - временные файлы, удаляются по достижении определенного возраста. При попытке перезаписи такого файла у него
обновляется время создания

cleanup_time - время жизни файлов (в секундах), на которые больше никто не ссылается. см. ppcMdsCleanup.pl и Direct::Storage::remove_dangling_files
    по умолчанию - одна неделя

=cut

our %MDS_FILE_TYPES = (
    banner_images_uploads => {
        check_access => sub { 1 }, # картинки доступны всем
        mime => sub { return 'image/png' }, # все картинки в png
        preprocess => sub {
            # разрешаем дописывать .png в конце url
            my ($filename) = @_;
            $filename =~ s!\.png$!!i;
            return $filename;
        },
        temporary => 1,
    },
    perf_feeds => {
    },
    camp_copy_report => {
        check_access => sub { 1 },
        mime => sub { return 'text/csv' }, # все отчеты в csv
        preprocess => sub {
            # разрешаем дописывать .csv в конце url
            my ($filename) = @_;
            $filename =~ s!\.csv$!!i;
            return $filename;
        },
        custom_name => 1,
    },
    common_file_export => {
        custom_name => 1,
        empty_client_id => 1,
    },
    certification_request => {
        custom_name => 1,
    },
    certification_csv => {
        custom_name => 1,
        empty_client_id => 1,
    },
    api_forecast => {
        custom_name => 1,
        check_access => sub { 1 },
        cleanup_time => 3600,
    },
    api_wordstat => {
        custom_name => 1,
        check_access => sub { 1 },
        cleanup_time => 3600,
    },
    api_forecast_nameless => {
        check_access => sub { 1 },
        cleanup_time => 3600,
        hashed_name => 1,
    },
    api_wordstat_nameless => {
        check_access => sub { 1 },
        cleanup_time => 3600,
        hashed_name => 1,
    },
    xls_history => {
        check_access => sub { 1 },
        custom_name => 1,
        temporary => 1,
    },
    offline_stat_reports => {
        custom_name => 1,
    },
    report_dynamic => {
        custom_name => 1,
        check_access => sub { 1 },
    },
    mod_licenses => {
    },
    report_performance => {
        custom_name => 1,
    },
    api5_offline_report => {
        check_access => sub { 1 },
        mime => sub { 'text/tab-separated-values' },
        hashed_name => 1,
    },
    api_report_stat => {
        check_access => sub { 1 },
        # у файла может быть расширение .xml или .gz
        # .xml отдаются как application/xml
        # .gz  отдаются как application/x-gzip
        # остальное отдаётся с типом по умолчанию, как задано в PSGIApp::Storage
        preprocess => sub {
            my ($filename) = @_;
            $filename =~ s/\.(?:xml|gz)$//i;
            return $filename;
        },
        mime => sub {
            my ($client_id, $type, $filename, $orig_filename) = @_;
            my ($extension) = ( $orig_filename =~ /\.(\w+)$/ );

            return unless $extension;

            if ( $extension eq 'gz' ) {
                return 'application/x-gzip';
            } elsif ( $extension eq 'xml' ) {
                return 'application/xml';
            }

            return;
        },
    },
    sale_reports => {
        custom_name => 1,
        empty_client_id => 1,
    },
    xls_reports => {
        custom_name => 1,
    },
    multicurrency_copy_to_new_logins_report => {
        empty_client_id => 1,
    },
    brand_safety_stats_report => {
    }
);

=head2 is_valid_mds_type($type)

Функция проверяет, существует ли такой тип файла

=cut

sub is_valid_mds_type
{
    my $type = shift;
    return exists $MDS_FILE_TYPES{$type} ? 1 : 0;
}

=head2 mds_check_access

Функция провеяет, можно ли отдать файл через веб
$client_id, $type, $filename, $orig_filename - части пути (должны быть провалидированы ранее)

Возвращает 0 или 1 (можно или нельзя), использует $MDS_FILE_TYPES{$type}{check_access}

=cut

sub mds_check_access
{
    my ($client_id, $type, $filename, $orig_filename) = @_;
    if (!$MDS_FILE_TYPES{$type}) {
        return 0;
    }
    if (!$MDS_FILE_TYPES{$type}{check_access}) {
        return 0;
    }
    return $MDS_FILE_TYPES{$type}{check_access}->($client_id, $type, $filename, $orig_filename);
}

=head2 mds_get_mime_type

Функция возвращает правильный content type для пути файла
$client_id, $type, $filename, $orig_filename - части пути (должны быть провалидированы ранее)

Использует $MDS_FILE_TYPES{$type}{mime}

=cut

sub mds_get_mime_type
{
    my ($client_id, $type, $filename, $orig_filename) = @_;
    # здесь считаем, что $type и $filename уже валидные, и не проверяем
    if (!$MDS_FILE_TYPES{$type}{mime}) {
        return undef;
    }
    return $MDS_FILE_TYPES{$type}{mime}->($client_id, $type, $filename, $orig_filename);
}

=head2 mds_preprocess_filename

Функция возвращает подготовленное для дальнейшей работы имя файла
(например, удаляет расширение)

Параметры: $type, $filename
Возвращает новый $filename

Использует $MDS_FILE_TYPES{$type}{preprocess}

=cut

sub mds_preprocess_filename
{
    my ($type, $filename) = @_;
    if ($MDS_FILE_TYPES{$type}{preprocess}) {
        return $MDS_FILE_TYPES{$type}{preprocess}->($filename);
    }
    return $filename;
}

=head2 mds_check_type_trait($type, $trait)

if (mds_check_type_trait($type, 'empty_client_id')) {
    # allowed
}

Функция проверяет, что для типа $type выставлен признак $trait

=cut

sub mds_check_type_trait
{
    my ($type, $trait) = @_;
    return (exists $MDS_FILE_TYPES{$type} && exists $MDS_FILE_TYPES{$type}->{$trait} && $MDS_FILE_TYPES{$type}->{$trait}) ? 1 : 0;
}

1;
