package Direct::Strategy;


=head1 DESCRIPTION

Базовый класс для стратегий

ДОБАВЛЕНИЕ НОВЫХ СТРАТЕГИЙ

- Создаём новый класc-потомок Direct::Strategy
- Добавляем его в список @Direct::Strategy::Tools::STRATEGY_CLASSES
- Определяем константные методы name и title
- Все параметры относящиеся к стратегии перечисляем в виде Mouse-аттрибутов
- get_strategy_hash, get_flat_strategy_hash

- Реализуем метод set_camp_values(), который переносит значения атрибутов в модель кампании, а также безусловно
  заполняет релевантные поля там же (например, autobudget=XXX).

=cut


use Direct::Modern;
use Mouse;

use Yandex::I18n;

use JSON;
use List::Util qw/first/;
use Yandex::ListUtils qw/xdiff/;
use Yandex::HashUtils qw/hash_merge/;
use Yandex::TimeCommon qw/human_datetime str_round_day today ts_round_day mysql2unix/;


=head2 $FORMAT_VERSION

Версия формата хранения стратегии.
Пока не используется, задел на будущее

=cut

our $FORMAT_VERSION = 1;


has is_autobudget => (is => 'rw', isa => 'Bool', required => 1, default => 0);


=head2 name

Символьный код стратегии.
Нужно переопределить в подклассе.

=cut

sub name {
    croak 'Not implemented in base class';
}


=head2 title

Человекочитаемое название стратегии.
Нужно переопределить в подклассе.

=cut

sub title {
    croak 'Not implemented in base class';
}


=head2 title_expanded

Расширенный список названий, в зависимости от параметров

=cut

sub title_expanded { undef }



=head2 get_params_list

Список параметров, специфичных для стратегии
Нужно переопределить в подклассе.

=cut

sub get_params_list {
    croak 'Not implemented in base class';
}


=head2 supporded_campaign_types

Список поддерживаемых типов кампаний.
Нужно переопределить в подклассе.

=cut

sub supporded_campaign_types {
    croak 'Not implemented in base class';
}


sub _is_campaign_type_supported {
    my $self = shift;
    my ($type) = @_;
    return !!grep {$type eq $_} @{ $self->supporded_campaign_types };
}


=head2 get_fields_to_check

Возвращает список полей для валидации и параметров их проверки.

    [
        $field_name => [ $field_value => %check_params ],
        ...
    ]

=cut

sub get_fields_to_check {
    return [];
#    croak "Not implemented in base class";
}


=head2 from_strategy_app_hash

Конструктор из хеша параметров применения стратегии

=cut

sub from_strategy_app_hash {
    my $class = shift;
    my ($hash) = @_;

    my $is_search_stop = !!$hash->{is_search_stop} || $hash->{search}->{name} eq 'stop';
    my $is_net_stop = !!$hash->{is_net_stop} || $hash->{net}->{name} eq 'stop';
    my $param_src =
        !$is_search_stop ? $hash->{search} :
        !$is_net_stop    ? $hash->{net} :
        {};

    return $class->from_strategy_hash($param_src);
}


=head2 from_strategy_hash

Конструктор из хеша стратегии

=cut

sub from_strategy_hash {
    my $class = shift;
    my ($hash) = @_;

    my $attr = $class->_get_attrs_from_strategy_hash($hash);

    return $class->new(%$attr);
}





sub _get_attrs_from_strategy_hash {
    my $class = shift;
    my ($hash) = @_;

    # need name check?

    # берём все параметры стратегии, не переименовывая
    my %attr = map {$_ => length($hash->{$_}) ? $hash->{$_} : undef} keys %$hash;

    return \%attr;
}

sub _get_attrs_from_campaign_hash {
    my $class = shift;
    my ($camp) = @_;

    # пробрасываем все autobudget_* параметры
    my %attr =
        map { /^autobudget_(.*)/ ? ($1 => $camp->{$_}) : ()}
        keys %$camp;

    $attr{is_autobudget} = !!$camp->{autobudget} && $camp->{autobudget} ne 'No';

    return \%attr;
}



=head2 get_strategy_hash

Возвращаем параметры стратегии в виде хеша

=cut

sub get_strategy_hash {
    my $self = shift;

    return {
        name => $self->name,
    };
}

=head2 filter_strategy_data_for_json

Отфильтровываем параметры которые не хотим сохранять в БД

=cut

sub _filter_strategy_data_for_json {
    my ($self, $save_data) = @_;

    # чистим от null-полей
    for my $key (keys %$save_data) {
        next if defined $save_data->{$key};
        delete $save_data->{$key};
    }
    return $save_data;
}

=head2 get_strategy_json

Возвращаем параметры стратегии в виде хеша, сериализованного в json,
для записи в strategy_data

=cut

sub get_strategy_json {
    my $self = shift;

    # version - заготовка на будущее
    my $save_data = hash_merge {version => $FORMAT_VERSION}, $self->get_strategy_hash;

    $self->_filter_strategy_data_for_json($save_data);

    return to_json($save_data, {canonical => 1});
}


=head2 get_flat_strategy_hash

Возвращаем параметры стратегии в виде плоского хеша

=cut

sub get_flat_strategy_hash {
    my $self = shift;

    return {
        strategy => $self->name,
        autobudget => $self->is_autobudget ? 'Yes' : 'No',
        _strategy_name => $self->name,
    };
}


=head2 set_camp_values

Заполняет в модели кампании все поля, релевантные для текущей стратегии.

=cut

sub set_camp_values {
    my $self = shift;
    my ($camp) = @_;

    $camp->_autobudget($self->is_autobudget ? 'Yes' : 'No');

    return;
}


=head2 is_differs_from

Проверка, отличаются ли объекты стратегии

=cut

sub is_differs_from {
    my $self = shift;
    my ($other) = @_;

    return 1 if !$other;
    return $self->get_strategy_json() ne $other->get_strategy_json();
}

=head2 process_conversion_strategy_last_bidder_restart_time

Проверяет произошел ли рестарт конверсионной стратегии с точки зрения биддера
Считаем, что рестарт произошел, если была изменена цель или модель аттрибуции

И обновляет поля:
 - last_bidder_restart_time - дата последнего изменения параметров стратегии, рестартящего стратегию в биддере

Возвращает:
    1 - в случае, если записываем last_bidder_restart_time
    0 - в остальных случаях

=cut
sub process_conversion_strategy_last_bidder_restart_time {
    my ($self, $old_strategy, $is_attribution_model_changed) = @_;
    return 0 if !$self;
    return 0 if !(
        $self->has_last_bidder_restart_time_option()
    );

    my $old_strategy_is_conversion_strategy = ($old_strategy->has_last_bidder_restart_time_option()) &&
        defined $old_strategy->goal_id;
    if (!$old_strategy_is_conversion_strategy ||
        (defined $self->goal_id && $self->goal_id != $old_strategy->goal_id) || $is_attribution_model_changed) {
        $self->last_bidder_restart_time(human_datetime());
        return 1;
    } else {
        my $old_last_bidder_restart_time = $old_strategy->last_bidder_restart_time;
        if (defined $old_last_bidder_restart_time) {
            $self->last_bidder_restart_time($old_last_bidder_restart_time);
        }
    }

    return 0;
}

=head2 has_last_bidder_restart_time_option

Метод проверяет может ли у данной стратегии быть опция last_bidder_restart_time.

=cut

sub has_last_bidder_restart_time_option {
    my $self = shift;
    return $self->isa('Direct::Strategy::AutobudgetAvgCpa') ||
        $self->isa('Direct::Strategy::AutobudgetAvgCpaPerCamp') ||
        $self->isa('Direct::Strategy::AutobudgetAvgCpaPerFilter') ||
        $self->isa('Direct::Strategy::AutobudgetWeekSum') ||
        $self->isa('Direct::Strategy::AutobudgetAvgCpi') ||
        $self->isa('Direct::Strategy::AutobudgetCrr');
}

=head2 process_strategy_update_time_fields

Проверяет произошло ли изменение параметров в стратегии AutobudgetCpmCustomPeriodBase/AutobudgetAvgCpvCustomPeriodBase
И обновляет поля:
 - last_update_time - дата последнего изменения параметров стратегии,
 - daily_change_count  - счетчик изменения параметрв в течение дня

Пропускаем проверку, если:
 - происходит смена стратегии, а не параметров внутри стратегии
 - стратегия отлична от AutobudgetCpmCustomPeriodBase/AutobudgetAvgCpvCustomPeriodBase

Возвращaет:
   1 - в случае, если произошло изменение параметров стратегии и обновилось поле last_update_time
   0 - в остальных случаях

=cut

sub process_strategy_update_time_fields {
    my ($self, $other) = @_;

    return 0 if !$other;
    return 0 if !$self->isa('Direct::Strategy::AutobudgetCpmCustomPeriodBase')
        && !$self->isa('Direct::Strategy::AutobudgetAvgCpvCustomPeriodBase');

    if($self->is_params_differs_from($other, ("auto_prolongation"))) {
        $self->last_update_time(human_datetime());

        #если текущая периодная стратегия уже редактировалась сегодня (без изменения начала периода), увеличиваем счетчик daily_change_count;
        #во всех остальных случаях сбрасываем счетчик
        if(($other->isa('Direct::Strategy::AutobudgetCpmCustomPeriodBase')
            || $other->isa('Direct::Strategy::AutobudgetAvgCpvCustomPeriodBase'))
            && $self->start eq $other->start
            && $self->name eq $other->name
            && $self->is_period_strategy_in_progress
            && $other->last_update_time
            && str_round_day($other->last_update_time) eq str_round_day(today())) {
            $self->daily_change_count(($other->daily_change_count || 0)+1);
        } else {
            $self->daily_change_count(1);
        }
        return 1;
    } else {
        $self->last_update_time($other->last_update_time);
        $self->daily_change_count($other->daily_change_count);
    }
    return 0;
}

=head2 is_period_strategy_in_progress

Проверка выполняется ли в данный момент периодная стратегия
Выполнятеся для стратегий AutobudgetCpmCustomPeriodBase/AutobudgetAvgCpvCustomPeriodBase; для остальных возвращает undef

Возвращает:
  1 - если текущий момент находится внутри периода выполнения стратегии
  0 - если период стратегии еще не наступил или уже завершился
  undef - если метод вызывается для непериодной стратегии

=cut

sub is_period_strategy_in_progress {
    my $self = shift;

    if ($self->isa('Direct::Strategy::AutobudgetCpmCustomPeriodBase')
        || $self->isa('Direct::Strategy::AutobudgetAvgCpvCustomPeriodBase')) {
        my $now_ts = ts_round_day(time());
        my $strategy_start_ts = ts_round_day(mysql2unix($self->start));
        my $strategy_finish_ts = ts_round_day( mysql2unix( $self->finish ) );
        return ($now_ts >= $strategy_start_ts && $now_ts <= $strategy_finish_ts);
    }
    return undef;
}

=head2 is_params_differs_from

Проверка, отличаются ли параметры одинаковых стратегий
Если стратегии разные параметры не проверяем
Нужна для для стратегий параметры которых нельзя менять по одному, а можно только весь набор

=cut

sub is_params_differs_from {
    my ($self , $other, @exclude_params ) = @_;
    return 1 if !$other;
    return 1 if $self->name ne $other->name;

    my $params = xdiff($self->get_params_list(), \@exclude_params);
    for my $param ( @$params ) {
        if ( defined $self->$param() && defined $other->$param() && ($self->$param() ne $other->$param()) ) {
            return 1;
        }
    }
    return 0;
}

sub _as_num {
    my $self = shift;
    my $val = shift;
    return undef  if !defined $val;
    return 0 + $val;
}


sub _as_str {
    my $self = shift;
    my $val = shift;
    return undef  if !defined $val;
    return "$val";
}


__PACKAGE__->meta->make_immutable;
1;

