package Direct::Strategy::Tools;

use Direct::Modern;

use JSON;
use List::Util qw/first/;

#use Primitives qw/ detect_strategy /;

use Direct::Strategy::HighestPosition;
use Direct::Strategy::NoPremium;

use Direct::Strategy::AutobudgetWeekSum;
use Direct::Strategy::AutobudgetWeekBundle;
use Direct::Strategy::AutobudgetAvgClick;
use Direct::Strategy::AutobudgetAvgCpa;
use Direct::Strategy::AutobudgetAvgCpi;
use Direct::Strategy::AutobudgetCrr;
use Direct::Strategy::AutobudgetRoi;
use Direct::Strategy::AutobudgetAvgCpcPerCamp;
use Direct::Strategy::AutobudgetAvgCpcPerFilter;
use Direct::Strategy::AutobudgetAvgCpaPerCamp;
use Direct::Strategy::AutobudgetAvgCpaPerFilter;
use Direct::Strategy::AutobudgetAvgCpv;
use Direct::Strategy::AutobudgetAvgCpvCustomPeriod;

use Direct::Strategy::AutobudgetMaxImpressions;
use Direct::Strategy::AutobudgetMaxReach;
use Direct::Strategy::AutobudgetMaxImpressionsCustomPeriod;
use Direct::Strategy::AutobudgetMaxReachCustomPeriod;
use Direct::Strategy::CpmDefault;
use Direct::Strategy::PeriodFixBid;

use Direct::Strategy::AutobudgetMedia;

use base 'Exporter';

our @EXPORT_OK = qw/
    strategy_from_campaign_hash
    strategy_from_strategy_hash
    strategy_from_strategy_app_hash
    strategy_names_human
    strategy_params_lists
/;


=head2 @STRATEGY_CLASSES

Список поддерживаемых классов стратегий

=cut

our @STRATEGY_CLASSES = qw/
    Direct::Strategy::HighestPosition
    Direct::Strategy::NoPremium

    Direct::Strategy::AutobudgetMedia

    Direct::Strategy::AutobudgetWeekSum
    Direct::Strategy::AutobudgetWeekBundle
    Direct::Strategy::AutobudgetAvgClick
    Direct::Strategy::AutobudgetAvgCpa
    Direct::Strategy::AutobudgetAvgCpi
    Direct::Strategy::AutobudgetRoi
    Direct::Strategy::AutobudgetCrr
    Direct::Strategy::AutobudgetAvgCpaPerCamp
    Direct::Strategy::AutobudgetAvgCpaPerFilter
    Direct::Strategy::AutobudgetAvgCpcPerCamp
    Direct::Strategy::AutobudgetAvgCpcPerFilter
    Direct::Strategy::AutobudgetAvgCpv
    Direct::Strategy::AutobudgetAvgCpvCustomPeriod

    Direct::Strategy::AutobudgetMaxImpressions
    Direct::Strategy::AutobudgetMaxReach
    Direct::Strategy::CpmDefault
    Direct::Strategy::AutobudgetMaxImpressionsCustomPeriod
    Direct::Strategy::AutobudgetMaxReachCustomPeriod
    Direct::Strategy::PeriodFixBid
/;

our @STRATEGY_NAMES = map {$_->name} @STRATEGY_CLASSES;

our %STRATEGY_CLASS_BY_NAME = (
    # ??? сделать отдельный класс?
    cpa_optimizer => 'Direct::Strategy::AutobudgetWeekSum',
    strategy_no_premium => 'Direct::Strategy::NoPremium',
    maximum_coverage => 'Direct::Strategy::HighestPosition',
    map { $_->name => $_ } @STRATEGY_CLASSES,
);


=head2 strategy_from_campaign_hash

    my $strategy_object = strategy_from_campaign_hash($camp);

Создаёт инстанс объекта стратегии из полей кампании.

=cut

sub strategy_from_campaign_hash {
    my ($camp, %opt) = @_;

    my $strategy_hash = $camp->{strategy_decoded} || from_json($camp->{strategy_data});
    return strategy_from_strategy_hash($strategy_hash);
}


=head2 strategy_from_strategy_hash

    my $strategy_object = strategy_from_strategy_hash($strategy_hash);

Создаёт инстанс объекта стратегии из хеша стратегии

=cut

sub strategy_from_strategy_hash {
    my ($hash) = @_;

    my $name = $hash->{name};
    croak 'Strategy not identified'  if !$name;

    my $class = $STRATEGY_CLASS_BY_NAME{$name};
    croak "Unsupported strategy '$name'"  if !$class;

    my $strategy = $class->from_strategy_hash($hash);
    return $strategy;
}


=head2 strategy_from_strategy_app_hash

    my $strategy_object = strategy_from_strategy_app_hash($old_style_strategy);

Создаёт инстанс объекта стратегии из хеша применения стратегии

Опции:

  * skip_unknown

=cut

sub strategy_from_strategy_app_hash {
    my ($hash, %opt) = @_;

    my $name = first {$_ && $_ ne 'stop'} map {$hash->{$_}->{name}} qw/ search net /;
    $name = $hash->{name}  if !$name;
    return undef  if !$name && $opt{skip_unknown};
    croak 'Strategy not identified'  if !$name;

    # костыль для баяна
    $name = 'autobudget_media'  if $name eq 'autobudget' && defined $hash->{search}->{date};

    my $class = $STRATEGY_CLASS_BY_NAME{$name};
    return undef  if !$class && $opt{skip_unknown};
    croak "Unsupported strategy '$name'"  if !$class;

    my $strategy = $class->from_strategy_app_hash($hash);
    return $strategy;
}


=head2 strategy_names_human

Возвращает отображение название стратегии в человекочитаемый вид.
Используется для добавления данных в %Campaign::STRATEGY_NAME.

=cut

sub strategy_names_human {
    my %result = map {($_->name => $_->title_expanded || $_->title)} @STRATEGY_CLASSES;
    return %result;
}


=head2 strategy_params_lists

Возвращает список доступных параметров для каждой стратегии.
Используется для добавления данных в Campaign::strategy_struct_sample.

=cut

sub strategy_params_lists {
    my %result = map {($_->name => $_->get_params_list)} @STRATEGY_CLASSES;
    return %result;
}

1;

