package Direct::TargetingCategories;

use Carp;
use Direct::Modern;
use Mouse;
use Settings;

use Yandex::DBTools;
use Yandex::HashUtils;
use Cache::SizeAwareMemoryCache;

use Direct::Model::TargetCategory;

has 'items' => (is => 'ro', isa => 'ArrayRef[Direct::Model::TargetCategory]');

around BUILDARGS => sub {
        my ($orig, $class) = (shift, shift);
        $class->$orig( @_ == 1 ? (items => $_[0]) : @_ )
    };

my $CACHE = Cache::SizeAwareMemoryCache->new( { namespace => 'TargetingCategories', default_expires_in => 60 * 60 } );

=head2 clear_cache

    очистить кеш

=cut
sub clear_cache {
    $CACHE->clear();
}


=head2 get_by

=cut

sub get_by {
    my ($class, $key, $vals) = @_;

    croak "only `targeting_type` keys are supported" unless $key =~ /^targeting_type$/;

    $vals = [ $vals // () ] if ref($vals) ne 'ARRAY';
    return $class->new( [ ] ) if !@$vals;

    my $rows = _get_targeting_categories_from_db($vals);
    return $class->new( [ ] ) if !@$rows;

    return $class->new( items => Direct::Model::TargetCategory->from_db_hash_multi( $rows ) );
}

=head2 get_rmp_interests

=cut

sub get_rmp_interests {
    my ($class) = @_;
    return $class->get_by( targeting_type => 'rmp_interest' );
}

=head2 items_by


=cut

sub items_by {
    my ($self, $key) = @_;

    $key //= 'id';

    my %result;
    if ($key =~ '(id|import_id)') {
        $result{$_->$1} = $_ for @{$self->items};
    } else {
        croak "Unknown items_by key: `$key`";
    }

    return \%result;
}

sub _get_targeting_categories_from_db {
    my ($targeting_type) = @_;

    my $result = $CACHE->get( 'targeting_categories' );
    return $result if $result;

    my $targeting_categories = get_all_sql(
        PPCDICT, [
            qq/SELECT ${ \Direct::Model::TargetCategory->get_db_columns('targeting_categories') } FROM targeting_categories/,
            where => [ targeting_type => $targeting_type, state => 'Submitted' ],
            q/ORDER BY order_num, category_id/
        ]
    );

    # Доступны для выбора в качестве условий показа только листия дерева
    my %targeting_category_with_childs = map { $_->{parent_category_id} => undef } grep { $_->{parent_category_id} } @$targeting_categories;
    $_->{available} = exists $targeting_category_with_childs{$_->{category_id}} ? 0 : 1 foreach @$targeting_categories;

    $CACHE->set( targeting_categories => $targeting_categories );

    return $targeting_categories;
}

=head2 build_category_tree($targeting_type)

Строит дерево категорий таргетинга по интересам определенного типа ($targeting_type)

Параметры:
    $targeting_type -> какого типа дерево необходимо построить, сейчас поддерживается только rmp_interest


=cut

sub build_category_tree {
    my ($class, $targeting_type) = @_;

    croak q/is class-only method/ if blessed($class);
    croak qq/Unsupport targeting_type: $targeting_type/ unless $targeting_type eq 'rmp_interest';

    my $rows = _get_targeting_categories_from_db($targeting_type);

    return [ ] unless @$rows;

    my %category_id2row;
    my $parent_category_id2childs_categories_ids;
    foreach my $row (@$rows) {
        $category_id2row{$row->{category_id}} = $row;
        push @{ $parent_category_id2childs_categories_ids->{ $row->{parent_category_id} // 'ROOT' } //= [ ] }, $row->{category_id};
    }

    my ($cache, @stack, @result);
    my @categories_ids = @{ $parent_category_id2childs_categories_ids->{ROOT} };
    while (@categories_ids) {
        my $category_id = pop @categories_ids;
        my $target_category = hash_merge(Direct::Model::TargetCategory->from_db_hash( $category_id2row{$category_id}, \$cache )->to_template_hash, { childs => [ ] });

        unless (exists $parent_category_id2childs_categories_ids->{$category_id}) { # если у элемента нет детей
            unless (@stack) { # если это топ, то кладем в корень
                push @result, $target_category;
            } else {
                if ($category_id2row{$category_id}->{parent_category_id} && $category_id2row{$category_id}->{parent_category_id} == $stack[$#stack]->{target_category_id}) {
                    # если это дочерний элемент, то кладем в childs родителя
                    push @{ $stack[$#stack]->{childs} }, Direct::Model::TargetCategory->from_db_hash( $category_id2row{$category_id}, \$cache )->to_template_hash;
                } else {
                    pop @stack; # иначе выкидываем сверху стека последний элемент
                    push @categories_ids, $category_id; # а сам элемент отправляем на повторную проверку
                }
            }
        } else {
            unless (@stack) {
                push @result, $target_category; # если это топ, то кладем в корень
            } else {
                push @{ $stack[$#stack]->{childs} }, $target_category; # иначе кладем в childs родителя
            }
            push @categories_ids, @{ $parent_category_id2childs_categories_ids->{$category_id} };  # положим все дочерние категории сверху
            push @stack, $target_category; # положим родителя на стек дерева под-категорий
        }
    }

    return \@result;
}

1;
