package Direct::TurboLandings;

use Mouse;
use Direct::Modern;

use JSON qw/from_json/;
use List::MoreUtils qw/any uniq/;
use Yandex::ListUtils qw/xminus chunks/;
use Try::Tiny qw/try catch/;

use EnvTools;
use Settings;
use Yandex::DBShards;
use Yandex::DBTools;
use Yandex::Trace;

use BannerStorage;
use Client;
use MetrikaCounters;
use Direct::Model::TurboLanding;
use JavaIntapi::UpdateCounterGrants qw//;
use DirectContext;


our $TURBOLANDING_PREFIX = 'yandex\.\w+(?:\.tr)?/turbo';
our $MAX_CLIENTS_PER_GRANTS_UPDATE = 500;
our $REST_TIME_FOR_GRANT_UPDATES = 10; #секунд

has 'items' => (is => 'ro', isa => 'ArrayRef[Direct::Model::TurboLanding]');
around BUILDARGS => sub { my ($orig, $class) = (shift, shift); $class->$orig(@_ == 1 ? (items => $_[0]) : @_) };


=head2 get_by($key, $vals, %options)

    По заданному критерию возвращает instance с выбранными турболендингами.
    
    Параметры:
        $key -> по какому ключу выбирать баннеры, в настоящий момент поддерживается только client_id
        $vals -> (Int|ArrayRef[Int]); список идентификаторов
        %options -> id => id|[id1, id2, ...] - вернуть только турболендинги с заданными id

=cut

sub get_by {
    my ($class, $key, $vals, %options) = @_;
    
    croak 'only `client_id` key is supported' unless $key eq 'client_id';
    
    $vals = [$vals // ()] if ref($vals) ne 'ARRAY';
    return $class->new([]) if !@$vals;
    
    my %additional_filters;
    
    if (exists $options{id}) {
        $additional_filters{tl_id} = $options{id}
    }
    
    my $client_ids = $vals;
    my @columns = Direct::Model::TurboLanding->get_db_columns_list('turbolandings');
    my $landings = get_all_sql(PPC(ClientID => $client_ids), [
        'SELECT '.join(', ', @columns), FROM => 'turbolandings',
        WHERE => {ClientID => SHARD_IDS, %additional_filters}
    ]);
    my $cache;
    
    return $class->new([map {Direct::Model::TurboLanding->from_db_hash($_, \$cache)} @$landings]);
}

=head2 get_url_prefix()
    
    Возвращает префикс, специфичный для турбостраниц.
    Используется при проверке, указывает ли заданный href на турбостраницу
    
=cut

sub get_url_prefix {
    return $TURBOLANDING_PREFIX;
}

=head2 get_turbolandings_by_id_multi_without_sharding($dbh, $set_ids)
    
    Для указанного инстанса БД разворачивает список переданных id турболендингов в пары { id => ... , href => ... }.
    Возвращает hashref:
    {
        id1 =>   { id => id1 , href => href1 },
        id2 =>   { id => id2 , href => href2 },
        ...
    }
    
=cut

sub get_turbolandings_by_id_multi_without_sharding {
    my ($dbh, $set_ids) = @_;
    return {} if !@$set_ids;

    my $landings = get_hash_sql($dbh, ['SELECT tl_id, href FROM turbolandings', WHERE => { tl_id => $set_ids }]);
    return $landings;
}

=head2  sync_by_client_id ($client_id)
    void
    
    Делает запрос в BS, проучает оттуда 150 последних турболендингов клиента.
    Если среди них есть отсутствующие в БД - добавляет их в таблицу turbolandings.
    Также, обновляет имена уже имеющихся в БД турболендингов, если их имена изменены в конструкторе.
    
=cut

sub sync_by_client_id {
    my ($class, $client_id, $tl_ids) = @_;
    
    my %filter = defined $tl_ids ? (id => $tl_ids) : ();
    my $known_turbolandings = $class->get_by(client_id => $client_id, %filter)->items;
    
    my $need_bs_storage_call = 1;
    my $new_tl_ids;
    
    if (defined $tl_ids) {
        $new_tl_ids = xminus($tl_ids, [map {$_->id} @$known_turbolandings]);
        
        return $known_turbolandings unless @$new_tl_ids;
    }
    
    
    my $bs_turbolandings_hash;
    
    if (!defined $tl_ids || $new_tl_ids) {
        try {
            $bs_turbolandings_hash = BannerStorage::get_bs_turbolandings_by_client_id($client_id, $new_tl_ids);
        }
        catch {
            warn 'BannerStorage fault: ', @_;
        };
    }

    foreach my $tl (@$known_turbolandings){
        next if not exists $bs_turbolandings_hash->{$tl->id};
        next if $bs_turbolandings_hash->{$tl->id}->{name} ne $tl->name;
        delete $bs_turbolandings_hash->{$tl->id};
    }   
    return unless keys %$bs_turbolandings_hash;
    
    _save_turbolandings($client_id, [values %$bs_turbolandings_hash]);
    
    if ($new_tl_ids) {
        return $class->get_by(client_id => $client_id, id => $new_tl_ids)->items;
    }
    
    return;
}

sub _save_turbolandings {
    my ($client_id, $turbolandings) = @_;

    my $insert_query = 'INSERT INTO turbolandings (tl_id, ClientID, name, href, metrika_counters_json) VALUES %s
    ON DUPLICATE KEY UPDATE
    name = VALUES(name),
    metrika_counters_json = VALUES(metrika_counters_json)
    ';
    if ( is_beta() ){
        $insert_query .=',
            href = VALUES(href),
            ClientID = VALUES(ClientID)
        ';
    }
    do_mass_insert_sql(PPC(ClientID => $client_id),
            $insert_query,
            [map {[ @$_{qw/tl_id client_id name href metrika_counters_json/} ]} @$turbolandings]
    );
}

sub _update_goals {
    my ($client_id, $cg_for_update) = @_;
    #Т.к. число изменяемых целей турболендинга исчисляется единицами, а в таблицу может быть как insert, так и update,
    #сделаем @goals простых запросов вместо одного сложного
    my $ppc = PPC(ClientID => $client_id);
    foreach my $cid (keys $cg_for_update){
        while (my ($goal, $increment) = each %{$cg_for_update->{$cid}}){
            do_sql($ppc, [q/INSERT INTO camp_metrika_goals (cid, goal_id, links_count, goal_role) VALUES/,
                '(', join(', ', sql_quote($cid) => sql_quote($goal) => 'GREATEST(0, '.sql_quote($increment).')' => sql_quote('combined')), ')',
                'ON DUPLICATE KEY UPDATE ', join( ', ',
                    'links_count = GREATEST(0,COALESCE(links_count,0) + '.sql_quote($increment).')' => 'goal_role=CONCAT(goal_role, ",combined")' )
            ]);
        }
    }

    return;
}

=head2  get_metrika_counters_and_goals_by_tl_id($db, $tl_ids)
    $db - хендлер БД в нотации DBTools (PPC( cid => $cid), и т.п.)
    $tl_ids - ссылка на массив идентификаторов извлекаемых турболендингов 

    Возвращает ссылку на хеш:
    { id_турболендинга => {
        counters => [счетчики_турболендинга],
        goals    => [цели_турболендинга] }
    }
=cut

sub get_metrika_counters_and_goals_by_tl_id {
    my ($db, $tl_ids) = @_;
    
    my $db_data = get_all_sql($db,
                [q/SELECT tl_id, metrika_counters_json FROM turbolandings/,
                        WHERE => {tl_id => $tl_ids}
                ]
        );
    my %result;
    
    foreach my $row (@$db_data) {
        $result{$row->{tl_id}} //= {counters =>  [], goals => []};
        my $data = $result{$row->{tl_id}};
        $data = _extract_counters_and_goals($data,$row->{metrika_counters_json});
    }
    
    return \%result
}

sub _extract_counters_and_goals {
    my ($data, $counters_json) = @_;

    foreach my $counter ( @{from_json($counters_json)}){
        push @{$data->{counters}}, $counter->{id};
        push @{$data->{goals}}, @{$counter->{goals}} if $counter->{goals};
        $data->{isUserCounter}->{$counter->{id}} = 1 if $counter->{isUserCounter};
    }
    $data->{goals} = [uniq @{$data->{goals}}];

    return $data;
}

=head2 refresh_metrika_grants($metrika_counters_by_client_id)
    Проставляет для всех указаных счетчиков метрики права доступа 'view' всем пользователям,
    имеющим доступ к ресурсам соответствующего клиента (представители клиента, менеджеры, представители агенства)
    
    Возвращает HashRef с признаками успешности обновления списков доступа для каждого из счетчиков:
    {counter_id1 => 0|1, counter_id2 => 0|1 ...}

=cut

my $USE_JAVA_UPDATE_COUNTER_GRANTS_PROPERTY_NAME = 'use_java_update_counter_grants';

sub refresh_metrika_grants {
    my ($counters_by_client_id, $operator_uid) = @_;
    
    my $related_users_by_client_id = Client::get_related_users_by_client_id([keys %$counters_by_client_id]);

    my $grants;
    my $uids_by_counter = {};
    foreach my $client_id (keys %$related_users_by_client_id) {
        my $counters = $counters_by_client_id->{$client_id};
        next unless $counters;
        
        foreach my $counter (@$counters){
            my @related_users = @{ $related_users_by_client_id->{$client_id} };
            push @{ $grants->{$counter} }, (map { {user_login => $_->{login}, perm => 'view'} } @related_users);
            push @{ $uids_by_counter->{$counter} }, (map { $_->{uid} } @related_users);
        }
    }

    if (!$operator_uid && $DirectContext::current_context) {
        $operator_uid = $DirectContext::current_context->UID;
    }

    if ($operator_uid && Property->new($USE_JAVA_UPDATE_COUNTER_GRANTS_PROPERTY_NAME)->get()) {
        my $result = scalar(keys %$uids_by_counter) 
            ? JavaIntapi::UpdateCounterGrants->new(uids_by_counter_id => $uids_by_counter, operator_uid => $operator_uid)->call() 
            : {};
        return $result;
    } else {
        return MetrikaCounters::update_counter_grants($grants);
    }
}

=head2 mass_refresh_metrika_grants_for_all_client_counters($client_ids)
    
    Обновляет списки доступа для всех счетчиков метрики турболендингов каждого клиента

    Возвращает 1 если доступ для всех счетчиков обновлен успешно и 0 - если нет.
    
=cut

sub mass_refresh_metrika_grants_for_all_client_counters {
    my ($client_ids, $operator_uid) = @_;
    
    my $profile = Yandex::Trace::new_profile('turbolandings:refresh_metrika_grants_for_clients', obj_num => scalar @$client_ids);
    my $counters_by_client_id = _get_metrika_counters_by_client_id($client_ids, except_user_counters => 1);
    my $success = refresh_metrika_grants($counters_by_client_id, $operator_uid);
    undef $profile;
    
    return (any {!$_} values %$success) ? 1 : 0;
}

sub _get_metrika_counters_by_client_id {
    my ($client_ids, %opts) = @_;
    
    my $with_user_counters = $opts{except_user_counters} ? 0 : 1;

    my $counters_json = get_all_sql(PPC(ClientID => $client_ids),
             [q/SELECT DISTINCT ClientID, metrika_counters_json FROM turbolandings/,
                     WHERE => {ClientID => SHARD_IDS}
             ]
    );
    
    my %counters_hash;
    foreach my $row (@$counters_json){
        my $parsed = eval{from_json($row->{metrika_counters_json})};
        next unless ref $parsed eq 'ARRAY';
        foreach my $counter (@$parsed){
            next unless ref $counter eq 'HASH' && $counter->{id};
            next if $counter->{isUserCounter} && !$with_user_counters;

            $counters_hash{$row->{ClientID}}->{$counter->{id}} //=1;
        }
    }
    
    return {map {$_ => [keys %{$counters_hash{$_}}] } keys %counters_hash};
}

=head2 refresh_metrika_grants_for_agency_clients($agency_id)
    void
    Обновить права доступа к счетчикам всем клиентам агенства, имеющим турболендинги
=cut

sub refresh_metrika_grants_for_agency_clients {
    my ($agency_client_id) = @_;
    
    my $agency_clients = get_one_column_sql(PPC(shard => 'all'),
             [q/SELECT DISTINCT t.ClientID FROM turbolandings t JOIN clients c ON (c.ClientID = t.ClientID)/,
                     WHERE => {'c.agency_client_id' => $agency_client_id}
             ]
    );
    
    my $profile = Yandex::Trace::new_profile('turbolandings:refresh_metrika_grants_for_agency', obj_num => scalar @$agency_clients);
    _mass_grants_update($agency_clients);
    undef $profile;
    
    return;    
}

sub _mass_grants_update {
    my ($client_ids) = @_;
    
    foreach my $clients_chunk (chunks($client_ids, $MAX_CLIENTS_PER_GRANTS_UPDATE)){
        mass_refresh_metrika_grants_for_all_client_counters($clients_chunk);
        sleep $REST_TIME_FOR_GRANT_UPDATES if @$client_ids > $MAX_CLIENTS_PER_GRANTS_UPDATE;
    }

    return;
}

1;
