package Direct::Validation::AdGroups;
## no critic (TestingAndDebugging::RequireUseStrict, TestingAndDebugging::RequireUseWarnings)

use Direct::Modern;

use base qw(Exporter);
our @EXPORT = qw/
    validate_add_adgroups
    validate_update_adgroups
    validate_delete_adgroups

    validate_add_banners_adgroup
    validate_banners_adgroup
/;

use Direct::Validation::Errors;
use Direct::Validation::MinusWords;
use Direct::Validation::Banners qw/validate_banner_geo_targeting/;
use Direct::Validation::HierarchicalMultipliers qw/
    validate_mobile_multiplier
    validate_desktop_multiplier
    validate_video_multiplier
    validate_demography_multiplier
    validate_retargeting_multiplier
    validate_performance_tgo_multiplier
/;
use TextTools qw/smartstrip2/;
use GeoTools qw/validate_geo/;
use Yandex::I18n;
use Yandex::HashUtils;
use List::MoreUtils qw/any/;
use List::UtilsBy qw/count_by/;
use MinusWords;

=head2 validate_add_adgroups($groups, $campaign, %options)

    Валидация добавления группы объявлений в кампанию

    Параметры:
        $adgroups - [Direct::Model::AdGroup],
        $campaign - [Direct::Model::Campaign]
        %options -
            translocal_tree => транслокальное дерево, используемое для проверки соответствия текста баннеров региону

    Результат:
        Direct::ValidationResult

=cut

sub validate_add_adgroups {
    my ($adgroups, $campaign, %options) = @_;

    my $vr = validate_adgroups($adgroups, translocal_tree => $options{translocal_tree});
    if ($campaign->adgroups_count + @$adgroups > $campaign->adgroups_limit) {
        $vr->add_generic(error_ReachLimit(
            iget("Достигнуто максимальное количество групп объявлений в кампании - %d", $campaign->adgroups_limit)
        ));
    }

    return $vr;
}


=head2 validate_update_adgroups($groups)

    Проверка обновления группы объявлений

    Параметры:
        $adgroups - [Direct::Model::AdGroup],
        %options -
            translocal_tree - транслокальное дерево, используемое для проверки соответствия текста баннеров региону
    Результат:
        хранится по полям, возможные поля с ошибками: name, geo, minus_words

=cut

sub validate_update_adgroups {
    my ($groups, %options) = @_;

    my $validation_result = validate_adgroups($groups, with_banners => 1, translocal_tree => $options{translocal_tree});

    return $validation_result;
}

=head2 validate_adgroups($groups, %options)

    Проверка группы объявлений

    Параметры:
        $groups - ссылка на массив групп
            каждая группа это тип Direct::Model::AdGroup
            должна содержать campaign с заполненным content_lang
            может содержать баннеры, поле banners - тип Direct::Model::Banner
        %options
            with_banners => 1 (значение по умолчанию 0) - проверка группы объявлений вместе с вложенными баннерами
                                т.е. при with_banners => 1 будут выполнена проверка соответствия гео языку баннеров
                                и проверена обязательность имени группы объявлений
            translocal_tree - транслокальное дерево, используемое для проверки соответствия текста баннеров региону,

    Результат:
        хранится по полям, возможные поля с ошибками: name, geo, minus_words, hierarchical_multipliers

=cut

sub validate_adgroups {
    my ($groups, %options) = @_;
    
    my $validation_result = new Direct::ValidationResult();
    foreach my $group (@$groups) {

        my $group_vr = $validation_result->next;

        my $name = $group->has_adgroup_name ? smartstrip2($group->adgroup_name) : undef;
        if (!defined $name) {
            $group_vr->add(
                adgroup_name => error_ReqField(iget('Необходимо задать название группы'))
            );
        } elsif (defined $name && length $name == 0) {
            $group_vr->add(
                adgroup_name => error_InvalidFormat(iget('Имя группы не может быть пустой строкой ""'))
            );
        }

        my $geo = smartstrip2($group->geo);
        if (defined $geo && $geo =~ /\S/) {
            my $geo_error_text = validate_geo( $group->geo, undef,
               $options{translocal_tree}
                    ? {tree => $options{translocal_tree} } : {ClientID => $group->client_id} );

            if ($geo_error_text) {
                $group_vr->add(geo => error_BadGeo($geo_error_text));
            } else {
                my $campaign = $group->campaign;
                foreach my $banner ($options{with_banners} ? @{$group->banners} : ()) {
                    # tree => undef - вычислится дерево для ClientID
                    my $errors = validate_banner_geo_targeting(
                        $banner, $group->geo, $campaign->content_lang, for_adgroup => 1,
                        $options{translocal_tree} ? (translocal_opt => {tree => $options{translocal_tree}}) : ()
                    );
                    if ($errors) {
                        $group_vr->add(geo => $errors);
                        last;
                    }
                }
            }
        } else {
            $group_vr->add(geo => error_ReqField(iget('Необходимо задать геотаргетинг группы объявлений')));
        }

        # has_ значит загружалось, при этом undef если реально связки нет
        if ($group->has_minus_words && @{$group->minus_words}) {
                $group_vr->add(minus_words => validate_group_minus_words($group->minus_words, MinusWords::polish_minus_words_array($group->minus_words)));
        }

        if ($group->has_hierarchical_multipliers) {
            my $hierarchical_multipliers = $group->hierarchical_multipliers;
            my $hm_vr = Direct::ValidationResult->new();

            # Вместо campaign->campaign_type будем передавать adgroup_type, т.к. они практически совпадают

            if ($hierarchical_multipliers->{mobile_multiplier}) {
                my $desktop_multiplier_pct =
                        $hierarchical_multipliers->{desktop_multiplier} ? $hierarchical_multipliers->{desktop_multiplier}->{multiplier_pct} : undef;
                $hm_vr->add(mobile_multiplier => validate_mobile_multiplier(
                    $group->adgroup_type, $group->client_id,
                        hash_merge(
                            $hierarchical_multipliers->{mobile_multiplier},
                            {desktop_multiplier_pct => $desktop_multiplier_pct}
                        )
                ));
            }
            if ($hierarchical_multipliers->{desktop_multiplier}) {
                my $mobile_multiplier_pct =
                    ($hierarchical_multipliers->{mobile_multiplier} && !$hierarchical_multipliers->{mobile_multiplier}->{os_type})
                    ? $hierarchical_multipliers->{mobile_multiplier}->{multiplier_pct}
                    : undef;
                $hm_vr->add(desktop_multiplier => validate_desktop_multiplier(
                        $group->adgroup_type, $group->client_id,
                        hash_merge(
                            $hierarchical_multipliers->{desktop_multiplier},
                            {mobile_multiplier_pct => $mobile_multiplier_pct}
                        )
                    ));
            }
            if ($hierarchical_multipliers->{video_multiplier}) {
                $hm_vr->add(video_multiplier => validate_video_multiplier(
                    $group->adgroup_type, $hierarchical_multipliers->{video_multiplier}
                ));
            }
            if ($hierarchical_multipliers->{demography_multiplier}) {
                $hm_vr->add(demography_multiplier => validate_demography_multiplier(
                    $group->adgroup_type, $hierarchical_multipliers->{demography_multiplier}
                ));
            }
            if ($hierarchical_multipliers->{retargeting_multiplier}) {
                $hm_vr->add(retargeting_multiplier => validate_retargeting_multiplier(
                    $group->adgroup_type, $group->client_id, $hierarchical_multipliers->{retargeting_multiplier}
                ));
            }
            if ($hierarchical_multipliers->{performance_tgo_multiplier}) {
                $hm_vr->add(performance_tgo_multiplier => validate_performance_tgo_multiplier(
                    $group->adgroup_type, $hierarchical_multipliers->{performance_tgo_multiplier}
                ));
            }

            $group_vr->add(hierarchical_multipliers => $hm_vr) unless $hm_vr->is_valid;
        }
    }

    return $validation_result;
}

sub _get_banner_creative_type
{
    my ($banner) = @_;
    if ($banner->has_creative) {
        return $banner->creative->creative->creative_type;
    }
    return '';
}

=head2 validate_delete_adgroups($adgroups)

    Валидирует удаление групп объявлений.

    Параметры:
        $adgroups -> массив объектов Direct::Model::AdGroup,
        если в объекте есть banners - валидируем удаление баннеров, если нет - валидируем удаление пустой группы
        $campaign -> кампания (Direct::Model::Campaign), с полями sum, is_in_bs_queue (роль BsQueue)

    Результат:
        объект Direct::ValidationResult с общими ошибками по объектам групп

=cut

sub validate_delete_adgroups {
    my ($adgroups, $campaign) = @_;

    my $vr = new Direct::ValidationResult();
    foreach my $adgroup (@$adgroups) {
        my $adgroup_vr = $vr->next;

        if ($adgroup->has_banners) {
            if (any {Direct::Validation::Banners::validate_delete_banner($_, $adgroup, $campaign)} @{$adgroup->banners}) {
                $adgroup_vr->add_generic(error_BadStatus(iget("Группу объявлений невозможно удалить")));
            }
        } elsif ($adgroup->banners_count || $adgroup->has_show_conditions) {
            $adgroup_vr->add_generic(
                error_CantDelete(iget("Нельзя удалить группу, содержащую объявления или условия показа"))
            );
        }
    }

    return $vr;
}

=head2 validate_add_banners_adgroup($rules, $banners, $adgroup, %options)

    Валидирует добавление баннеров в групп

    Параметры:
        $rules - функции валидации кокретного типа баннера, в формате class_name => \&validate_function,
                 прототип должен быть validate_function([Direct::Model::Banner], Direct::Model::AdGroup, %options) => Direct::ValidationResult
        $banners - массив [Direct::Model::Banner] проверяемых баннер
        $adgroup - группа (Direct::Model::AdGroup) в которую добавляются баннеры
        %options - передаются как есть в validate_function, за исключением
                   banners_count_to_delete - сколько баннеров будет удалено из группы
                   permitted_pixel_providers - хэш доступных клиенту пикселов по типам кампаний

    Результат:
        Direct::ValidationResult
              
=cut

sub validate_add_banners_adgroup {
    my ($rules, $banners, $adgroup, %options) = @_;

    my $vr_main = validate_banners_adgroup($rules, $banners, $adgroup, %options);
    # Проверим, не будет ли превышено число допустимых объявлений в группе
    my $banners_count_to_delete = $options{banners_count_to_delete} // 0;
    if ($adgroup->banners_count - $banners_count_to_delete + @$banners > $Settings::DEFAULT_CREATIVE_COUNT_LIMIT) {
        $vr_main->add_generic(
            error_LimitExceeded(iget("Количество объявлений в группе не может быть больше %s", $Settings::DEFAULT_CREATIVE_COUNT_LIMIT))
        );
    }
    
    if ($adgroup->has_adgroup_type) {
        if ($adgroup->adgroup_type =~ /cpm_(banner|video)/) {
            my %creative_types_count = count_by { _get_banner_creative_type($_) } @$banners;
            if ($adgroup->adgroup_type eq 'cpm_video') {
                if (keys %creative_types_count > 1 || !$creative_types_count{video_addition}) {
                    # в группе cpm_video могут быть только видео-баннеры
                    $vr_main->add_generic(error_BadStatus('Несовместимые типы баннера и группы'));
                }
            } elsif ($creative_types_count{video_addition}) {
                # в группе cpm_banner не может быть видео-баннеров
                $vr_main->add_generic(error_BadStatus('Группа не может содержать видео-баннеры'));
            }
        }
    }

    return $vr_main;
}

=head2 validate_banners_adgroup($rules, $banners, $adgroup, %options)

    см. validate_add_banners_adgroup

=cut

sub validate_banners_adgroup {
    my ($rules, $banners, $adgroup, %options) = @_;

    my %validators;
    for my $class (keys %$rules) {
        $validators{$class} = {
            banners => [],
            code => $rules->{$class}
        };
    }

    my @dst_idx;
    for my $banner (@$banners) {
        my $class = ref $banner;
        if (exists $validators{$class}) {
            my $validator = $validators{$class};
            push @dst_idx, scalar @{$validator->{banners}};
            push @{$validator->{banners}}, $banner;
        } else {
            push @dst_idx, undef;
        }
    }

    for my $validator (values %validators) {
        next unless @{$validator->{banners}};
        $validator->{vr} = $validator->{code}->($validator->{banners}, $adgroup,
            translocal_opt => $options{translocal_opt}, permitted_pixel_providers => $options{permitted_pixel_providers});
    }

    my $vr_main = Direct::ValidationResult->new();
    my $src_idx = 0;
    for my $banner (@$banners) {
        my $class = ref $banner;
        if (exists $validators{$class}) {
            my $banner_vr = $validators{$class}->{vr}->get_nested_vr_by_index($dst_idx[$src_idx]);
            $vr_main->next($banner_vr);
        } else {
            die 'Unsupported Banner';
        }
        $src_idx++;
    }

    for my $validator (values %validators) {
        next unless $validator->{vr};
        $vr_main->add_generic(
            $validator->{vr}->get_generic_errors()
        );
        $vr_main->add_generic(
            $validator->{vr}->get_generic_warnings()
        );
    }
    return $vr_main;
}

1;
