package Direct::Validation::AdGroupsDynamic;

use Direct::Modern;

use base qw(Exporter);

use Settings;

use Yandex::I18n;
use Yandex::DBTools;
use Yandex::HashUtils;
use GeoTools qw/validate_geo/;
use Yandex::IDN qw/is_valid_domain/;
use MinusWords;

use Direct::Validation::Errors;
use Direct::ValidationResult;

use Direct::Validation::AdGroups qw/validate_delete_adgroups validate_banners_adgroup validate_add_banners_adgroup/;
use Direct::Validation::MinusWords qw/validate_group_minus_words/;
use Direct::Validation::BannersDynamic qw/validate_add_dynamic_banners validate_update_dynamic_banners/;
use Direct::Validation::HierarchicalMultipliers qw/
    validate_mobile_multiplier
    validate_desktop_multiplier
    validate_video_multiplier
    validate_demography_multiplier
    validate_retargeting_multiplier
    validate_performance_tgo_multiplier
/;
use Direct::Validation::Banners qw/
    validate_banner_geo_targeting
/;

our @EXPORT = qw/
    validate_add_dynamic_adgroups
    validate_update_dynamic_adgroups
    validate_add_banners_dynamic_adgroup
    validate_update_banners_dynamic_adgroup
/;

=head2 validate_add_dynamic_adgroups($adgroups, $campaign, %options)

Валидация добавления динамических групп в кампанию.

Параметры:
    $adgroups - ссылка на массив добавляемых динамических групп ([Direct::Model::AdGroupDynamic, ...])
    $campaign - кампания, куда добавляются группы (Direct::Model::Campaign),
                с полями adgroups_count, adgroups_limit (роль AdGroupsCount)
    %options:
        translocal_opt => параметры транслокальности

=cut

sub validate_add_dynamic_adgroups {
    my ($adgroups, $campaign, %options) = @_;

    my $vr_main = validate_dynamic_adgroups($adgroups, translocal_opt => $options{translocal_opt});

    if ($campaign->adgroups_count + @$adgroups > $campaign->adgroups_limit) {
        $vr_main->add_generic(error_ReachLimit(
            iget("Достигнуто максимальное количество групп объявлений в кампании - %d", $campaign->adgroups_limit)
        ));
    }

    return $vr_main;
}

=head2 validate_update_dynamic_adgroups($adgroups, %options)

Валидация обновления динамических групп.

Параметры:
    $adgroups - ссылка на массив динамических групп ([Direct::Model::AdGroupDynamic]),
                с баннерами (если поменялся гео-таргетинг)
    %options:
        translocal_opt => параметры транслокальности

=cut

sub validate_update_dynamic_adgroups {
    my ($adgroups, %options) = @_;
    return validate_dynamic_adgroups($adgroups, translocal_opt => $options{translocal_opt}, with_banners => 1);
}

=head2 validate_dynamic_adgroups($adgroups, %options)

Базовая валидация динамических групп.

Параметры:
    $adgroups - ссылка на массив динамических групп ([Direct::Model::AdGroupDynamic])
    %options:
        with_banners => проверка группы объявлений вместе с вложенными баннерами.
                        Т.е. при with_banners => 1 будет выполнена проверка соответствия гео-таргетинга языку баннеров
                        и обязательность имени группы.
                        По умолчанию false.

        translocal_opt => параметры транслокальности (по умолчанию: {ClientID => $adgroup->client_id})

=cut

sub validate_dynamic_adgroups {
    my ($adgroups, %options) = @_;

    my $vr_main = Direct::ValidationResult->new();

    my $feed_owners = get_hash_sql(PPC(ClientID => [map { $_->client_id } @$adgroups]), [
        "select feed_id, ClientID from feeds",
        where => { feed_id => [map { $_->feed_id } grep { $_->has_feed_id && defined $_->feed_id } @$adgroups] },
    ]);

    for my $adgroup (@$adgroups) {
        my $vr = $vr_main->next;

        #
        # adgroup_name ("Название группы объявлений")
        #
        my $adgroup_name = $adgroup->has_adgroup_name ? $adgroup->adgroup_name : undef;
        if (!defined $adgroup_name || $adgroup_name !~ /\S/) {
            $vr->add(adgroup_name => error_ReqField(iget("Необходимо задать название группы объявлений")));
        }

        #
        # geo ("Гео-таргетинг группы объявлений")
        #
        my $geo = $adgroup->has_geo ? $adgroup->geo : undef;
        if (!(defined $geo && $geo =~/\S/)) {
            $vr->add(geo => error_ReqField(iget('Необходимо задать геотаргетинг группы объявлений')));
        } else {
            my $translocal_opt = $options{translocal_opt} // {ClientID => $adgroup->client_id};
            if (my $geo_error_text = validate_geo($geo, undef, $translocal_opt)) {
                $vr->add(geo => error_BadGeo($geo_error_text));
            } elsif ($options{with_banners}) {
                for my $banner (@{$adgroup->banners}) {
                    if (my $errors = validate_banner_geo_targeting($banner, $geo, $adgroup->campaign->content_lang, translocal_opt => $translocal_opt, for_adgroup => 1)) {
                        $vr->add(geo => $errors);
                        last;
                    }
                }
            }
        }

        #
        # minus_words ("Единые минус-слова на группу объявлений")
        #
        if ($adgroup->has_minus_words && @{$adgroup->minus_words}) {
            $vr->add(minus_words => validate_group_minus_words($adgroup->minus_words, MinusWords::polish_minus_words_array($adgroup->minus_words)));
        }

        #
        # main_domain ("Основной домен группы объявлений")
        # feed_id ("ID фида")
        #
        my $main_domain = $adgroup->has_main_domain ? $adgroup->main_domain : undef;
        my $feed_id = $adgroup->has_feed_id ? $adgroup->feed_id : undef;
        if (!!$main_domain + !!$feed_id != 1) {
            $vr->add(main_domain => error_ReqField('#feed_or_domain_required#'));
        } elsif ($main_domain) {
            # TODO: Проверка доступности домена?
            if ($main_domain =~ /\s/ || !is_valid_domain($main_domain)) { # is_valid_domain перед валидацией удаляет начальные/конечные пробельные символы, поэтому их валидируем сами
                $vr->add(main_domain => error_InvalidFormat(iget('Указан некорректный основной домен группы объявлений')));
            }
        } elsif ($feed_id) {
            if (!exists $feed_owners->{$feed_id} || $adgroup->client_id != $feed_owners->{$feed_id}) {
                $vr->add(feed_id => error_NotFound(iget('Указанный фид не существует')));
            }
        }

        #
        # href_params ("Параметры для подстановки в URL ссылки")
        #
        my $href_params = $adgroup->has_href_params ? $adgroup->href_params : undef;
        if (defined $href_params) {
            if (length($href_params) > 1024) {
                $vr->add(href_params => error_MaxLength(undef, length => 1024));
            }
        }

        #
        # hierarchical_multipliers
        #
        if ($adgroup->has_hierarchical_multipliers) {
            my $hierarchical_multipliers = $adgroup->hierarchical_multipliers;
            my $hm_vr = Direct::ValidationResult->new();

            # Вместо campaign->campaign_type будем передавать adgroup_type, т.к. они практически совпадают

            if ($hierarchical_multipliers->{mobile_multiplier}) {
                my $desktop_multiplier_pct =
                        $hierarchical_multipliers->{desktop_multiplier} ? $hierarchical_multipliers->{desktop_multiplier}->{multiplier_pct} : undef;
                $hm_vr->add(mobile_multiplier => validate_mobile_multiplier(
                        $adgroup->adgroup_type, $adgroup->client_id,
                        hash_merge(
                            $hierarchical_multipliers->{mobile_multiplier},
                            {desktop_multiplier_pct => $desktop_multiplier_pct}
                        )
                    ));
            }
            if ($hierarchical_multipliers->{desktop_multiplier}) {
                my $mobile_multiplier_pct =
                        ($hierarchical_multipliers->{mobile_multiplier} && !$hierarchical_multipliers->{mobile_multiplier}->{os_type})
                    ? $hierarchical_multipliers->{mobile_multiplier}->{multiplier_pct}
                    : undef;
                $hm_vr->add(desktop_multiplier => validate_desktop_multiplier(
                        $adgroup->adgroup_type, $adgroup->client_id,
                        hash_merge(
                            $hierarchical_multipliers->{desktop_multiplier},
                            {mobile_multiplier_pct => $mobile_multiplier_pct}
                        )
                    ));
            }
            if ($hierarchical_multipliers->{video_multiplier}) {
                $hm_vr->add(video_multiplier => validate_video_multiplier(
                    $adgroup->adgroup_type, $hierarchical_multipliers->{video_multiplier}
                ));
            }
            if ($hierarchical_multipliers->{demography_multiplier}) {
                $hm_vr->add(demography_multiplier => validate_demography_multiplier(
                    $adgroup->adgroup_type, $hierarchical_multipliers->{demography_multiplier}
                ));
            }
            if ($hierarchical_multipliers->{retargeting_multiplier}) {
                $hm_vr->add(retargeting_multiplier => validate_retargeting_multiplier(
                    $adgroup->adgroup_type, $adgroup->client_id, $hierarchical_multipliers->{retargeting_multiplier}
                ));
            }
            if ($hierarchical_multipliers->{performance_tgo_multiplier}) {
                $hm_vr->add(performance_tgo_multiplier => validate_performance_tgo_multiplier(
                    $adgroup->adgroup_type, $hierarchical_multipliers->{performance_tgo_multiplier}
                ));
            }

            $vr->add(hierarchical_multipliers => $hm_vr) unless $hm_vr->is_valid;
        }
    }

    return $vr_main;
}

=head2 validate_add_banners_dynamic_adgroup

    см. validate_update_banners_dynamic_adgroup

=cut

sub validate_add_banners_dynamic_adgroup {
    my ($banners, $adgroup, %options) = @_;
    return validate_add_banners_adgroup(
        { 'Direct::Model::BannerDynamic' => \&validate_add_dynamic_banners },
        $banners, $adgroup,
        translocal_tree => $options{translocal_tree},
    );
}

=head2 validate_update_banners_dynamic_adgroup

    Валидация баннеров добавляемых в динамическую группу

    Параметры:
        $banners - ссылка на массив баннеров ([])
        $adgroup - группа, в которую добавляются или которой принадлежат баннеры, объект типа Direct::Model::Adgroup
        %options:
            translocal_tree - параметры транслокальности

=cut

sub validate_update_banners_dynamic_adgroup {
    my ($banners, $adgroup, %options) = @_;
    return validate_banners_adgroup(
        { 'Direct::Model::BannerDynamic' => \&validate_update_dynamic_banners },
        $banners, $adgroup, translocal_tree => $options{translocal_tree}
    );
}

1;

