package Direct::Validation::AdGroupsMobileContent;

use Direct::Modern;

use base qw(Exporter);

use Yandex::I18n;
use List::MoreUtils qw/any none/;

use MobileContent;
use URLDomain qw/validate_banner_href/;
use Yandex::URL qw/get_host/;
use Yandex::CheckMobileRedirect qw/parse_store_url/;

use Direct::Validation::Errors;
use Direct::Validation::AdGroups qw/validate_banners_adgroup validate_add_banners_adgroup/;
use Direct::Validation::BannersMobileContent qw/validate_add_mobile_banners validate_update_mobile_banners/;
use Direct::Validation::BannersImageAd qw/validate_add_imagead_banners validate_update_imagead_banners/;
use Direct::Validation::BannersCpcVideo qw/validate_add_cpc_video_banners validate_update_cpc_video_banners/;


our @EXPORT_OK = qw/
    validate_add_mobile_adgroups
    validate_update_mobile_adgroups
    validate_add_banners_mobile_adgroup
    validate_update_banners_mobile_adgroup
/;

my @allowed_store_domains = ('play.google.com', 'itunes.apple.com', 'apps.apple.com');

=head2 validate_add_mobile_adgroups
    
см. описание validate_mobile_adgroups

=cut

sub validate_add_mobile_adgroups {
    my ($adgroups, $campaign, %options) = @_;
    
    my $vr_main = validate_mobile_adgroups($adgroups, translocal_tree => $options{translocal_tree});
    if ($campaign->adgroups_count + @$adgroups > $campaign->adgroups_limit) {
        $vr_main->add_generic(error_LimitExceeded(
            iget("Достигнуто максимальное количество групп объявлений в кампании - %d", $campaign->adgroups_limit)
        ));
    }
    return $vr_main;
}

=head2 validate_update_mobile_adgroups

см. описание validate_mobile_adgroups

=cut

sub validate_update_mobile_adgroups {    
    my ($adgroups, %options) = @_;
    return validate_mobile_adgroups(
        $adgroups,
        immutable_href => 1, translocal_tree => $options{translocal_tree},
        with_banners => 1
    );
}

=head2 validate_mobile_adgroups($adgroups, %options)

Параметры:
    $adgroups - [Direct::AdGroupMobileContent]
    %options - 
        immutable_href => 1|0 - 1 - проверить что рекламируемое приложение не изменилось
                            (для этого $adgroup должен реализовать роль Update)
        translocal_tree => транслокальное дерево, используемое для проверки соответствия текста баннеров региону
        with_banners => 1|0 см. Direct::Validation::AdGroups::validate_adgroups
    
Результат:
    $validation_result

=cut

sub validate_mobile_adgroups {
    my ($adgroups, %options) = @_;

    my $validation_result;
    {
        foreach (@$adgroups) {
            $_->campaign(new Direct::Model::Campaign(id => $_->campaign_id)) if (!$_->has_campaign && $_->has_campaign_id);
        }
        $validation_result = Direct::Validation::AdGroups::validate_adgroups($adgroups,
            with_banners => $options{with_banners},
            translocal_tree => $options{translocal_tree}
        );
    }

    my $i = 0;
    for my $adgroup (@$adgroups) {
        my $adgroup_vr = $validation_result->get_objects_results->[$i++];        
        my $href = $adgroup->has_store_content_href ? $adgroup->store_content_href : undef;
        if (!defined $href || $href !~ /\S/) {
            $adgroup_vr->add(store_content_href => error_ReqField(iget('Необходимо задать ссылку рекламируемого приложения')));
        } else {
            my $href_host = get_host($href);
            if (my @errors = validate_banner_href($href)) {
                $adgroup_vr->add(store_content_href => [map { error_InvalidFormat($_) } @errors]);
            } elsif (! grep { $_ eq $href_host } @allowed_store_domains ) {
                $adgroup_vr->add(store_content_href => error_UnsupportedStore(
                    iget('Магазин приложений не поддерживается (разрешены только %s)', join ', ', @allowed_store_domains))
                );
            } elsif (!parse_store_url($href)) {
                $adgroup_vr->add(store_content_href => error_InvalidFormat(
                    iget('Не удалось получить информацию о приложении'))
                );
            } else {
                my $new_content = parse_store_url($href);
                if ($options{immutable_href}) {
                    my $old_content = parse_store_url($adgroup->old->store_content_href);
                    if (any { $new_content->{$_} ne $old_content->{$_} } qw/store_content_id store_country os_type content_type/) {
                        $adgroup_vr->add(store_content_href => error_CantChangeMobileContent(
                            iget('Запрещено изменять рекламируемое приложение'))
                        );
                    }
                }

                my $min_os_version = $adgroup->has_min_os_version ? $adgroup->min_os_version : undef;
                if (!defined $min_os_version || $min_os_version !~ /\S/) {
                    $adgroup_vr->add(min_os_version => error_ReqField(iget('Необходимо задать минимальную версию ОС')));
                } elsif (none { $_ eq $min_os_version } @{$MobileContent::OS_VERSIONS{$new_content->{os_type}}}) {
                    $adgroup_vr->add(min_os_version => error_InvalidFormat(iget('Версия ОС задана некорректно')));
                }
            }
        }
        
        if (!$adgroup->has_device_type_targeting || !@{$adgroup->device_type_targeting}) {
            $adgroup_vr->add(device_type_targeting => error_ReqField('Необходимо задать устройства для таргетинга'));
        }
        
        if (!$adgroup->has_network_targeting || !@{$adgroup->network_targeting}) {
            $adgroup_vr->add(network_targeting => error_ReqField('Необходимо задать тип связи для таргетинга'));
        }
    }
    
    return $validation_result;
}

=head2 validate_add_banners_mobile_adgroup

    см. validate_update_banners_mobile_adgroup

=cut

sub validate_add_banners_mobile_adgroup {
    my ($banners, $adgroup, %options) = @_;
    return validate_add_banners_adgroup(
        { 'Direct::Model::BannerMobileContent' => \&validate_add_mobile_banners,
            'Direct::Model::BannerImageAd' => \&validate_add_imagead_banners,
            'Direct::Model::BannerCpcVideo' => \&validate_add_cpc_video_banners,
        },
        $banners, $adgroup,
        translocal_tree => $options{translocal_tree},
    );
}

=head2 validate_update_banners_mobile_adgroup

    Валидация баннеров добавляемых в мобильную группу

    Параметры:
        $banners - ссылка на массив баннеров ([])
        $adgroup - группа, в которую добавляются или которой принадлежат баннеры, объект типа Direct::Model::Adgroup
        %options:
            translocal_tree => параметры транслокальности

=cut

sub validate_update_banners_mobile_adgroup {
    my ($banners, $adgroup, %options) = @_;
    return validate_banners_adgroup(
        { 'Direct::Model::BannerMobileContent' => \&validate_update_mobile_banners,
            'Direct::Model::BannerImageAd' => \&validate_update_imagead_banners,
            'Direct::Model::BannerCpcVideo' => \&validate_update_cpc_video_banners
        },
        $banners, $adgroup, translocal_tree => $options{translocal_tree}
    );
}



1;
