package Direct::Validation::AdGroupsPerformance;
use Direct::Modern;
use base qw/Exporter/;

use Settings;
use Yandex::I18n;
use Yandex::Validate qw/is_valid_id/;

use Direct::Errors::Messages;
use Direct::Validation::AdGroups qw/validate_banners_adgroup validate_add_banners_adgroup/;
use Direct::Validation::BannersPerformance qw/validate_add_performance_banners validate_update_performance_banners/;

our @EXPORT_OK = qw/validate_add_performance_adgroups
                    validate_update_performance_adgroups
                    validate_add_banners_performance_adgroup
                    validate_update_banners_performance_adgroup
                   /;


our $FIELD_TO_USE_AS_NAME_MAX_LENGTH = 40;
our $FIELD_TO_USE_AS_BODY_MAX_LENGTH = $FIELD_TO_USE_AS_NAME_MAX_LENGTH;

=head2 validate_add_performance_adgroups

    my $validation_result = validate_add_performance_adgroups($adgroups, $campaign, %options);

Валидация групп при их добавлении к кампании.

=cut
sub validate_add_performance_adgroups {
    my ($adgroups, $campaign, %options) = @_;
    my $vr = validate_performance_adgroups($adgroups, %options);

    if ($campaign->adgroups_count + @$adgroups > $campaign->adgroups_limit) {
        $vr->add_generic(error_LimitExceeded(
            iget("Достигнуто максимальное количество групп объявлений в кампании - %d", $campaign->adgroups_limit)
        ));
    }
    return $vr;
}

=head2 validate_update_performance_adgroups

см. validate_performance_adgroups

=cut
sub validate_update_performance_adgroups {
    return validate_performance_adgroups(@_);
}


=head2 validate_performance_adgroups

    my $validation_result = validate_performance_adgroups($adgroups, %options);

Базовая валидация перфоманс групп.

Параметры:
    $adgroups - ссылка на массив перфоманс групп ([Direct::Model::AdGroupPerformance]).
    %options:
       - translocal_opts - опции для выбора геодерева

=cut
sub validate_performance_adgroups {
    my ($adgroups, %options) = @_;

    my $validation_result = Direct::Validation::AdGroups::validate_adgroups($adgroups);

    my $feeds = Direct::Feeds->get_by(
        [map { $_->client_id } @$adgroups],
        id => [ grep { $_ } map { $_->has_feed_id && $_->feed_id } @$adgroups ]
    )->items_by('id');

    my $vr_idx = 0;
    for my $adgroup (@$adgroups) {
        my $adgroup_vr = $validation_result->get_objects_results->[$vr_idx++];
        add_feed_validation_result($adgroup, $adgroup_vr, $feeds);

        if ($adgroup->has_field_to_use_as_name && defined $adgroup->field_to_use_as_name) {
            my $use_as_name = $adgroup->field_to_use_as_name;
            if (length $use_as_name > $FIELD_TO_USE_AS_NAME_MAX_LENGTH) {
                $adgroup_vr->add(field_to_use_as_name => error_InvalidField(iget('Имя поля для заголовка не должно превышать %d символов', $FIELD_TO_USE_AS_NAME_MAX_LENGTH)));
            }
        }
        
        if ($adgroup->has_field_to_use_as_body && defined $adgroup->field_to_use_as_body) {
            my $use_as_body = $adgroup->field_to_use_as_body;
            if (length $use_as_body > $FIELD_TO_USE_AS_BODY_MAX_LENGTH) {
                $adgroup_vr->add(field_to_use_as_body => error_InvalidField(iget('Имя поля для текста смарт-объявления не должно превышать %d символов', $FIELD_TO_USE_AS_BODY_MAX_LENGTH)));
            }
        }

        for my $banner (@{$adgroup->banners}) {
            my $geo_error = Direct::Validation::BannersPerformance::validate_creative_geo($banner->creative, $adgroup, translocal_opts => $options{translocal_opts});
            if ($geo_error) {
                $adgroup_vr->add(geo => $geo_error);
                last;
            }
        }
    }

    return $validation_result;
}


=head2 add_feed_validation_result

Валидирует указанные в группе фид, результаты валидации записывает в переданный ValidationResult.

=cut
sub add_feed_validation_result {
    my ($adgroup, $adgroup_vr, $feeds) = @_;

    if (!$adgroup->has_feed_id) {
        $adgroup_vr->add(feed_id => error_ReqField(iget('Идентификатор фида должен быть задан')));
        return;
    }

    if (!is_valid_id($adgroup->feed_id)) {
        $adgroup_vr->add(feed_id => error_InvalidField(iget('Идентификатор фида должен быть целым положительным числом')));
        return;
    }

    if (!exists $feeds->{$adgroup->feed_id} || $adgroup->client_id != $feeds->{$adgroup->feed_id}->client_id) {
        $adgroup_vr->add(feed_id => error_NotFound(iget('Указанный фид не существует')));
        return;
    }
}

=head2 validate_add_banners_performance_adgroup

    см. validate_update_banners_performance_adgroup

=cut

sub validate_add_banners_performance_adgroup {
    my ($banners, $adgroup, %options) = @_;
    return validate_add_banners_adgroup(
        { 'Direct::Model::BannerPerformance' => \&validate_add_performance_banners },
        $banners, $adgroup,
        translocal_tree => $options{translocal_tree},
        banners_count_to_delete => $options{banners_count_to_delete}
    );
}

=head2 validate_update_banners_performance_adgroup

    Валидация баннеров добавляемых в смарт-баннеры группу

    Параметры:
        $banners -> ссылка на массив баннеров ([])
        $adgroup -> группа, в которую добавляются или которой принадлежат баннеры, объект типа Direct::Model::Adgroup
        %options:
            translocal_tree - параметры транслокальности 
            banners_count_to_delete - 

=cut

sub validate_update_banners_performance_adgroup {
    my ($banners, $adgroup, %options) = @_;
    return validate_banners_adgroup(
        { 'Direct::Model::BannerPerformance' => \&validate_update_performance_banners },
        $banners, $adgroup, translocal_tree => $options{translocal_tree}
    );
}



1;
