package Direct::Validation::Banners;

use Direct::Modern;

use base qw(Exporter);

our @EXPORT = qw//;

our @EXPORT_OK = qw/
    validate_banners
    validate_banner_title
    validate_banner_body
    validate_banner_template
    validate_banner_geo_targeting
    validate_banner_display_href
    validate_delete_banner
/;

use Direct::Validation::Errors;
use Direct::ValidationResult;

use Direct::Validation::BannerImages qw/validate_banner_images/;
use Direct::Validation::HRef qw/validate_href/;
use Direct::Validation::TurboLandings qw/is_valid_turbolanding_href check_turbolandings_ownership/;
use Direct::Validation::Creative qw/validate_banner_creative/;

use Direct::BannersPermalinks;

use Settings qw/
    $MAX_TITLE_UNINTERRUPTED_LENGTH
    $MAX_BODY_UNINTERRUPTED_LENGTH
    $MAX_TITLE_LENGTH
    $NEW_MAX_TITLE_LENGTH
    $MAX_TITLE_LENGTH_CONTENT_PROMOTION
    $MAX_TITLE_EXTENSION_LENGTH
    $MAX_BODY_LENGTH
    $MAX_BODY_LENGTH_MOBILE
    $MAX_BODY_LENGTH_CONTENT_PROMOTION
    $DISALLOW_BANNER_LETTER_RE
    $MAX_NUMBER_OF_NARROW_CHARACTERS
    $NARROW_SYMBOLS_RE
/;
use TextTools qw/get_word_for_digit/;
use BannerTemplates qw/$TEMPLATE_METKA $TEMPLATE_METKA_OLD/;
use Yandex::I18n;
use List::MoreUtils qw/any uniq/;

use geo_regions;
use GeoTools qw/is_targeting_in_region/;


our $MAX_DISPLAY_HREF_LENGTH ||= 20;


my %REGION_ERRORS = (
    uk => {
        # Крым нужен для клиентов со страной Россия, им тоже позволенно создавать украинские объявления на Крым
        geo => [$geo_regions::UKR, $geo_regions::KRIM],
        error => {
            for_banner => iget_noop('Текст объявления написан на украинском языке и не соответствует установленному региону показа. Возможен таргетинг только на Украину.'),
            for_adgroup => iget_noop('Геотаргетинг не может быть изменён, так как текст объявления %s на украинском языке. Возможен таргетинг только на Украину.'),
        },
    },

    kk => {
        geo => [$geo_regions::KAZ],
        error => {
            for_banner  => iget_noop('Текст объявления написан на казахском языке и не соответствует установленному региону показа. Возможен таргетинг только на Казахстан.'),
            for_adgroup => iget_noop('Геотаргетинг не может быть изменён, так как текст объявления %s на казахском языке. Возможен таргетинг только на Казахстан.'),
        },
    },

    uz => {
        geo => [$geo_regions::UZB],
        error => {
            for_banner  => iget_noop('Текст объявления написан на узбекском языке и не соответствует установленному региону показа. Возможен таргетинг только на Узбекистан.'),
            for_adgroup => iget_noop('Геотаргетинг не может быть изменён, так как текст объявления %s на узбекском языке. Возможен таргетинг только на Узбекистан.'),
        },
    },

    tr => {
        geo => [$geo_regions::TR],
        error => {
            for_banner  => iget_noop('Текст объявления написан на турецком языке и не соответствует установленному региону показа. Возможен таргетинг только на Турцию.'),
            for_adgroup => iget_noop('Геотаргетинг не может быть изменён, так как текст объявления %s на турецком языке. Возможен таргетинг только на Турцию.'),
        },
    },

    be => {
        geo => [$geo_regions::BY],
        error => {
            for_banner => iget_noop('Текст объявления написан на белорусском языке и не соответствует установленному региону показа. Возможен таргетинг только на Беларусь.'),
            for_adgroup => iget_noop('Геотаргетинг не может быть изменён, так как текст объявления %s на белорусском языке. Возможен таргетинг только на Беларусь.'),
        },
    },

    vie => {
        geo => [$geo_regions::ASIA],
        error => {
            for_banner => iget_noop('Текст объявления написан на вьетнамском языке и не соответствует установленному региону показа. Возможен таргетинг только на Азию.'),
            for_adgroup => iget_noop('Геотаргетинг не может быть изменён, так как текст объявления %s на вьетнамском языке. Возможен таргетинг только на Азию.'),
        },
    },

    en => {
        geo => [$geo_regions::RUS, $geo_regions::KRIM],
        warning => iget_noop("Если для показов объявления выбран регион Россия, то текст объявления должен быть на русском языке или продублирован на русском языке."),
    },

    de => {
        geo => [$geo_regions::RUS, $geo_regions::KRIM],
        warning => iget_noop("Если для показов объявления выбран регион Россия, то текст объявления должен быть на русском языке или продублирован на русском языке."),
    },
);

# Символы, которые считаем пробелами
my $SPACES = " \x{00a0}"; # \x{00a0} -- неразрывный пробел

=head2 validate_delete_banner($banner, $adgroup)

    Проверка удаления баннера.

    Параметры:
        $banner - баннер (Direct::Model::Banner)
        $adgroup - группа, в которую входит баннер (Direct::Model::AdGroup)
            Direct::Model::AdGroup должен иметь поля: campaign (+ is_in_bs_queue)

    Результат:
        Ошибка валидации или undef

=cut

sub validate_delete_banner {

    my ($banner, $adgroup, $campaign) = @_;

    # У баннера есть номер в БК
    if ($banner->bs_banner_id > 0) {
        return error_BadStatus(iget('Объявление не может быть удалено'));
    }

    $campaign //= $adgroup->campaign;

    # Кампания на экспорте в БК и баннер не черновик
    if ($campaign->is_in_bs_queue && !($banner->status_moderate eq 'New' && $banner->status_post_moderate eq 'No')) {
        return error_BadStatus(iget('Объявление не может быть удалено'));
    }

    # Сумма зачисленного на кампании > 0 (TODO: Добавить поддержку ОС)
        # Баннер и группа приняты на модерации
        # Баннер принят или отклонен на постмодерации
        # Группа принята или отклонена на постмодерации и баннер тоже (?)
    if ($campaign->sum > 0) {
        if (
            ($banner->status_moderate eq 'Yes' && $adgroup->status_moderate eq 'Yes') ||
            (any { $banner->status_post_moderate eq $_ } qw/Yes Rejected/) ||
            ((any { $adgroup->status_post_moderate eq $_ } qw/Yes Rejected/) && $banner->status_post_moderate ne 'No')
        ) {
            return error_BadStatus(iget('Объявление не может быть удалено'));
        }
    }

    return undef;
}

=head2 validate_banners($banners, $adgroup, %params)

    Базовая проверка баннеров.

    Параметры:
        $banners - ссылка на массив баннеров [Direct::Model::Banner, ...]
        $adgroup - Direct::Model::AdGroup - группа в которой находится баннер
            должна содержать campaign с заполненным content_lang
        %params:
            translocal_opt => параметры транслокальности, используемые для проверки соответствия текста баннеров региону
            allowed_image_type => допустимые форматы изображения (small regular wide), если не указано - по-умолчанию qw/ regular wide /

    Результат:
        Результат проверки (ValidationResult)
        Ошибки хранятся по баннерам (field and generic)

=cut

sub validate_banners {
    my ($banners, $adgroup, %params) = @_;

    my $vr_main = Direct::ValidationResult->new();
    my $campaign = $adgroup->campaign;

    #Проверим турболендинги баннеров на принадлежность клиенту
    my ($turbolandings_by_client_id, $wrong_turbolandings);
    foreach my $banner (@$banners){
        next unless $banner->has_turbolanding;
        push @{$turbolandings_by_client_id->{$banner->client_id}}, $banner->turbolanding;
    }
    foreach my $client_id (keys %$turbolandings_by_client_id){
        my $invalid_turbolandings = check_turbolandings_ownership($client_id => $turbolandings_by_client_id->{$client_id});
        next unless $invalid_turbolandings;
        $wrong_turbolandings->{$_->id} //= 1 foreach @$invalid_turbolandings;
    }

    # Отдельно массово получаем доступные организации Справочника
    my $permalinks = [map { $_->permalink } grep { $_->is_permalink_supported() && $_->has_permalink } @$banners];
    my $valid_permalinks = {};
    if (@$permalinks) {
        my $client_id = $banners->[0]{client_id};
        my $chief_uid = Rbac::get_chief(ClientID => $client_id);
        my $access_by_uid = Direct::BannersPermalinks::check_organizations_access([ $chief_uid ], $permalinks) // {};
        $valid_permalinks = { map { $_ => 1 } @{$access_by_uid->{$chief_uid} // []} };
    }

    my @banner_images_to_validate;
    for my $banner (@$banners) {
        my $vr = $vr_main->next;

        #
        # title
        #
        my $title = $banner->title;
        if (!defined $title || $title !~ /\S/) {
            $vr->add(title => error_ReqField);
        } else {
            if ($banner->banner_type eq 'mobile_content'){
                $vr->add(title => validate_banner_title($title, use_mobile_content_validation => 1));
            } else {
                $vr->add(title => validate_banner_title($title));
            }
        }

        #
        # title_extension
        #
        my $title_extension = $banner->title_extension;
        if (defined $title_extension) {
            if ($title_extension !~ /\S/) {
                $vr->add( title_extension => error_EmptyField );
            } else {
                $vr->add( title_extension => validate_banner_title_extension($title_extension) );
            }
        }

        #
        # body
        #
        my $body = $banner->body;
        if (!defined $body || $body !~ /\S/) {
            $vr->add(body => error_ReqField);
        } else {
            if ($banner->banner_type eq 'mobile_content'){
                $vr->add(body => validate_banner_body($body, use_mobile_content_validation => 1));
            } else {
                $vr->add(body => validate_banner_body($body));
            }
        }

        #
        # template
        #
        validate_banner_template($banner, $vr);

        #
        # href
        #
        my $href = $banner->has_href ? $banner->href : undef;
        my $href_errors = [];
        if (defined $href) {
            if ($href !~ /\S/) {
                $vr->add(href => error_EmptyField);
            } else {
                $href_errors = validate_href($href);
                $vr->add(href => $href_errors);
                if (is_valid_turbolanding_href($href)) {
                    $vr->add(href => error_InconsistentState(iget('Ссылка баннера не может указывать на Турбо-страницу')));
                }
            }
        }

        #
        # display href
        #
        my $display_href = $banner->has_display_href ? $banner->display_href : undef;
        if (defined $display_href) {
            if (!($banner->has_href() && defined $banner->href)) {
                $vr->add_generic(error_InconsistentState(iget('Задана отображаемая ссылка, но не указана основная ссылка в объявлении')));
            }
            $vr->add(display_href => validate_banner_display_href($display_href));
        }

        #
        # sitelinks
        #
        if ($banner->sitelinks_set_id || $banner->has_sitelinks_set) {
            if (!(($banner->has_href && defined $banner->href) || $banner->has_turbolanding)) {
                $vr->add_generic(error_InconsistentState(iget('Заданы быстрые ссылки, но не указана основная ссылка в объявлении')));
            }
        }

        #
        # image
        #
        if ($banner->has_image_hash && defined $banner->image_hash && (!$banner->has_id || $banner->is_image_hash_changed)) {
            push @banner_images_to_validate, [$banner->image, $vr];
        }

        #
        # adgroup->geo (валидация геотаргетинга)
        #
        $vr->add(text_lang => validate_banner_geo_targeting($banner, $adgroup->geo, $campaign->content_lang, translocal_opt => $params{translocal_opt}));

        #
        # turbolanding
        #
        if ($banner->has_turbolanding && $wrong_turbolandings->{$banner->turbolanding->id}) {
            $vr->add(turbolanding => error_InconsistentState(iget('Задана несуществующая Турбо-страница')));
        }

        #дополнительные параметры ссылок турболендингов
        if ($banner->has_turbolanding_href_params) {
            if (length($banner->turbolanding_href_params // '') > $Settings::MAX_URL_LENGTH) {
                $vr->add(href_params => error_MaxLength(undef, length => $Settings::MAX_URL_LENGTH));
            }
        }

        $vr->add(creative => validate_banner_creative($adgroup, $banner));

        #
        # проверка доступа к организациям Справочника
        #
        if ($banner->is_permalink_supported() && $banner->has_permalink() && !$valid_permalinks->{$banner->permalink()}) {
            $vr->add(permalink => error_NotFound(iget('Организация не найдена')));
        }
    }

    # Отдельно массово валидируем изображения
    if (@banner_images_to_validate) {
        my $bim_vr_main = validate_banner_images([map { $_->[0] } @banner_images_to_validate], allowed_type => $params{allowed_image_type});
        my $i = -1;
        for (@banner_images_to_validate) {
            my ($bim, $banner_vr) = @$_;
            my $bim_vr = $bim_vr_main->get_objects_results->[++$i];
            if (!$bim_vr->is_valid) {
                $banner_vr->add(image => $bim_vr);
            }
        }
    }

    return $vr_main;
}

=head2 validate_banner_title($title)

    Проверка заголовка баннера.

    Параметры:
        $title - заголовок баннера
        %options:
        use_mobile_content_validation => валидировать заголовок объявления по правилам для РМП
        validate_title_extension => валидировать второй заголовок

    Результат:
        Массив с ошибками (или пустой массив, если ошибок не было).

=cut

sub validate_banner_title {
    my ($title, %options) = @_;

    return undef unless defined $title;

    my $count_title = scalar(my @x = $title =~ /$TEMPLATE_METKA/g);
    return error_BadTemplate(
        iget("В поле #field# можно использовать шаблон только один раз")
    ) if $count_title > 1;

    my $MAX_TITLE_UNINTERRUPTED_LENGTH_PLUS_1 = $MAX_TITLE_UNINTERRUPTED_LENGTH + 1;

    if ($title =~ $DISALLOW_BANNER_LETTER_RE) {
        return error_InvalidChars_AlphaNumPunct(iget("В поле #field# можно использовать только буквы латинского, турецкого, русского, украинского, казахского или белорусского алфавита, цифры и знаки пунктуации"));
    }

    my $validation_result;

    state $increase_ad_text_limits_prop = Property->new('increase_ad_text_limits');
    my $use_new_ad_text_limits = $increase_ad_text_limits_prop->get(120);
    my $max_title_length = $use_new_ad_text_limits ? $NEW_MAX_TITLE_LENGTH : $MAX_TITLE_LENGTH;

    if ($options{use_mobile_content_validation}){
        $validation_result = validate_banner_title_length_mobile($title, $max_title_length, $count_title);
    } elsif ($options{validate_title_extension}) {
        $validation_result = validate_banner_title_length($title, $MAX_TITLE_EXTENSION_LENGTH, $count_title);
    } else {
        $validation_result = $use_new_ad_text_limits ?
            validate_banner_title_length_mobile($title, $max_title_length, $count_title) :
            validate_banner_title_length($title, $max_title_length, $count_title);
    }

    if ($validation_result) {
        return $validation_result;
    } elsif ($title =~ /([^ \#\-]{$MAX_TITLE_UNINTERRUPTED_LENGTH_PLUS_1,})/) {
        my $exceeding_word = $1;
        my $truncated_exceeding_word = TextTools::truncate_text($exceeding_word, $MAX_TITLE_UNINTERRUPTED_LENGTH + 2, '...');
        my $msg = get_word_for_digit($MAX_TITLE_UNINTERRUPTED_LENGTH,
            iget_noop('В поле #field# недопустимы слова длиной более %d знака: "%s"'),
            iget_noop('В поле #field# недопустимы слова длиной более %d знаков: "%s"'),
            iget_noop('В поле #field# недопустимы слова длиной более %d знаков: "%s"'),
        );
        return error_BadUsage(
            iget($msg, $MAX_TITLE_UNINTERRUPTED_LENGTH, $truncated_exceeding_word)
        );
    } elsif ($title =~ /[$SPACES],[^$SPACES]/) {
        return error_BadUsage(
            iget('Неправильное использование знаков препинания в поле #field#')
        );
    }

    return undef;
}

=head2 validate_banner_title_extension($title_extension)

    Проверка второго заголовка баннера.

    Параметры:
        $title_extension - заголовок баннера

    Результат:
        Массив с ошибками (или пустой массив, если ошибок не было).

=cut

sub validate_banner_title_extension {
    my $title_extension = shift;

    return undef if !defined $title_extension;

    return validate_banner_title($title_extension, validate_title_extension => 1);
}

=head2 validate_banner_title_length($title, $number_of_templates)

Проверка длины заголовка баннера

Параметры:
    $title - заголовок баннера
    $max_length - максимальная длина заголовка
    $number_of_templates - количество шаблонов в заголовке

Возвращает ошибку или undef, если таковых нет.

=cut

sub validate_banner_title_length {
    my ($title, $max_length, $number_of_templates) = @_;

    my $count_of_narrow_symbols = (my $title_without_narrow_symbols = $title) =~ s/$NARROW_SYMBOLS_RE//g;
    my $length_without_narrow_symbols = length($title_without_narrow_symbols);

    if ($length_without_narrow_symbols > $max_length + 2 * $number_of_templates) {
        return error_MaxLength(undef, length => $max_length + 2 * $number_of_templates);
    } elsif ($count_of_narrow_symbols > $MAX_NUMBER_OF_NARROW_CHARACTERS) {
        my $msg = get_word_for_digit($MAX_NUMBER_OF_NARROW_CHARACTERS,
            iget_noop('В поле #field# вы можете использовать не более %d точки, запятой, двоеточия, точки с запятой, кавычки и восклицательного знака'),
            iget_noop('В поле #field# вы можете использовать не более %d точек, запятых, двоеточий, точек с запятой, кавычек и восклицательных знаков'),
            iget_noop('В поле #field# вы можете использовать не более %d точек, запятых, двоеточий, точек с запятой, кавычек и восклицательных знаков'),
        );
        return error_BadUsage(
            iget($msg, $MAX_NUMBER_OF_NARROW_CHARACTERS)
        );
    }
    return undef;
}

=head2 validate_banner_title_length_mobile($title, $number_of_templates)

Проверка длины заголовка баннера РМП

Параметры:
    $title - заголовок баннера
    $number_of_templates - количество шаблонов в заголовке

Возвращает ошибку или undef, если таковых нет.

=cut

sub validate_banner_title_length_mobile {
    my ($title, $max_length, $number_of_templates) = @_;

    if (length($title) > $max_length + 2 * $number_of_templates) {
        return error_MaxLength(undef, length => $max_length + 2 * $number_of_templates);
    }
    return undef;
}


=head2 validate_banner_body($body, %options)

Проверка основного текста баннера.

Параметры:
    $body - тело (текст) баннера
    %options:
        skip_template => проверять заголок баннера без учета наличия шаблона
        use_mobile_content_validation => валидировать текст объявления по правилам для РМП

Возвращает ошибку или undef, если таковых нет.

=cut

sub validate_banner_body {
    my ($body, %options) = @_;

    return undef unless defined $body;

    my $count_body = 0;

    if (!$options{skip_template}) {
        $count_body = scalar(my @x = $body =~ /$TEMPLATE_METKA/g);
        return error_BadTemplate(
            iget("В поле #field# можно использовать шаблон только один раз")
        ) if $count_body > 1;
    }

    my $MAX_BODY_UNINTERRUPTED_LENGTH_PLUS_1 = $MAX_BODY_UNINTERRUPTED_LENGTH + 1;

    if ($body =~ $DISALLOW_BANNER_LETTER_RE) {
        return error_InvalidChars_AlphaNumPunct(iget("В поле #field# можно использовать только буквы латинского, турецкого, русского, украинского, казахского или белорусского алфавита, цифры и знаки пунктуации"));
    }

    my $validation_result;

    if ($options{use_mobile_content_validation}){
        $validation_result = validate_banner_body_length_mobile($body, $count_body);
    } else {
        $validation_result = validate_banner_body_length($body, $count_body);
    }

    if ($validation_result) {
        return $validation_result;
    } elsif ($body =~ /([^ \#\-]{$MAX_BODY_UNINTERRUPTED_LENGTH_PLUS_1,})/) {
        my $exceeding_word = $1;
        my $truncated_exceeding_word = TextTools::truncate_text($exceeding_word, $MAX_BODY_UNINTERRUPTED_LENGTH + 2, '...');
        my $msg = get_word_for_digit($MAX_BODY_UNINTERRUPTED_LENGTH,
            iget_noop('В поле #field# недопустимы слова длиной более %d знака: "%s"'),
            iget_noop('В поле #field# недопустимы слова длиной более %d знаков: "%s"'),
            iget_noop('В поле #field# недопустимы слова длиной более %d знаков: "%s"'),
        );
        return error_BadUsage(
            iget($msg, $MAX_BODY_UNINTERRUPTED_LENGTH, $truncated_exceeding_word)
        );
    } elsif ($body =~ /[$SPACES],[^$SPACES]/) {
        return error_BadUsage(
            iget('Неправильное использование знаков препинания в поле #field#')
        );
    }

    return undef;
}

=head2 validate_banner_body_length($body, $number_of_templates)

Проверка длины текста баннера

Параметры:
    $body - тело (текст) баннера
    $number_of_templates - количество шаблонов в тексте

Возвращает ошибку или undef, если таковых нет.

=cut

sub validate_banner_body_length {
    my ($body, $number_of_templates) = @_;

    my $count_of_narrow_symbols = (my $body_without_narrow_symbols = $body) =~ s/$NARROW_SYMBOLS_RE//g;
    my $length_without_narrow_symbols = length($body_without_narrow_symbols);

    if ($length_without_narrow_symbols > $MAX_BODY_LENGTH + 2 * $number_of_templates) {
        return error_MaxLength(undef, length => $MAX_BODY_LENGTH + 2 * $number_of_templates);
    } elsif ($count_of_narrow_symbols > $MAX_NUMBER_OF_NARROW_CHARACTERS) {
        my $msg = get_word_for_digit($MAX_NUMBER_OF_NARROW_CHARACTERS,
            iget_noop('В поле #field# вы можете использовать не более %d точки, запятой, двоеточия, точки с запятой, кавычки и восклицательного знака'),
            iget_noop('В поле #field# вы можете использовать не более %d точек, запятых, двоеточий, точек с запятой, кавычек и восклицательных знаков'),
            iget_noop('В поле #field# вы можете использовать не более %d точек, запятых, двоеточий, точек с запятой, кавычек и восклицательных знаков'),
        );
        return error_BadUsage(
            iget($msg, $MAX_NUMBER_OF_NARROW_CHARACTERS)
        );
    }
    return undef;
}

=head2 validate_banner_body_length_mobile($body, $number_of_templates)

Проверка длины текста баннера РМП

Параметры:
    $body - тело (текст) баннера
    $number_of_templates - количество шаблонов в тексте

Возвращает ошибку или undef, если таковых нет.

=cut

sub validate_banner_body_length_mobile {
    my ($body, $number_of_templates) = @_;

    if (length($body) > $MAX_BODY_LENGTH_MOBILE + 2 * $number_of_templates) {
        return error_MaxLength(undef, length => $MAX_BODY_LENGTH_MOBILE + 2 * $number_of_templates);
    }
    return undef;
}

=head2 validate_banner_template($banner, $vr)

    Проверка шаблона в баннере.
    Проверяется использование шаблона в заголовке и теле баннера.

    Параметры:
        $banner - ссылка на баннер (Direct::Model::Banner)
        $vr     - объект Direct::ValidationResult для сохранения ошибок

    Результат:
        Массив с ошибками (или пустой массив, если ошибок не было).

=cut

sub validate_banner_template {
    my ($banner, $vr) = @_;
    my ($title, $title_extension, $body) = ($banner->title, $banner->title_extension, $banner->body);

    if (defined $title) {
        $vr->add(title => error_BadTemplate(iget('В поле #field# используется старый формат шаблона'))) if $title =~ /$TEMPLATE_METKA_OLD/;
        $vr->add(title => error_BadTemplate(iget('В поле #field# указан недопустимый шаблон'))) if $title =~ /^$TEMPLATE_METKA$/ && length($1) == 0;
        while ($title =~ /$TEMPLATE_METKA/g) {
            my $default_phrase = $1;
            if ($default_phrase =~ /^шаблон[ы]?$/i) {
                $vr->add(title => error_BadTemplate(iget('В поле #field# нельзя использовать "%s" в качестве фразы по умолчанию для шаблона', $default_phrase)));
            }
        }
    }

    if (defined $title_extension) {
        $vr->add(title_extension => error_BadTemplate(iget('В поле #field# используется старый формат шаблона'))) if $title_extension =~ /$TEMPLATE_METKA_OLD/;
        $vr->add(title_extension => error_BadTemplate(iget('В поле #field# указан недопустимый шаблон'))) if $title_extension =~ /^$TEMPLATE_METKA$/ && length($1) == 0;
        while ($title_extension =~ /$TEMPLATE_METKA/g) {
            my $default_phrase = $1;
            if ($default_phrase =~ /^шаблон[ы]?$/i) {
                $vr->add(title_extension => error_BadTemplate(iget('В поле #field# нельзя использовать "%s" в качестве фразы по умолчанию для шаблона', $default_phrase)));
            }
        }
    }

    if (defined $body) {
        $vr->add(body => error_BadTemplate(iget('В поле #field# используется старый формат шаблона'))) if $body =~ /$TEMPLATE_METKA_OLD/;
        $vr->add(body => error_BadTemplate(iget('В поле #field# указан недопустимый шаблон'))) if $body =~ /^$TEMPLATE_METKA$/ && length($1) == 0;
        while ($body =~ /$TEMPLATE_METKA/g) {
            my $default_phrase = $1;
            if ($default_phrase =~ /^шаблон[ы]?$/i) {
                $vr->add(body => error_BadTemplate(iget('В поле #field# нельзя использовать "%s" в качестве фразы по умолчанию для шаблона', $default_phrase)));
            }
        }
    }

    return $vr;
}

=head2 validate_banner_geo_targeting($banner, $geo, $content_lang, %options)

    Проверка корректности геотаргетинга у баннера по тексту баннера (заголовок, тело, сайтлинки).

    Параметры:
        $banner - ссылка на баннер (Direct::Model::Banner)
        $geo - строка с гео таргентингом (плюс и минус регионы)
        $content_lang - язык баннеров в кампании
        $options:
            for_adgroup    => проверка относительно группы объвлений
            translocal_opt => параметры транслокальности (по умолчанию: {ClientID => $banner->client_id})

    Результат:
        Массив с ошибками (или пустой массив, если ошибок не было).

=cut

sub validate_banner_geo_targeting {
    my ($banner, $geo, $content_lang, %options) = @_;

    my $translocal_opt = $options{translocal_opt};
    $translocal_opt //= $options{translocal_tree} ? {tree => $options{translocal_tree}} : {ClientID => $banner->client_id};
    my $lang = $content_lang || $banner->detect_lang;
    if (defined $lang && $REGION_ERRORS{$lang}
        && $REGION_ERRORS{$lang}->{error}
        && !is_targeting_in_region($geo, join(',', @{$REGION_ERRORS{$lang}->{geo}}), $translocal_opt)) {

        my $region = $REGION_ERRORS{$lang};
        return $options{for_adgroup}
            ? error_InvalidGeoTargeting(iget($region->{error}{for_adgroup}, $banner->id))
            : error_BadLang(iget($region->{error}{for_banner}));
    }

    return undef;
}


=head2 validate_banner_display_href($display_href)

    Проверка отображаемого урла.

    Параметры:
        $display_href

    Результат:
        Ошибка (или undef, если ошибок не было).

=cut

sub validate_banner_display_href {
    my ($display_href) = @_;

    return undef if !defined $display_href;

    return error_InvalidFormat(iget('Поле #field# не должно быть пустым'))  if !length $display_href;

    my $count_templates = scalar(my @x = $display_href =~ /$TEMPLATE_METKA/g);
    return error_BadTemplate(iget("В поле #field# можно использовать шаблон только один раз"))  if $count_templates > 1;

    return error_MaxLength(undef, length => $MAX_DISPLAY_HREF_LENGTH)
        if length($display_href) - 2*$count_templates > $MAX_DISPLAY_HREF_LENGTH;

    state $spec_chars = '-/№%#';
    state $allowed_letters = quotemeta join q// => uniq sort map {split //} (
        '0123456789',
        $spec_chars,
        $Settings::ALLOWED_ALPHABET_LETTERS,
    );

    my $invalid_chars = join q// => uniq($display_href =~ m|([^$allowed_letters])|gxms);
    return error_InvalidChars(iget('Недопустимые символы в поле #field#: <#chars#>. В поле разрешены только буквы латинского, турецкого, русского, украинского, казахского или белорусского алфавита, цифры, символы /- № # %'), chars => $invalid_chars)  if $invalid_chars;

    state $chars_re = quotemeta $spec_chars;
    state $doubles_re = qr/([$chars_re])\1/;
    return error_InvalidFormat(iget('Cпецсимволы в поле #field# не должны повторяться'))  if $display_href =~ $doubles_re;

    return undef;
}

1;
