package Direct::Validation::BannersAdditions;
use Direct::Modern;

use base qw/Exporter/;
our @EXPORT_OK = qw/validate_callouts validate_banner_link_callout/;

use List::MoreUtils qw/uniq/;
use Yandex::I18n;

use Settings;

use Direct::Validation::Errors;
use Direct::ValidationResult;

=head2 validate_callouts

Параметр: массив объектов Direct::Model::AdditionsItemCallout

=cut

sub validate_callouts {
    my ($items_callouts) = @_;

    my $vr = Direct::ValidationResult->new;

    if (@$items_callouts) {
        # подготовка для ограничения на клиента
        my $client_ids = [uniq map {$_->client_id} @$items_callouts];
        my $all_exists_callouts = Direct::AdditionsItemCallouts->get_by(client_id => $client_ids, with_deleted => 1)->items_by('client_id');

        my %new_callout_texts_count; # пришедшие уточнения
        for my $callout (@$items_callouts) {
            $new_callout_texts_count{$callout->client_id}->{$callout->callout_text}++;
        }

        my %uniq_callout_texts;
        my %uniq_callout_texts_with_deleted;
        my %first_limit_error; # какой из лимитов сработал первым, такую ошибку и продолжать отдавать для следующих коллаутов

        for my $client_id (@$client_ids) {
            my $exists_callouts = $all_exists_callouts->{$client_id};
            $uniq_callout_texts{$client_id} = {map {$_->callout_text => 1} grep {! $_->is_deleted} @$exists_callouts};
            $uniq_callout_texts_with_deleted{$client_id} = {map {$_->callout_text => 1} @$exists_callouts};
        }

        for my $callout (@$items_callouts) {
            my $vr_item = $vr->next;
            if (! $callout->has_callout_text) {
                $vr_item->add_generic(error_ReqField(iget("Не указан текст уточнения")));
            } elsif (length($callout->callout_text) == 0) {
                $vr_item->add_generic(error_EmptyField(iget("Не указан текст уточнения")));
            } elsif ($callout->callout_text =~ /^\s+$/) {
                $vr_item->add_generic(error_EmptyField(iget("Не указан текст уточнения")));
            } elsif (length($callout->callout_text) > $Settings::MAX_CALLOUT_LENGTH) {
                $vr_item->add_generic(error_MaxLength(iget("Превышена допустимая длина текста одного уточнения ('%s') в #length# символов", $callout->callout_text), length => $Settings::MAX_CALLOUT_LENGTH));
            } elsif ($callout->callout_text =~ $Settings::DISALLOW_BANNER_LETTER_RE || $callout->callout_text =~ /[!?]/) {
                $vr_item->add_generic(error_InvalidChars(iget('В тексте уточнений можно использовать только буквы латинского, турецкого, русского, украинского, казахского или белорусского алфавита, знаки пунктуации, за исключением !,?')));
            }

            # все уточнения должны быть уникальны
            if ($new_callout_texts_count{$callout->client_id}->{$callout->callout_text} > 1) {
                $vr_item->add_generic(error_DuplicateField(iget("Уточнения должны быть уникальны")));
            }

            # ограничения на клиента
            # если текста нет в старых коллаутах (или в обработанных вперед), и общее кол-во превышает лимит
            if (! exists $uniq_callout_texts{$callout->client_id}->{$callout->callout_text}
                && scalar(keys %{$uniq_callout_texts{$callout->client_id}}) + 1 > $Settings::MAX_CALLOUTS_COUNT_ON_CLIENT
                && ! exists $first_limit_error{$callout->client_id}
               )
            {
                $first_limit_error{$callout->client_id}->{active} = 1;
            }

            if (! exists $uniq_callout_texts_with_deleted{$callout->client_id}->{$callout->callout_text}
                && scalar(keys %{$uniq_callout_texts_with_deleted{$callout->client_id}}) + 1 > $Settings::MAX_CALLOUTS_COUNT_ON_CLIENT_WITH_DELETED
                && ! exists $first_limit_error{$callout->client_id}
               )
            {
                $first_limit_error{$callout->client_id}->{all} = 1;
            }

            if (exists $first_limit_error{$callout->client_id} && $first_limit_error{$callout->client_id}->{active}) {
                $vr_item->add_generic(error_ReachLimit(iget("Превышено количество уточнений (%s)", $Settings::MAX_CALLOUTS_COUNT_ON_CLIENT)));
            } elsif (exists $first_limit_error{$callout->client_id} && $first_limit_error{$callout->client_id}->{all}) {
                $vr_item->add_generic(error_ReachLimit(iget("Превышено общее количество уточнений (%s)", $Settings::MAX_CALLOUTS_COUNT_ON_CLIENT_WITH_DELETED)));
            }

            $uniq_callout_texts{$callout->client_id}->{$callout->callout_text} = 1;
            $uniq_callout_texts_with_deleted{$callout->client_id}->{$callout->callout_text} = 1;
        }
    }

    return $vr;
}

=head2 validate_banner_link_callout

Параметр: объект Direct::Model::Banner

=cut

sub validate_banner_link_callout {
    my ($banner) = @_;

    my $vr = Direct::ValidationResult->new;

    if ($banner->additions_callouts) {
        my $vr_callouts = validate_callouts($banner->additions_callouts);
        $vr->add_generic($vr_callouts->get_errors) if ! $vr_callouts->is_valid;

        # количество привязанных уточнений на баннер, предпологаем, что в объекте передан полный список уточнений
        if (scalar(@{$banner->additions_callouts}) > $Settings::MAX_CALLOUTS_COUNT_ON_BANNER) {
            $vr->add_generic(error_LimitExceeded(iget("К одному объявлению можно указать до %s уточнений", $Settings::MAX_CALLOUTS_COUNT_ON_BANNER)));
        }
    }

    return $vr;
}


=head2 validate_callouts_owner

Проверка на принадлежность уточнений клиенту
Параметры:
  массив объектов Direct::Model::AdditionsItemCallout
  ClientID

=cut

sub validate_callouts_owner($$) {
    my ($items_callouts, $ClientID) = @_;

    my $vr = Direct::ValidationResult->new;

    if (@$items_callouts) {
        # подготовка для ограничения на клиента
        my $all_callouts = Direct::AdditionsItemCallouts->get_by(client_id => $ClientID, id => [map {$_->id} @$items_callouts]);
        my $all_callouts_hash = {map {$_->id => 1} @{$all_callouts->items}};

        for my $callout (@$items_callouts) {
            my $vr_item = $vr->next;
            $vr_item->add_generic(error_BadCalloutID(iget("Задан неверный id уточнения: %s", $callout->id))) unless $all_callouts_hash->{$callout->id};
        }
    }

    return $vr;
}

1;
