package Direct::Validation::BannersCpcVideo;

use Direct::Modern;

use base qw(Exporter);

our @EXPORT = qw//;

our @EXPORT_OK = qw/
    validate_add_cpc_video_banners
    validate_update_cpc_video_banners
/;

use Client;
use Direct::Validation::Creative qw/validate_banner_creative/;
use Direct::Validation::Errors;
use Direct::Validation::HRef qw/validate_href/;
use Settings;

use Yandex::I18n qw/iget/;

=head2 validate_add_cpc_video_banners($banners, $adgroup)

    Валидация добавления ВО баннеров в группу.

=cut

sub validate_add_cpc_video_banners {
    my ($banners, $adgroup, %options) = @_;
    return _validate_cpc_video_banners($banners, $adgroup, translocal_tree => $options{translocal_tree});
}

=head2 validate_update_cpc_video_banners($banners, $adgroup)

    Валидация изменения ВО баннеров в группе.

=cut

sub validate_update_cpc_video_banners {
    my ($banners, $adgroup, %options) = @_;

    my $vr = _validate_cpc_video_banners($banners, $adgroup, translocal_tree => $options{translocal_tree});
    return $vr;
}

=head2 _validate_cpc_video_banners($banners, $adgroup, %options)

    Валидация ВО баннеров в группе.

    Параметры:
        $banners -> ссылка на массив ВО баннеров ([Direct::Model::BannersCpcVideo])
        $adgroup -> группа, в которую добавляются или которой принадлежат баннеры, объект типа Direct::Model::Adgroup
        %options:
            translocal_tree - параметры транслокальности 

=cut

sub _validate_cpc_video_banners {
    my ($banners, $adgroup, %options) = @_;

    my $vr_main = Direct::ValidationResult->new();

    if ($adgroup->type eq "mobile_content" && !Client::ClientFeatures::has_mobile_content_cpc_video_allowed($adgroup->client_id)) {
        $vr_main->add_generic(error_UnsupportedBanner(iget("Недопустимый тип баннера")));
        return $vr_main;
    }

    for my $banner (@$banners) {
        my $vr = $vr_main->next();

        my $adgroup_type = $adgroup->adgroup_type;
        my $banner_href  = $banner->has_href ? $banner->href : undef;
        if ($adgroup_type eq 'mobile_content') {
            if (defined $banner_href) {
                if ($banner_href !~ /\S/) {
                    $vr->add(href => error_InvalidField(iget('Неправильный формат ссылки')));
                } else {
                    my $errors = validate_href($banner_href);
                    if (@$errors) {
                        $vr->add(href => $errors);
                    } elsif (my $defect = URLDomain::is_valid_tracking_href($banner_href)) {
                        $vr->add(href => $defect);
                    }
                }
            }
        } else {
            if (!$banner_href || $banner_href !~ /\S/) {
                $vr->add(href => error_EmptyField(iget('В объявлении должна быть указана ссылка'))) if !$banner->has_turbolanding;
            } else {
                my $errors = validate_href($banner_href);
                $vr->add(href => $errors) if @$errors;
            }
        }
        $vr->add(creative => validate_banner_creative($adgroup, $banner));
    }


    return $vr_main;
}

1;

