package Direct::Validation::BannersCpmBanner;

use Direct::Modern;

use base qw(Exporter);

use Yandex::I18n;

use Direct::Validation::Banners qw/validate_banner_geo_targeting/;
use Direct::Validation::Errors;
use Direct::Validation::HRef qw/validate_href/;
use Direct::Validation::Pixels qw/validate_pixels/;
use Direct::Validation::Creative qw/validate_banner_creative/;
use Direct::ValidationResult;

our @EXPORT_OK = qw/
    validate_add_cpm_banner_banners
    validate_update_cpm_banner_banners
/;

=head2 validate_add_cpm_banner_banners($banners, $adgroup)

    Валидация добавления cpm_banner объявлений в группу.

    Параметры:
        см. _validate_cpm_banner_banners

=cut

sub validate_add_cpm_banner_banners {
    my ($banners, $adgroup, %options) = @_;
    return _validate_cpm_banner_banners($banners, $adgroup,
        translocal_tree => $options{translocal_tree}, permitted_pixel_providers => $options{permitted_pixel_providers});
}

=head2 validate_update_cpm_banner_banners($banners, $adgroup)

    Валидация обновления cpm_banner баннеров в группе

    Параметры:
        см. _validate_cpm_banner_banners
=cut

sub validate_update_cpm_banner_banners {
    my ($banners, $adgroup, %options) = @_;

    return _validate_cpm_banner_banners($banners, $adgroup,
        translocal_tree => $options{translocal_tree}, permitted_pixel_providers => $options{permitted_pixel_providers});
}

=head2 _validate_cpm_banner_banners($banners, $adgroup, %options)

    Валидация cpm_banner объявлений в группе.

    Параметры:
        $banners -> ссылка на массив cpm_banners баннеров ([Direct::Model::BannerCpmBanner, ...])
        $adgroup -> группа, в которую добавляются или которой принадлежат баннеры, объект типа Direct::Model::Adgroup
        %options:
            translocal_tree - параметры транслокальности
            permitted_pixel_providers - хэш доступных клиенту пикселов по типам кампаний

=cut

sub _validate_cpm_banner_banners {
    my ($banners, $adgroup, %options) = @_;

    my $vr_main = Direct::ValidationResult->new();

    my $permitted_pixel_providers;
    $permitted_pixel_providers = $options{permitted_pixel_providers}{$adgroup->campaign->campaign_type} if $options{permitted_pixel_providers};

    for my $banner (@$banners) {
        my $vr = $vr_main->next();

        my $banner_href  = $banner->has_href ? $banner->href : undef;
        if (!$banner_href || $banner_href !~ /\S/) {
            $vr->add(href => error_EmptyField(iget('В объявлении должна быть указана ссылка'))) 
                if !$banner->has_turbolanding;
        } else {
            my $errors = validate_href($banner_href);
            $vr->add(href => $errors) if @$errors;
        }

        if ($banner->has_pixels) {
            if (my $errors = validate_pixels($banner->pixels, $permitted_pixel_providers, $adgroup)) {
                if (!$errors->is_valid) {
                    $vr->add(pixels => $errors);
                }
            }
        }
        $vr->add(creative => validate_banner_creative($adgroup, $banner));
    }

    return $vr_main;
}

1;
