package Direct::Validation::BannersDynamic;

use Direct::Modern;

use base qw(Exporter);

use Settings;

use BannerTemplates qw/$TEMPLATE_METKA $TEMPLATE_METKA_OLD/;
use Yandex::I18n;

use Direct::Validation::Errors;
use Direct::ValidationResult;

use Direct::Validation::BannerImages qw/validate_banner_images/;
use Direct::Validation::Banners qw/
    validate_banner_body
    validate_banner_geo_targeting
    validate_delete_banner
/;
use Direct::Validation::BannersAdditions qw/validate_banner_link_callout/;

our @EXPORT = qw/
    validate_add_dynamic_banners
    validate_update_dynamic_banners
/;

=head2 validate_add_dynamic_banners($banners, $adgroup, %options)

Валидация добавления динамических баннеров в группу.

Параметры:
    $banners - ссылка на массив динамических баннеров ([Direct::Model::BannerDynamic, ...])
    $adgroup - группа, куда добавляются баннеры (Direct::Model::AdGroupDynamic)
    %options:
        translocal_opt => параметры транслокальности

Результат:
    Результат проверки (ValidationResult)
    Ошибки хранятся по баннерам (field and generic)

=cut

sub validate_add_dynamic_banners {
    my ($banners, $adgroup, %options) = @_;
    return validate_dynamic_banners($banners, $adgroup, translocal_opt => $options{translocal_opt});
}

=head2 validate_update_dynamic_banners($banners, $adgroup, %options)

Валидация обновления динамических баннеров в группе.

Параметры:
    $banners - ссылка на массив динамических баннеров ([Direct::Model::BannerDynamic, ...])
    $adgroup - общая группа обновляемых баннеров (Direct::Model::AdGroupDynamic)
    %options:
        translocal_opt => параметры транслокальности

Результат:
    Результат проверки (ValidationResult)
    Ошибки хранятся по баннерам (field and generic)

=cut

sub validate_update_dynamic_banners {
    my ($banners, $adgroup, %options) = @_;
    return validate_dynamic_banners($banners, $adgroup, translocal_opt => $options{translocal_opt});
}

# TODO: validate_suspend_dynamic_banners
# TODO: validate_resume_dynamic_banners
# TODO: validate_archive_dynamic_banners
# TODO: validate_unarchive_dynamic_banners
# TODO: validate_moderate_dynamic_banners

=head2 validate_dynamic_banners($banners, %options)

Базовая валидация динамических баннеров.

Параметры:
    $banners - ссылка на массив динамических баннеров [Direct::Model::BannerDynamic, ...]
    $adgroup - динамическая группа, в которой находятся баннеры (Direct::Model::AdGroupDynamic)
            должна содержать campaigns с заполненным camp_lang
    %options:
        translocal_opt => параметры транслокальности, используемые для проверки соответствия текста баннеров региону
        smart_check    => валидировать только измененные поля

Результат:
    Результат проверки (ValidationResult)
    Ошибки хранятся по баннерам (field and generic)

=cut

sub validate_dynamic_banners {
    my ($banners, $adgroup, %options) = @_;

    my $vr_main = Direct::ValidationResult->new();

    my @banner_images_to_validate;
    for my $banner (@$banners) {
        my $vr = $vr_main->next;

        next if $options{smart_check} && !$banner->is_changed;

        #
        # body ("Текст объявления")
        #
        if ($options{smart_check} ? $banner->is_body_changed : 1) {
            my $body = $banner->has_body ? $banner->body : undef;
            if (!defined $body || $body !~ /\S/) {
                $vr->add(body => error_ReqField());
            } else {
                if (my $body_error = validate_banner_body($body, skip_template => 1)) {
                    $vr->add(body => $body_error);
                } elsif ($body =~ /$TEMPLATE_METKA/ || $body =~ /$TEMPLATE_METKA_OLD/) {
                    $vr->add(body => error_BadTemplate(iget('В поле #field# использование шаблонов не допустимо')));
                }
            }
        }

        #
        # image ("Картинка объявления")
        #
        if ($options{smart_check} ? $banner->is_image_hash_changed : 1) {
            if ($banner->has_image_hash && defined $banner->image_hash && (!$banner->has_id || $banner->is_image_hash_changed)) {
                push @banner_images_to_validate, [$banner->image, $vr];
            }
        }

        #
        # adgroup->geo (валидация соответствия геотаргетингу группы)
        #
        $vr->add(text_lang => validate_banner_geo_targeting(
            $banner, $adgroup->geo, $adgroup->campaign->content_lang, translocal_opt => $options{translocal_opt}, smart_check => $options{smart_check}
        ));

        #
        # banner->additions_callouts (валидация дополнений к баннеру)
        #
        if ($banner->has_additions_callouts && @{$banner->additions_callouts}) {
            my $additions_vr = validate_banner_link_callout($banner);
            $vr->add(callout => $additions_vr);
        }
    }

    # Отдельно массово валидируем изображения
    if (@banner_images_to_validate) {
        my $bim_vr_main = validate_banner_images([map { $_->[0] } @banner_images_to_validate]);
        my $i = -1;
        for (@banner_images_to_validate) {
            my ($bim, $banner_vr) = @$_;
            my $bim_vr = $bim_vr_main->get_objects_results->[++$i];
            if (!$bim_vr->is_valid) {
                $banner_vr->add(image => $bim_vr);
            }
        }
    }

    return $vr_main;
}

1;
