package Direct::Validation::BannersImageAd;

use Direct::Modern;

use base qw(Exporter);

our @EXPORT = qw//;

our @EXPORT_OK = qw/
    validate_add_imagead_banners
    validate_update_imagead_banners
/;

use Direct::Validation::Errors;
use Direct::Validation::HRef qw/validate_href/;
use Direct::ValidationResult;
use Direct::Validation::BannerImages qw/validate_banner_images/;
use Direct::Validation::Banners qw/validate_banner_geo_targeting/;
use Settings;

use URLDomain qw//;

use Yandex::I18n qw/iget/;

=head2 validate_add_imagead_banners($banners, $adgroup)

    Валидация добавления новых графических баннеров в группу.

    Параметры:
        см. _validate_imagead_banners

=cut

sub validate_add_imagead_banners {
    my ($banners, $adgroup, %options) = @_;
    return _validate_imagead_banners($banners, $adgroup, translocal_tree => $options{translocal_tree});
}

=head2 validate_update_imagead_banners($banners, $adgroup)

    Валидация изменения графических баннеров в группе.

    Параметры:
        см. _validate_imagead_banners

=cut

sub validate_update_imagead_banners {
    my ($banners, $adgroup, %options) = @_;

    my $vr = _validate_imagead_banners($banners, $adgroup, translocal_tree => $options{translocal_tree});
    for my $i (0 .. $#$banners) {
        my $banner = $banners->[$i];
        my $banner_vr = $vr->get_objects_results->[$i];
        my $banner_width = $banner->get_width;
        if ($banner_width >= 0 && ($banner_width != $banner->old->get_width
            || $banner->get_height != $banner->old->get_height)) {

            $banner_vr->add_generic(error_InconsistentState(iget('Размеры нового и предыдущего изображений должны быть одинаковыми')));
        }
    }
    return $vr;
}

=head2 _validate_imagead_banners($banners, $adgroup, %options)

    Валидация графических баннеров в группе.

    Параметры:
        $banners -> ссылка на массив графических баннеров ([Direct::Model::BannersImageAd, ...])
        $adgroup -> группа, в которую добавляются или которой принадлежат баннеры, объект типа Direct::Model::Adgroup
        %options:
            translocal_tree - параметры транслокальности 

=cut

sub _validate_imagead_banners {
    my ($banners, $adgroup, %options) = @_;

    my $vr_main = Direct::ValidationResult->new();

    for my $banner (@$banners) {
        my $vr = $vr_main->next();

        my $adgroup_type = $adgroup->adgroup_type;
        my $banner_href  = $banner->has_href ? $banner->href : undef;
        if ($adgroup_type eq 'base') {
            if (!$banner_href || $banner_href !~ /\S/) {
                $vr->add(href => error_EmptyField(iget('В объявлении должна быть указана ссылка'))) if !$banner->has_turbolanding;
            } else {
                my $errors = validate_href($banner_href);
                $vr->add(href => $errors) if @$errors;
            }
        } elsif ($adgroup_type eq 'mobile_content') {
            if (defined $banner_href) {
                if ($banner_href !~ /\S/) {
                    $vr->add(href => error_InvalidField(iget('Неправильный формат ссылки')));
                } else {
                    my $errors = validate_href($banner_href);
                    if (@$errors) {
                        $vr->add(href => $errors);
                    } elsif (my $defect = URLDomain::is_valid_tracking_href($banner_href)) {
                        $vr->add(href => $defect);
                    }
                }
            }
        }
        else {
            croak 'Not supported adgroup type: '. $adgroup_type;
        }

        if ($banner->does('Direct::Model::BannerImage::Role::Url') && $banner->has_url) {
            my $img_url_vr = validate_banner_images([$banner]);
            $vr->add(image => $img_url_vr->get_errors) unless $img_url_vr->is_valid;
        }
        
        if (!$banner->has_creative && !$banner->has_image_ad && !$banner->does('Direct::Model::BannerImage::Role::Url')) {
            $vr->add_generic(error_InconsistentState(iget('Необходимо задать изображение или креатив')));
        }

        if ($banner->has_creative && $banner->has_image_ad) {
            $vr->add_generic(error_InconsistentState(iget('Невозможно одновременно задать креатив и изображение')));
        }

        $vr->add(text_lang => validate_banner_geo_targeting($banner, $adgroup->geo, $adgroup->campaign->content_lang, translocal_tree => $options{translocal_tree}));
    }

    return $vr_main;
}

1;
