package Direct::Validation::BannersPerformance;

use Direct::Modern;

use base qw(Exporter);

use Settings;

use Yandex::I18n;

use Direct::ValidationResult;
use Direct::Errors::Messages;
use Direct::Validation::Banners qw//;
use Direct::Model::VideoAddition;

use Direct::Feeds;

use GeoTools qw/moderation_countries/;
use List::MoreUtils qw/all any/;
use Readonly;

our @EXPORT = qw/
    validate_add_performance_banners
    validate_update_performance_banners
    validate_delete_performance_banners
    validate_performance_banners
    validate_video_additions_layout_id
/;

=head2 %FEEDS_TO_SMART_CREATIVES_COMPATIBILITY

Хеш соответствий 'бизнес-тип фида' => 'бизнес-тип креатива'

=cut

Readonly::Hash our %FEEDS_TO_SMART_CREATIVES_COMPATIBILITY => (
    'retail' => ['retail', 'pharm'],
    'hotels' => ['hotels'],
    'auto' => ['auto'],
    'realty' => ['realty'],
    'news' => ['news'],
    'flights' => ['flights'],
    'other' => ['other', 'pharm'],
);

=head2 is_feed_compatible_with_creative($feed_business_type, $creative_business_type)

=cut

sub is_feed_compatible_with_creative {
    my ($feed_business_type, $creative_business_type) = @_;
    my $compatible_business_types = $FEEDS_TO_SMART_CREATIVES_COMPATIBILITY{$feed_business_type};
    return any {$creative_business_type eq $_} @$compatible_business_types;
}

=head2 validate_add_performance_banners($banners, $adgroup)

=cut

sub validate_add_performance_banners {
    my ($banners, $adgroup, %options) = @_;
    return validate_performance_banners($banners, $adgroup);
}

=head2 validate_update_performance_banners($banners, $adgroup)

=cut

sub validate_update_performance_banners {
    my ($banners, $adgroup) = @_;
    return validate_performance_banners($banners, $adgroup);
}


=head2 validate_delete_performance_banners($banners, $adgroup, $campaign)

=cut

sub validate_delete_performance_banners {
    my ($banners, $adgroup, $campaign) = @_;

    my $vr_main = Direct::ValidationResult->new();

    for my $banner (@$banners) {
        $vr_main->next->add_generic(Direct::Validation::Banners::validate_delete_banner($banner, $adgroup, $campaign));
    }

    return $vr_main;
}

=head2 validate_performance_banners

Валидация баннеров. На вход принимает полный набор баннеров группы (т.е. то, что должно получится после
выполнения валидируемой операции).

    my $validation_result = validate_performance_banners(\@banners);

=cut
sub validate_performance_banners {
    my ($banners, $adgroup) = @_;
    my $vr = Direct::ValidationResult->new;

    # ??? нехорошо при валидации ходить в БД
    my $feeds = Direct::Feeds->get_by($adgroup->client_id, id => $adgroup->feed_id)->items_by('id');

    my %creative_count;
    $creative_count{$_->creative_id}++ for @$banners;
    for my $banner (@$banners) {
        my $banner_vr = $vr->next;
        if ($creative_count{$banner->creative_id} > 1) {
            $banner_vr->add_generic(error_Duplicated(iget("Креатив %s используется более одного раза в рамках группы", $banner->creative_id)));
        }
        if ($banner->creative->client_id != $banner->client_id) {
            $banner_vr->add(creative => error_NotFound(iget('Креатив не принадлежит клиенту')));
        } elsif (all {$banner->creative->status_moderate ne $_ } qw/New Error/) {
            my $geo_error = validate_creative_geo($banner->creative, $adgroup);
            $banner_vr->add(creative => $geo_error) if $geo_error;
        }
        my $creative_business_type = $banner->creative->business_type;
        my $feed_business_type = $feeds->{$adgroup->feed_id}->business_type;
        if (!is_feed_compatible_with_creative($feed_business_type, $creative_business_type)) {
            $banner_vr->add(creative => error_NotEligibleObject(iget("Тип креатива не соответствует типу фида")));
        }
    }

    return $vr;
}

=head2 validate_creative_geo($creative, $adgroup, %options)

Проверка возможности использования креатива в группе

Параметры:
    $creative - креатив (Direct::Model::Creative)
    $adgroup - группа (Direct::Model::AdGroup)
    $options:
        translocal_opt => параметры транслокальности (по умолчанию: {ClientID => $adgroup->client_id})

Результат:
    Direct::Defect или undef если ошибок нет 

=cut

sub validate_creative_geo {
    my ($creative, $adgroup, %options) = @_;
    $options{translocal_opts} //= {ClientID => $adgroup->client_id};

    return undef unless defined $creative->sum_geo;

    my %adgroup_countries = map { $_ => undef } split /,/, moderation_countries($adgroup->geo, $options{translocal_opts});
    my %creative_countries = map { $_ => undef } split /,/, $creative->sum_geo;
    return (all { exists $creative_countries{$_} } keys %adgroup_countries)
        ? undef
        : error_BadGeo(iget('Геотаргетинг группы шире списка стран её cмарт-баннеров'));
}

=head2 validate_video_additions_layout_id($banner_type, $creative)

Проверка layout_id креатива в text,mobile_context баннерах

Параметры:
    $creative - креатив (Direct::Model::Creative)

Результат:
    текст ошибки

=cut

sub validate_video_additions_layout_id {
    my ($banner_type, $creative) = @_;

    return undef if !$creative->has_layout_id || !defined $creative->layout_id;

    my $layout_id = $creative->layout_id;
    my $layout_range = $Direct::Model::VideoAddition::COMPATIBLE_LAYOUT_IDS{$banner_type};
    croak "Unsupported type: $banner_type" if !$layout_range;

    return undef if any {$layout_id == $_} @$layout_range;
    return iget("Указан несовместимый шаблон видео-дополнения");
}

1;
