package Direct::Validation::Bids;

use Direct::Modern;

use base qw(Exporter);

use List::MoreUtils qw/each_array any/;
use Settings;
use Yandex::I18n;
use Direct::Errors::Messages;
use Currencies qw/get_currency_constant/;
use Currency::Format qw/format_const/;
use Client;
use Campaign;

our @EXPORT = qw/
    validate_relevance_matches_for_adgroup
    /;


=head2 validate_relevance_matches_for_adgroup($relevance_match, $adgroup, $campaign)

Валидация списка условий беcфразного таргетинга.
Параметры:
    $checked_target_interests   -> список проверяемых условий беcфразного таргетинга
    $adgroup        -> общая группа для таргетирования по интересам (Direct::Model::AdGroup)
    $campaign       -> общая кампания для таргетирования по интересам (HashRef)

=cut

sub validate_relevance_matches_for_adgroup {
    my ($relevance_matches, $adgroup, $campaign) = @_;

    my $vr_main = Direct::ValidationResult->new();

    # TODO: в первой версии беcфразного таргентинга условий показа может быть не более одного, но позже планируется добавить поддержку нескольких (отдельно сеть, отдельно поиск)
    # поэтому по коду везде рассматриваем relevance_match как массив
    if (scalar @$relevance_matches > 1) {
        $vr_main->add_generic(error_LimitExceeded(iget("Группа объявлений может содержать не более одного автотаргетинга")));
    }
    
    if ($adgroup->is_archived) {
        $vr_main->add_generic(error_BadStatus(iget('Группа объявлений № %s находится в архиве - редактирование ставок невозможно', $adgroup->id)));
    }
    
    return $vr_main unless $vr_main->is_valid;

    my ($currency, $min_price, $max_price);
    if (exists $campaign->{strategy}) { # если кампания не задана, то не проверяем соответствие стратегии и ставок
        $currency = $campaign->{currency};
        $min_price = get_currency_constant($currency, 'MIN_PRICE');
        $max_price = get_currency_constant($currency, 'MAX_PRICE');
    }
    
    my $strategy = $campaign->{strategy};
    if ($strategy && (any {$_->has_old && $_->has_price && $_->old->price != $_->price} @$relevance_matches)){
        $vr_main->add_generic(error_InvalidField(iget('Выбранная стратегия не предполагает ручного изменения ставок'))) if $strategy->{is_autobudget};
    }

    return $vr_main unless $vr_main->is_valid;

    my $is_search_stop = $campaign->{strategy}->{is_search_stop};
    my $is_net_stop = $campaign->{strategy}->{is_net_stop};
    my $is_different_places =  defined $campaign->{strategy}->{name} && $campaign->{strategy}->{name} eq 'different_places';
    my $has_extended_relevance_match = Campaign::has_context_relevance_match_feature($campaign->{type}, $campaign->{ClientID});
    
    for my $relevance_match (@$relevance_matches) {
        my $vr = $vr_main->next;

        for my $param_name (qw/href_param1 href_param2/) {
            my $has_param = "has_${param_name}";
            my $param_value = $relevance_match->$has_param ? $relevance_match->$param_name : undef;

            next if !defined $param_value;

            if (length($param_value) > $Settings::MAX_HREF_PARAM_LENGTH) {
                $vr->add($param_name => error_MaxLength());
            }

            if ($param_value =~ $Settings::DISALLOW_BANNER_LETTER_RE) {
                $vr->add($param_name => error_InvalidChars_AlphaNumPunct());
            }
        }

        # если кампания не задана, то не проверяем соответствие стратегии и ставок
        next unless $strategy;

        if ($campaign->{strategy}->{is_autobudget}) {
            my $autobudget_priority = $relevance_match->has_autobudget_priority ? $relevance_match->autobudget_priority : undef;
            if (!defined $autobudget_priority) {
                $vr->add(autobudget_priority => error_ReqField());
            } elsif ($autobudget_priority !~ /^(?:1|3|5)$/) {
                $vr->add(autobudget_priority => error_InvalidField());
            }
        } else {
            if (!$is_search_stop){
                if (!$relevance_match->has_price) {
                    $vr->add('price' => error_ReqField());
                } else {
                    if ($relevance_match->price < $min_price) {
                        $vr->add('price' => error_InvalidField(iget("Значение в поле #field# не может быть меньше %s", format_const($currency, 'MIN_PRICE'))));
                    }
                    if ($relevance_match->price > $max_price) {
                        $vr->add('price' => error_InvalidField(iget("Значение в поле #field# не может быть больше %s", format_const($currency, 'MAX_PRICE'))));
                    }
                }
            }

            if ($has_extended_relevance_match && ($is_different_places || $is_search_stop)) {
                if (!$relevance_match->has_price_context) {
                    $vr->add('price_context' => error_ReqField());
                    next;
                }
                if ($relevance_match->price_context < $min_price) {
                    $vr->add('price_context' => error_InvalidField(iget("Значение в поле #field# не может быть меньше %s", format_const($currency, 'MIN_PRICE'))));
                }
                if ($relevance_match->price_context > $max_price) {
                    $vr->add('price_context' => error_InvalidField(iget("Значение в поле #field# не может быть больше %s", format_const($currency, 'MAX_PRICE'))));
                }
            }
        }
    }

    return $vr_main;
}


1;
