
package Direct::Validation::Campaigns;

use Direct::Modern;

use base qw/ Exporter /;

our @EXPORT = qw/
    validate_campaign_geo
    validate_campaigns
    validate_add_campaign
    validate_update_campaign
    validate_delete_campaigns
    validate_resume_campaigns
    validate_suspend_campaigns
    validate_copy_campaigns_for_client
    validate_broad_match_flag
    validate_ab_segments
    validate_campaign_internal_distrib
    validate_campaign_internal_free
/;

use Try::Tiny;

use Client qw/ check_add_client_campaigns_limits get_client_limits/;

use Campaign qw/ is_cpm_campaign is_internal_distrib_camp is_internal_free_camp get_strategies get_campaigns_type_by_strategy_id /;
use Campaign::Types qw/ get_camp_kind_types /;
use CampaignTools;

use DeviceTargeting qw/ is_valid_device_targeting /;

use Direct::Errors::Messages::Client;
use Direct::Validation::Banners qw/ validate_banner_geo_targeting /;
use Direct::Validation::Campaigns::Constraints;
use Direct::Validation::Domains qw/ validate_disabled_domains /;
use Direct::Validation::Errors;
use Direct::Validation::HierarchicalMultipliers qw/ validate_hierarchical_multipliers /;
use Direct::Validation::Ips qw/ validate_disabled_ips /;
use Direct::Validation::MinusWords qw/ validate_campaign_minus_words /;
use Direct::ValidationResult;
use JavaIntapi::InternalAdPlaces::CanControlPlace;
use Stat::Tools;
use MinusWords;
use Models::AdGroup;
use PrimitivesIds;
use Currencies qw/get_currency_constant/;
use Currency::Format qw/format_const/;
use Campaign::Const qw/@PAGE_TYPE_ENUM/;
use Rbac qw/:const/;

use TextTools qw/get_num_array_by_str/;
use GeoTools qw/ validate_geo /;

use List::Util qw/ first none /;
use List::MoreUtils qw/ any uniq /;

use TimeTarget qw//;

use Yandex::I18n qw/ iget /;
use Yandex::IDN qw/ is_valid_domain is_valid_email /;
use Yandex::TimeCommon qw/ mysql2unix ts_round_day /;
use Yandex::Validate;
use Yandex::ListUtils qw/nsort xminus/;

# NB: переименовать, если потребуется Settings.pm, есть конфликт в названиях переменных (https://st.yandex-team.ru/DIRECT-107250#5db340548c7003001d3c65ed)
our $MAX_TITLE_LENGTH           = 255;
our $MAX_CLIENT_FIO_LENGTH      = 255;
our $MAX_EMAIL_LENGTH           = 255;
our $MAX_METRIKA_COUNTER_ID     = 2**31 - 1;
our $MAX_METRIKA_COUNTERS_COUNT_WITH_FEATURE = 100;
our $SMS_TIME_MULTIPLICITY      = 15;
our $MONEY_THRESHOLD_MIN        = 1;
our $MONEY_THRESHOLD_MAX        = 50;
our @POSITION_CHECK_INTERVAL    = qw/ 15 30 60 /;

# максимальное количество аб экспериментов на кампанию
our $MAX_COUNT_CAMP_AB_SECTIONS = 5;
# максимальное количество аб сегментов в эксперименте
our $MAX_COUNT_CAMP_AB_SEGMENTS_IN_SECTION = 100;

#максимальное количество стратегий и незаархивированных стратегий на клиента
our $MAX_STRATEGIES_COUNT = 1500;
our $MAX_UNARC_STRATEGIES_COUNT = 500;

our $MAX_NUMBER_OF_CAMPAIGNS_IN_STRATEGY = 100;

=head2 validate_campaign_geo

    Проверка геотаргетинга кампании

    Позиционные параметры:
        $campaign - кампания

    Именованные параметры:
        translocal_tree - использование транслокального дерева (для АПИ)
        with_banners    - проверять соответствие текстов баннеров и сайтлинков геотаргетингу

    аналог Common::validate_common_geo

=cut

sub validate_campaign_geo {
    my ( $camp, %options ) = @_;

    my $translocal_opt = $options{translocal_tree}
                            ? { tree => $options{translocal_tree} }
                            : { ClientID => $camp->client_id };

    my $vr = Direct::ValidationResult->new;

    my $geo_error = validate_geo( $camp->geo, undef, $translocal_opt );
    if ( $geo_error ) {
        $vr->add_generic( error_BadGeo( $geo_error ) );
    } else {
        if ( $options{with_banners} ) {

            my $error;
            my %langs;
            foreach my $group ( @{ $camp->adgroups } ) {
                foreach my $banner ( @{ $group->banners } ) {

                    # возвращаем только первую ошибку о несоответствии текстов баннера и его
                    # сайтлинков геотаргетингу кампании, т.к. в ином случае при указании
                    # некорректного значения клиент получит огромное кол-во однотипных ошибок
                    if ( ! $error ) {
                        $error = validate_banner_geo_targeting(
                            $banner, $camp->geo, $camp->content_lang,
                            for_adgroup    => 1,
                            translocal_opt => $translocal_opt
                        );
                        if ( $error ) {
                            $vr->add_generic( $error );
                        }
                    }
                }
            }
        }
    }

    return $vr;
}

=head2 _base_internal_validate

    Базовые валидации кампаний внутренней рекламы

    Позиционные параметры:
        $campaign - кампания
        $vr - объект типа Direct::ValidationResult
        $operator_uid - uid оператора

=cut

sub _base_internal_validate {
    my ( $camp, $vr, $operator_uid ) = @_;

    if ( !defined $camp->{is_mobile} ) {
        $vr->add( is_mobile => error_ReqField(iget('Не передано обязательное поле #field#'), field => iget("'Реклама мобильного приложения'")) );
    } elsif (!($camp->{is_mobile} == 1 || $camp->{is_mobile} == 0)) {
        $vr->add(is_mobile => error_InvalidField(
                    iget('Значение поля #field# должно быть равно 0 или 1'), field => iget("'Реклама мобильного приложения'")));
    }

    if ( defined $camp->{page_ids} ) {
        if ($camp->{page_ids} !~ /^[1-9][0-9]*(?:,[1-9][0-9]*)*$/) {
            $vr->add(page_ids => error_InvalidField(
                    iget('Значение поля #field# должно быть списком чисел через запятую'), field => iget("'Список площадок'")));
        }
    }

    if ( !defined $camp->{place_id}) {
        $vr->add( place_id => error_ReqField(iget('Не передано обязательное поле #field#'), field => iget("'Place id'")) );
    } else {
        if ( !is_valid_id($camp->{place_id}) ) {
            $vr->add(place_id => error_InvalidField(
                    iget('Значение поля #field# должно быть целым положительным числом'), field => iget("'Place id'")));
        }

        my $ClientID = get_clientid( login => $camp->{ulogin} );
        my $place_allowed = JavaIntapi::InternalAdPlaces::CanControlPlace->new(
            operator_uid => $operator_uid,
            product_client_id => $ClientID,
            place_id => $camp->{place_id},
        )->call;
        unless ($place_allowed) {
            $vr->add( place_id => error_InvalidField(
                iget('Размещать кампании на этом плейсе нельзя') ) );
        }
    }

    return $vr;
}

=head2 validate_campaign_internal_free

    Проверка бесплатных кампаний внутренней рекламы

    Позиционные параметры:
        $campaign - кампания
        $operator_uid - uid оператора

=cut

sub validate_campaign_internal_free {
    my ( $camp, $operator_uid ) = @_;
    my $vr = Direct::ValidationResult->new();
    _base_internal_validate($camp, $vr, $operator_uid);
    if ( defined $camp->{restriction_type} ) {
        my $value = $camp->{restriction_type};
        if (! any {$value eq $_} ('shows', 'clicks', 'days')) {
            $vr->add(restriction_type => error_InvalidField(iget('Значение поля #field# должно входить в ["shows", "clicks", "days"]'), field => iget("'Единица измерения'")));
        }
    } else {
        $vr->add( restriction_type => error_ReqField(iget('Не передано обязательное поле #field#'), field => iget("'Единица измерения'")));
    }

    if ( defined $camp->{restriction_value} ) {
        if ( $camp->{restriction_value} < 0) {
            $vr->add(restriction_value => error_InvalidField(iget('Значение поля #field# должно быть целым положительным числом'), field => iget("'Количество единиц'")));
        }
    } else {
        $vr->add( restriction_value => error_ReqField(iget('Не передано обязательное поле #field#'), field => iget("'Количество единиц'")) );
    }

    return $vr;
}

=head2 validate_campaign_internal_distrib

    Проверка дистрибуционных кампаний внутренней рекламы

    Позиционные параметры:
        $campaign - кампания
        $operator_uid - uid оператора

=cut

sub validate_campaign_internal_distrib {
    my ( $camp, $operator_uid ) = @_;

    my $vr = Direct::ValidationResult->new();

    _base_internal_validate($camp, $vr, $operator_uid);

    if ( defined $camp->{restriction_type} ) {
        if ( $camp->{restriction_type} ne 'money' ) {
            $vr->add(restriction_type => error_InvalidField(iget('Значение поля #field# должно быть равно "money"'), field => iget("'Единица измерения'")));
        }
    } else {
        $vr->add( restriction_type => error_InvalidField(iget('Не указано значение в поле #field#'), field => iget("'Единица измерения'")) );
    }

    if ( defined $camp->{restriction_value} ) {
        if ( $camp->{restriction_value} != 0) {
            $vr->add(restriction_value => error_InvalidField(iget('Значение поля #field# должно быть равно 0'), field => iget("'Количество единиц'")));
        }
    }

    if ( !defined $camp->{rotation_goal_id} ) {
        $vr->add( rotation_goal_id => error_ReqField(iget('Не передано обязательное поле #field#'), field => iget("'Id цели в Метрике'")) );
    } elsif ( !is_valid_int($camp->{rotation_goal_id}) ) {
        $vr->add( rotation_goal_id => error_InvalidField(
                    iget('Значение поля #field# должно быть целым числом'), field => iget("'Id цели в Метрике'")));
    } elsif ( $camp->{rotation_goal_id} < -1 ) {
        $vr->add( rotation_goal_id => error_InvalidField(
                    iget('Значение поля #field# должно быть целым положительным числом или 0 или -1'), field => iget("'Id цели в Метрике'")));
    } elsif ( !$camp->{is_mobile} && $camp->{rotation_goal_id} == 3 ) {
        $vr->add( rotation_goal_id => error_InvalidField(
                    iget('Значение поля #field# может быть 3 только для рекламы мобильных приложений'), field => iget("'Id цели в Метрике'")));
    }

    return $vr;
}



=head2 validate_campaigns

    Проверка списка кампаний

    аналог Common::validate_camp + новые модели

    Параметры:
        $campaigns - ссылка на массив кампаний, Direct::Model::Campaigns
            может содержать группы Direct::Model::AdGroup (каждая группа содержит баннеры, Direct::Model::Banner)
        %options
            is_easy         - выполняется ли проверка для пользователя легкого интерфейса
            translocal_tree - транслокальное дерево, используемое для проверки соответствие текстов баннеров и сайтлинков региону
            skip_not_changed_meaningful_goals - валидировать ключевые цели только если они менялись
            prefetched_meaningful_goals_data

    Результат:

    NB:
        * Поведение будет отличаться от исходника в порядке ошибок - нужно ли поддерживать?
        * что вернет has_xxx если в конструктор передали { ... xxx => '' , ... } - истина?

=cut

sub validate_campaigns {
    my ( $campaigns, %options ) = @_;
    my $vresults = Direct::ValidationResult->new;

    # инициализируется в цикле если есть необходимость.
    # если ни в одной кампании не используются запрещенные площадки, то инициализации не происходит
    my $client_limits;

    for my $camp ( @$campaigns ) {
        my $vr = $vresults->next;

        if ( ! $camp->has_campaign_name ) {
            $vr->add( campaign_name => error_ReqField() );
        } elsif ( ( $camp->campaign_name || '' ) !~ /\S/ ) {
            $vr->add( campaign_name => error_EmptyField() );
        } else {
            my $camp_name = $camp->campaign_name;

            if ( $camp_name =~ m/[<>]/ ) {
                $vr->add( campaign_name => error_InvalidChars( iget('Поле #field# содержит спецсимволы') ) );
            }

            if ( length( $camp_name ) > $MAX_TITLE_LENGTH ) {
                $vr->add( campaign_name => error_MaxLength() );
            }
        }

        if ( ! $camp->has_client_fio) {
            $vr->add( client_fio => error_ReqField() );
        } elsif ( ! $camp->has_old                                                  # новая кампания
                || ! $camp->old->has_client_fio                                     # в старой имя не задано
                || ($camp->old->client_fio // '') ne ($camp->client_fio // '')) {   # пользователь меняет имя
            my $client_fio = $camp->client_fio;

            if ( $client_fio =~ m/[<>]/ || $client_fio =~ /(\P{print})/ ) {
                $vr->add( client_fio => error_InvalidChars( iget('Поле #field# содержит спецсимволы') ) );
            }

            if ( length( $client_fio ) > $MAX_CLIENT_FIO_LENGTH ) {
                $vr->add( client_fio => error_MaxLength() );
            }
        }

        my $now_ts = ts_round_day( time() );

        my $start_ts;
        if ( ! $camp->has_start_date ) {
            $vr->add( start_date => error_ReqField() );
        } elsif ( ( $camp->start_date || '' ) !~ /\S/ ) {
            $vr->add( start_date => error_EmptyField() );
        } else {
            my $start_date = $camp->start_date;

            my $error;
            try {
                $start_ts = ts_round_day( mysql2unix( $camp->start_date ) );
            } catch {
                $error = 1;
            };

            if ( $error || ! $start_ts ) {
                $vr->add( start_date => error_InvalidFormat_IncorrectDate() );
            } else {

                # Проверяем дату старта при создании и при обновлении
                # (в том случае, когда она изменилась (нужно для Коммандера))

                my $need_start_ts_validation = 1;
                if ( $camp->has_old && $camp->old->start_date ) {
                    # тут предполагаем что $camp->old->start_date всегда валиден
                    my $old_start_ts = ts_round_day( mysql2unix( $camp->old->start_date ) );
                    # не валидируем если при update start_date не изменился
                    if ( $start_ts == $old_start_ts ) {
                        $need_start_ts_validation = 0;
                    }
                }

                if ( $need_start_ts_validation && $start_ts < $now_ts ) {
                    $vr->add( start_date => error_InvalidField(
                        iget('Значение даты в поле #field# не может быть меньше текущей даты')
                    ) );
                }
            }
        }

        if ( ! $camp->has_email ) {
            $vr->add( email => error_ReqField() );
        } elsif ( ( $camp->email || '' ) !~ /\S/ ) {
            $vr->add( email => error_EmptyField() );
        } else {
            my $email = $camp->email;

            if ( is_valid_email( $email ) ) {
                if ( length( $email ) > $MAX_EMAIL_LENGTH ) {
                    $vr->add( email => error_MaxLength() );
                }
            } else {
                $vr->add( email => error_InvalidFormat() );
            }
        }

        if ( $camp->has_finish_date && defined $camp->finish_date ) {
            if ( ( $camp->finish_date || '' ) !~ /\S/ ) {
                $vr->add( finish_date => error_EmptyField() );
            } else {
                my $finish_ts;
                my $error;

                try {
                    $finish_ts = ts_round_day( mysql2unix( $camp->finish_date ) );
                } catch {
                    $error = 1;
                };

                if ( $error || ! $finish_ts ) {
                    $vr->add( finish_date => error_InvalidFormat_IncorrectDate() );
                } else {

                    if ( $finish_ts < $start_ts ) {
                        $vr->add( finish_date => error_InconsistentState( iget('Значение даты в поле #from# не может быть больше значения даты в поле #to#') ) );
                    }

                    # Проверяем, что дата окончания ещё не прошла, только в том случае, когда она изменилась
                    # Это нужно, чтобы пользователь имел возможность что-нибудь поменять в старой закончившейся
                    # кампании, не запуская её снова
                    if ( $camp->has_old
                         && (($camp->old->finish_date // '') ne ($camp->finish_date // ''))
                         && $finish_ts < $now_ts
                    ) {
                        $vr->add( finish_date => error_InconsistentState( iget('Значение даты в поле #field# не может быть меньше текущей даты') ) );
                    }
                }
            }
        }

        if ( $camp->has_disabled_domains && @{$camp->disabled_domains} ) {
            if ($camp->campaign_type eq 'cpm_yndx_frontpage' || $camp->campaign_type eq 'content_promotion') {
                $vr->add(disabled_domains => error_InvalidField(iget('Поле #field# не поддерживается')));
            } else {
                $client_limits = get_client_limits($camp->client_id) unless defined $client_limits;
                my $can_have_internal_dont_show_domains = Client::ClientFeatures::has_can_have_internal_dont_show_domains_feature($camp->client_id);
                my @camp_opts = $camp->has_opts ? @{$camp->{opts}} : ();
                my $vres = validate_disabled_domains(
                    $camp->disabled_domains,
                    disable_any_domains_allowed => Client::ClientFeatures::has_disable_any_domains_allowed_feature($camp->client_id),
                    disable_mail_ru_domain_allowed => Client::ClientFeatures::has_disable_mail_ru_domain_allowed_feature($camp->client_id),
                    disable_number_id_and_short_bundle_id_allowed => Client::ClientFeatures::has_disable_number_id_and_short_bundle_id_allowed_feature($camp->client_id),
                    blacklist_size_limit => $client_limits->{general_blacklist_size_limit},
                    show_internal_pages_warning => !$can_have_internal_dont_show_domains &&
                        none { $_ eq 'require_filtration_by_dont_show_domains' } @camp_opts,
                );

                # если есть только ворнинги - их нужно добавить в ответ (DIRECT-116036)
                if ( ! $vres->is_valid || $vres->has_only_warnings ) {
                    $vr->add( disabled_domains => $vres );
                }
            }
        }

        if ( $camp->has_disabled_ips ) {
            my $vres = validate_disabled_ips( $camp->disabled_ips );
            if ( ! $vres->is_valid ) {
                $vr->add( disabled_ips => $vres );
            }
        }

        if ( ! TimeTarget::get_timezone( $camp->timezone_id ) ) {
            $vr->add( timezone_id => error_InvalidField() );
        }

        if ( ! $camp->has_time_target ) {
            $vr->add( time_target => error_ReqField() );
        } else {
            my $unpacked_tt = TimeTarget::parse_timetarget( $camp->time_target );
            my $campaign_new_min_days_limit = Client::ClientFeatures::has_campaign_new_min_days_limit($camp->client_id);
            my @errors = TimeTarget::validate_timetarget( $unpacked_tt, 1, $camp->client_id, $campaign_new_min_days_limit ); # DIRECT-50144: extended tt always allowed
            if ( @errors ) {
                my $vres = Direct::ValidationResult->new;
                for my $err ( @errors ) {
                    $vres->add_generic( error_InvalidFormat( $err ) );
                }
                $vr->add( time_target => $vres );
            }
        }

        if ( $camp->has_geo ) {
            my $vres = validate_campaign_geo(
                            $camp, with_banners => 1,
                            $options{translocal_tree} ? ( translocal_tree => $options{translocal_tree} ) : (),
                        );

            if ( ! $vres->is_valid ) {
                $vr->add( geo => $vres );
            }
        }

        if ($camp->has_placement_types && @{$camp->placement_types}) {
            if ($camp->campaign_type ne 'dynamic') {
                $vr->add(placement_types => error_InvalidField(iget('Поле #field# не поддерживается')));
            }
        }

        if (is_internal_distrib_camp($camp->campaign_type)) {
            my $vres = validate_campaign_internal_distrib($camp);
            if ( ! $vres->is_valid ) {
                $vr->add( internal_distrib => $vres );
            }
        }

        if (is_internal_free_camp($camp->campaign_type)) {
            my $vres = validate_campaign_internal_free($camp);
            if ( ! $vres->is_valid ) {
                $vr->add( internal_free => $vres );
            }
        }

        if ( $camp->has_minus_words && $camp->minus_words ) {
            if ($camp->campaign_type eq 'cpm_yndx_frontpage') {
                $vr->add(minus_words => error_InvalidField(iget('Поле #field# не поддерживается')));
            } else {
                my $vres = validate_campaign_minus_words( $camp->minus_words, MinusWords::polish_minus_words_array($camp->minus_words) );
                if ( ! $vres->is_valid ) {
                    $vr->add( minus_words => $vres );
                }
            }
        }

        if ( $camp->has_competitors_domains ) {
            if ($camp->campaign_type eq 'cpm_yndx_frontpage' || $camp->campaign_type eq 'content_promotion') {
                $vr->add(competitors_domains => error_InvalidField(iget('Поле #field# не поддерживается')));
            } else {
                my $vres = Direct::ValidationResult->new;

                for my $domain ( @{ $camp->competitors_domains } ) {
                    if ( ! is_valid_domain( $domain ) ) {
                        $vres->add_generic( error_InvalidFormat(
                            iget('Элемент %s списка #field# - неправильный формат домена', $domain)
                        ) );
                    }
                }
                if ( ! $vres->is_valid ) {
                    $vr->add( competitors_domains => $vres );
                }
            }
        }

        # Рассчитываем тут, что если broad_match_flag eq 'No' и переданы broad_match_limit,
        # то Direct::Model::Campaign::Manager не обновит соответствующие поля в БД
        if ( $camp->has_broad_match_flag && $camp->broad_match_flag eq 'Yes' ) {
            if ( $camp->has_broad_match_limit ) {
                my $limit = $camp->broad_match_limit;
                if ($limit < $Campaign::BROAD_MATCH_LIMIT_MIN || $limit > $Campaign::BROAD_MATCH_LIMIT_MAX) {
                    $vr->add( broad_match_limit => error_InvalidField() );
                }
            }
        }

        if ( $camp->has_metrika_counters ) {
            my $metrika_counters = $camp->metrika_counters;

            my $vres = Direct::ValidationResult->new;

            foreach my $counter_id ( @$metrika_counters ) {
                if ( $counter_id <= 0 || $counter_id > $MAX_METRIKA_COUNTER_ID ) {
                    $vres->add_generic( error_InvalidField(
                        iget('Элемент %s списка #field# - неверное значение, корректное значение должно быть целым положительным', $counter_id)
                    ) );
                }
            }
            my $counter_limit = $MAX_METRIKA_COUNTERS_COUNT_WITH_FEATURE;
            if ( scalar( @$metrika_counters ) > $counter_limit ) {
                $vres->add_generic( error_ReachLimit(
                    iget('Размер списка #field# превышает максимально допустимый размер %d', $counter_limit)
                ) );
            }

            if ( ! $vres->is_valid ) {
                $vr->add( metrika_counters => $vres );
            }
        }

        # для апи перевалидируем ключевые цели, только если они менялись
        if (
            $camp->has_meaningful_goals && (
                !$options{skip_not_changed_meaningful_goals} ||
                $camp->is_meaningful_goals_changed()
            )
        ) {
            my $metrika_counters = $camp->has_metrika_counters && $camp->metrika_counters;

            my $cid = $camp->has_id ? $camp->id : undef;
            my $vres = validate_meaningful_goals($camp->meaningful_goals, $cid,
                camp_type                 => $camp->campaign_type,
                camp_counters             => $metrika_counters,
                currency                  => $camp->currency,
                prefetched_data           => $options{prefetched_meaningful_goals_data},
                check_availability_to_use_in_strategy  => $options{strategy_use_meaningful_goals},
                uid                       => $options{uid},
                clientId                  => $camp->client_id,
                unavailable_auto_goals_allowed         => $options{unavailable_auto_goals_allowed},
                is_allowed_to_use_value_from_metrika   => $options{is_allowed_to_use_value_from_metrika},
            );

            if ( ! $vres->is_valid ) {
                $vr->add( meaningful_goals => $vres );
            }
        }

        if ( $camp->has_device_target ) {
            if ( ! is_valid_device_targeting( $camp->device_target ) ) {
                $vr->add( device_target => error_InvalidField() );
            }
        }

        if ( $camp->has_hierarchical_multipliers ) {
            if (is_cpm_campaign($camp->campaign_type)) {
                $vr->add(hierarchical_multipliers => error_InvalidField(iget('Поле #field# не поддерживается'), field => 'hierarchical_multipliers'));
            } else {
                my $vres = validate_hierarchical_multipliers( $camp->campaign_type, $camp->client_id, $camp->hierarchical_multipliers );
                if ( ! $vres->is_valid ) {
                    $vr->add( hierarchical_multipliers => $vres );
                }
            }
        }

        if ( $camp->has_sms_time_from_hours && $camp->has_sms_time_from_minutes ) {
            if ( ! _is_valid_sms_time( $camp->sms_time_from_hours, $camp->sms_time_from_minutes ) ) {
                $vr->add( sms_time => error_InvalidFormat_IncorrectTime( iget('Значение времени в поле #from# указано в неправильном формате') ) );
            } else {
                if ( $camp->sms_time_from_minutes % $SMS_TIME_MULTIPLICITY != 0 ) {
                    $vr->add( sms_time => error_InvalidField( iget('Значение времени в поле #from# должно быть кратно %s', $SMS_TIME_MULTIPLICITY) ) );
                }
            }
        }

        if ( $camp->has_sms_time_to_hours && $camp->has_sms_time_to_minutes ) {
            if ( ! _is_valid_sms_time( $camp->sms_time_to_hours, $camp->sms_time_to_minutes ) ) {
                $vr->add( sms_time => error_InvalidFormat_IncorrectTime( iget('Значение времени в поле #to# указано в неправильном формате') ) );
            } else {
                if ( $camp->sms_time_to_minutes % $SMS_TIME_MULTIPLICITY != 0 ) {
                    $vr->add( sms_time => error_InvalidField( iget('Значение времени в поле #to# должно быть кратно %s', $SMS_TIME_MULTIPLICITY) ) );
                }
            }
        }

        if ( ! $camp->has_money_warning_threshold ) {
            $vr->add( money_warning_threshold => error_ReqField() );
        } else {
            my $threshold = $camp->money_warning_threshold;
            if ( $threshold < $MONEY_THRESHOLD_MIN || $threshold > $MONEY_THRESHOLD_MAX ) {
                $vr->add( money_warning_threshold => error_InvalidField(
                    iget('Значение в поле #field# должно быть от %s до %s %%', $MONEY_THRESHOLD_MIN, $MONEY_THRESHOLD_MAX )
                ) );
            }
        }

        if ( ! $camp->has_position_check_interval ) {
            $vr->add( position_check_interval => error_ReqField() );
        } else {
            my $available_intervals    = join('|' => @POSITION_CHECK_INTERVAL);
            my $available_intervals_re = qr/($available_intervals)/;
            if ( $camp->position_check_interval !~ /^$available_intervals_re$/ ) {
                $vr->add( position_check_interval => error_InvalidField(
                    iget('Значение в поле #field# должно совпадать с одним из значений %s', join(', ' => @POSITION_CHECK_INTERVAL))
                ) );
            }
        }

        # NB: проверяем broad_match_goal_id только если новое значение не совпадает с предыдущим, т.е.:
        #       - создается новая кампания
        #       - обновляется существующая, при этом указано новое значение broad_match_goal_id
        if ( $camp->has_broad_match_goal_id
                && (   ! $camp->has_old
                    || ! $camp->old->broad_match_goal_id
                    || $camp->old->broad_match_goal_id != $camp->broad_match_goal_id)
        ) {
            if (my $error = validate_broad_match_goal_id($camp->broad_match_goal_id, $camp->has_id ? $camp->id : undef)) {
                $vr->add(broad_match_goal_id => $error);
            }
        }

        if ( $camp->has_broad_match_flag && $camp->has_platform ) {
            my $error = validate_broad_match_flag( $camp->broad_match_flag, $camp->_detect_is_search_stop );
            if ( $error ) {
                $vr->add( broad_match_flag => $error );
            }
        }

        for my $field (qw/broad_match_flag broad_match_limit broad_match_goal_id/){
            if ($camp->campaign_type eq 'cpm_yndx_frontpage' && $camp->{"has_$field"}->()) {
                $vr->add($field => error_InvalidField(iget('Поле #field# не поддерживается')));
            }
        }
        if ($camp->campaign_type eq 'cpm_yndx_frontpage') {
            if ( ! $camp->has_allowed_frontpage_types ) {
                $vr->add( allowed_frontpage_types => error_ReqField() );
            }
            for my $frontpage_type (@{$camp->allowed_frontpage_types}) {
                if (none { $_ eq  $frontpage_type} @PAGE_TYPE_ENUM) {
                    $vr->add( allowed_frontpage_types => error_InvalidChars( iget('Поле #field# содержит неверное значение перечисления') ) );
                }
            }
        }

        if (exists $camp->{attribution_model}) {
            my $cross_device_attribution_types_feature_disabled = !Client::ClientFeatures::cross_device_attribution_types_enabled($camp->client_id);
            my $is_cross_device_type = ($camp->{attribution_model} eq 'first_click_cross_device')
                || ($camp->{attribution_model} eq 'last_significant_click_cross_device')
                || ($camp->{attribution_model} eq 'last_yandex_direct_click_cross_device');
            if ($cross_device_attribution_types_feature_disabled && $is_cross_device_type) {
                $vr->add( attribution_type => error_InvalidField(iget('Указана некорректная модель атрибуции')));
            }
        }

        if ($camp->has_strategy_id && defined $camp->strategy_id && $camp->strategy_id) {
            if (my $error = validate_strategy_id($camp->strategy_id, $camp->client_id,
                $options{package_strategy_is_changed}, $options{wallet_cid}, $camp->id, $camp->campaign_type)) {
                $vr->add(strategy_id => $error);
            }
        }
    }

    return $vresults;
}

=head2 validate_add_campaign

    Проверка возможности создания новой кампании с указанными параметрами

=cut

sub validate_add_campaign {
    my ( $campaign, %options ) = @_;
    my $vres = validate_campaigns( [ $campaign ], %options );
    return $vres->get_objects_results->[0];
}

=head2 validate_update_campaign

    Проверка возможности обновления кампании с указанными параметрами

=cut

sub validate_update_campaign {
    my ( $campaign, %options ) = @_;
    my $vres = validate_campaigns( [ $campaign ], %options );
    return $vres->get_objects_results->[0];
}

=head2 validate_suspend_campaigns($campaigns)

    Проверка возможности остановки кампаний

    Параметры:
        $campaigns - ссылка на массив кампаний [Direct::Model::Сampaign, ...]

    Результат:
        Результат проверки (ValidationResult)
        Ошибки сохраняются по кампании (generic)

=cut

sub validate_suspend_campaigns {
    my ($campaigns) = @_;

    my $vr_main = Direct::ValidationResult->new();

    for my $campaign (@$campaigns) {
        my $vr = $vr_main->next;

        if ($campaign->status_empty eq 'Yes') {
            $vr->add_generic(error_NotFound_Campaign());
            next;
        }

        if ($campaign->status_show eq 'No') {
            $vr->add_generic(warning_AlreadySuspended_Campaign());
        }
    }

    return $vr_main;
}

=head2 validate_resume_campaigns($campaigns)

    Проверка возможности запуска кампаний

    Параметры:
        $campaigns - ссылка на массив кампаний [Direct::Model::Сampaign, ...]

    Результат:
        Результат проверки (ValidationResult)
        Ошибки сохраняются по кампании (generic)

=cut

sub validate_resume_campaigns {
    my ($campaigns, $is_cpm_banner_campaign_disabled) = @_;

    my $vr_main = Direct::ValidationResult->new();

    for my $campaign (@$campaigns) {
        my $vr = $vr_main->next;

        if ($campaign->status_empty eq 'Yes') {
            $vr->add_generic(error_NotFound_Campaign());
            next;
        }

        if ($campaign->status_archived eq 'Yes') {
            $vr->add_generic(error_BadStatus(iget('Необходимо сначала разархивировать кампанию')));
            next;
        }

        if (Campaign::is_cpm_campaign($campaign->campaign_type) && $is_cpm_banner_campaign_disabled) {
            $vr->add_generic(error_BadStatus('Нельзя запустить медийную кампанию'));
            next;
        }

        if ($campaign->status_show eq 'Yes') {
            $vr->add_generic(warning_NotSuspended_Campaign());
        }

    }

    return $vr_main;
}

=head2 validate_delete_campaigns($campaigns)

    Проверка возможности удаления кампаний

    Параметры:
        $campaigns - ссылка на массив кампаний [Direct::Model::Сampaign, ...]

    Результат:
        Результат проверки (ValidationResult)s
        Ошибки сохраняются по кампании (generic)

=cut

sub validate_delete_campaigns {
    my ($campaigns) = @_;

    my $vr_main = Direct::ValidationResult->new();

    for my $campaign (@$campaigns) {
        my $vr = $vr_main->next;

        if ($campaign->status_empty eq 'Yes') {
            $vr->add_generic(error_NotFound_Campaign());
            next;
        }

        if ( $campaign->is_in_bs_queue
            || ($campaign->converted eq 'Yes' && $campaign->currency eq 'YND_FIXED')
        ) {
            $vr->add_generic(error_CantDelete(iget('Невозможно удалить кампанию')));
            next;
        }

        if ( $campaign->sum != 0 || $campaign->sum_to_pay != 0
            || $campaign->sum_last != 0 || $campaign->bs_order_id != 0
        ) {
            $vr->add_generic(error_CantDelete(iget('Запрещено удалять кампанию с деньгами, либо на которую выставлен счет на оплату')));
            next;
        }
    }

    return $vr_main;
}


=head2 validate_meaningful_goals

Проверка выбранных ключевых целей

    my $vr = validate_meaningful_goals($meaningful_goals, $cid);

Опции:

    camp_type, camp_counters, prefetched_data - пробрасываются в get_available_meaningful_goals
    currency
    check_availability_to_use_in_strategy - проверить что ключевые цели могут использоваться в стратегии.
                                            Для этого список ключевых целей должен содержать не только цели по-умолчанию

=cut

sub validate_meaningful_goals {
    my ($meaningful_goals, $cid, %opt) = @_;

    my $vr = Direct::ValidationResult->new();

    my $default_goal = Campaign::get_default_meaningful_goal(cid => $cid, %opt);
    my $available_goals;


    if (defined $opt{check_availability_to_use_in_strategy} && $opt{check_availability_to_use_in_strategy} &&
        (!@$meaningful_goals || !any {$_->{goal_id} != $default_goal->{goal_id} } @$meaningful_goals))
    {
        $vr->add_generic(error_BadParams(iget('Удаление ключевых целей не допускается, поскольку в настройках стратегии выбрана оптимизация по ключевым целям')));
    }
    return $vr  if !@$meaningful_goals;

    my $currency = $opt{currency};
    croak 'Currency required'  if !$currency;

    my %seen_goals;
    for my $goal (@$meaningful_goals) {
        my $goal_id = $goal->{goal_id} || 0;

        # skip default goal id check to avoid unnecessary metrika call
        if ($goal_id != $default_goal->{goal_id}) {
            $available_goals ||= Campaign::get_available_meaningful_goals($cid, %opt);
            if (!$available_goals->{$goal_id}) {
                $vr->add_generic(error_NotFound(iget('Цель %s из поля #field# не найдена', $goal_id)));
            }
        }

        if ($seen_goals{$goal_id} && $seen_goals{$goal_id} == 1) {
            $vr->add_generic(error_DuplicateField(iget('Цель %s в поле #field# указана более одного раза', $goal_id)));
        }
        $seen_goals{$goal_id}++;

        my $goal_value = $goal->{value};
        if (!is_valid_float($goal_value)) {
            $vr->add_generic(error_InvalidFormat(iget("Неверно указана ценность цели %s", $goal_id)));
        }
        else {
            if ($goal_value < get_currency_constant($currency, 'MIN_PRICE')) {
                $vr->add_generic(error_InvalidField(iget("Для цели %s указана ценность меньше минимальной %s",
                        $goal_id, format_const($currency, 'MIN_PRICE'))
                    ));
            }
            if ($goal_value > get_currency_constant($currency, 'MAX_AUTOBUDGET')) {
                $vr->add_generic(error_InvalidField(iget("Для цели %s указана ценность больше максимальной %s",
                        $goal_id, format_const($currency, 'MAX_AUTOBUDGET'))
                    ));
            }
        }

        my $is_metrika_source_of_value = $goal->{is_metrika_source_of_value};
        if (defined $is_metrika_source_of_value && $is_metrika_source_of_value &&
            defined $opt{is_allowed_to_use_value_from_metrika} && !$opt{is_allowed_to_use_value_from_metrika}) {
            $vr->add_generic(error_InvalidField(iget("Для цели %s указан недоступный источник дохода в рамках указанной стратегии",
                $goal_id)
            ));
        }
    }

    return $vr;
}


=head2 validate_broad_match_goal_id

Проверка цели ДРФ

    my $defect = validate_broad_match_goal_id($goal_id, $cid);

=cut

sub validate_broad_match_goal_id {
    my ($goal_id, $cid) = @_;

    return if !$goal_id;
    return error_InvalidField_NotPositive()  if $goal_id =~ /\D/;

    if ($cid) {
        my $goals = Stat::Tools::orders_goals(cid => $cid);
        my $goal = first {$_->{goal_id} == $goal_id} @$goals;
        return  if $goal && $goal->{status} ne 'Deleted';
    }

    return error_NotFound(iget('Указанная в поле #field# цель не найдена'));
}

=head2 validate_broad_match_flag

    Проверка флага показов по дополнительным релевантным фразам

=cut

sub validate_broad_match_flag {
    my ($broad_match_flag, $is_search_stop) = @_;

    # NB: $broad_match_flag может иметь значение 1/0, Yes/No, 1/undef -
    #     считаем тут что ДРФ включен когда флаг определён и не No
    my $broad_match_enabled = $broad_match_flag && $broad_match_flag ne 'No';

    if ($is_search_stop && $broad_match_enabled) {
        return error_DoesNotMatchStrategy_BroadMatch();
    }

    return;
}

=head2 validate_ab_segments

Валидация аб сегментов
 - проверка на дубликаты.
 - проверка на кол-во сегментов

Опции:


=cut

sub validate_ab_segments {
    my ($ab_sections_statistic, $ab_segments_retargeting, $metrika_segments, $camp_counters) = @_;
    my $vr_main = Direct::ValidationResult->new();

    my %all_segments_by_section = map {$_ => 0} @$ab_sections_statistic;
    my %camp_counters_by_id = map {$_ => 1} @{get_num_array_by_str($camp_counters)};
    my %segment_ids;
    for (@$ab_segments_retargeting) {
        if ($segment_ids{$_}++) {
            $vr_main->add_generic(error_Duplicated(iget("Ид сегмента должно быть уникальным")));
        }
    }

    my %ab_metrika_segments_by_id = map {$_->{segment_id} => $_} @$metrika_segments;
    my %ab_metrika_segments_by_section_id = map {$_->{section_id} => $_} @$metrika_segments;

    for my $section_id (keys %all_segments_by_section) {
        if (!$ab_metrika_segments_by_section_id{$section_id}) {
            $vr_main->add_generic(error_NotFound(iget("Эксперимент №%d в архиве", $section_id)));
        } else {
            #выбираем любой сегмент из эксперимента counter_ids у сегментов одинаковы
            my $segment = $ab_metrika_segments_by_section_id{$section_id};
            if (!any {$camp_counters_by_id{$_}} @{$segment->{counter_ids}}) {
                $vr_main->add_generic(error_NotFound(iget("Эксперимент №%d не соответсвует счетчикам", $section_id)));
            }
        }
    }

    for my $segment_id (@$ab_segments_retargeting) {
        my $vr = Direct::ValidationResult->new();
        my $segment = $ab_metrika_segments_by_id{$segment_id};
        if (!defined $segment) {
            $vr->add_generic(error_NotFound(iget("Сегмент №%d в архиве", $segment_id)));
            $vr_main->add($segment_id, $vr);
            next;
        }
        if (!defined $all_segments_by_section{$segment->{section_id}}) {
            $vr->add_generic(error_InconsistentState(iget("Сегмент должен быть включен в эксперимент")));
            $vr_main->add($segment_id, $vr);
            next;
        }
        $all_segments_by_section{$segment->{section_id}}++;
    }

    if (scalar keys %all_segments_by_section > $MAX_COUNT_CAMP_AB_SECTIONS) {
        $vr_main->add_generic(error_LimitExceeded(iget("Количество экспериментов суммарно должно быть не больше %s", $MAX_COUNT_CAMP_AB_SECTIONS)));
    }

    if (scalar values %all_segments_by_section > $MAX_COUNT_CAMP_AB_SEGMENTS_IN_SECTION) {
        $vr_main->add_generic(error_LimitExceeded(iget("Количество сегментов в эксперименте должно быть не больше %s", $MAX_COUNT_CAMP_AB_SEGMENTS_IN_SECTION)));
    }

    return $vr_main;
}

=head2 validate_copy_campaigns_for_client

    Проверка возможности копирования кампаний клиентом

    Параметры:
        $campaigns - ссылка на массив кампаний [Direct::Model::Сampaign, ...]
        $uid - ид клиента

    Результат:
        Результат проверки (ValidationResult)

=cut

sub validate_copy_campaigns_for_client {
    my ($campaigns, $client_id, $perminfo) = @_;

    my $allowed_types = get_camp_kind_types('copyable_by_client');

    if ($perminfo->{role} ne $ROLE_SUPER && !Client::ClientFeatures::has_cpm_deals_allowed_feature($client_id)) {
        $allowed_types = xminus $allowed_types, [qw/cpm_deals/];
    }
    if ($perminfo->{role} ne $ROLE_SUPER && !Client::ClientFeatures::has_content_promotion_video_allowed_feature($client_id) &&
            !Client::ClientFeatures::is_feature_content_promotion_collection_enabled($client_id)) {
        $allowed_types = xminus $allowed_types, [qw/content_promotion/];
    }

    my $cids_with_rejected_creatives = CampaignTools::campaigns_with_rejected_creatives([map { $_->id } @$campaigns]);

    my $vr_main = Direct::ValidationResult->new();

    for my $campaign (@$campaigns) {
        my $vr = Direct::ValidationResult->new();
        if ($campaign->client_id != $client_id) {
            $vr->add_generic(error_AccessDenied(iget('Кампания %d не принадлежит пользователю', $campaign->id)));
        }

        if (!any { $campaign->campaign_type eq $_ } @$allowed_types) {
            $vr->add_generic(error_InvalidCampaignType(iget('Тип кампании %s не соответствует интерфейсу', $campaign->campaign_type)));
        }

        if ($campaign->status_archived eq 'Yes') {
            $vr->add_generic(error_CampaignArchived(iget('Кампания %d была перенесена в архив', $campaign->id)));
        }
        my $completed_groups = Models::AdGroup::is_completed_groups(PrimitivesIds::get_pids(cid => $campaign->id));
        if (!%$completed_groups) {
            $vr->add_generic(error_CampaignHasntCompletedGroups('Кампания содержит только пустые группы - копирование невозможно.'));
        }

        if (!$campaign->has_banners) {
            $vr->add_generic(error_NoBanners(iget('В кампании %d нет баннеров для копирования', $campaign->id)));
        }

        if ($campaign->is_in_camp_operations_queue) {
            $vr->add_generic(error_AlreadyInCampQueue(iget(
                        'Над кампанией %d уже совершаются какие-то фоновые операции. Пожалуйста, дождитесь их завершения.',
                        $campaign->id)));
        }
        if ($campaign->is_in_camp_operations_queue_copy) {
            $vr->add_generic(error_AlreadyInCopyQueue(iget(
                        'Кампания %d уже находятся в очереди для копирования. Пожалуйста, дождитесь завершения операции.',
                        $campaign->id)));
        }
        if ($cids_with_rejected_creatives->{$campaign->id}) {
            $vr->add_generic(error_InconsistentState(iget(
                'В кампании %d есть объявления с некорректными креативами',
                $campaign->id
            )));
        }

        $vr_main->add($campaign->id, $vr);
    }

    my $camp_limit_error = check_add_client_campaigns_limits(
        ClientID => $client_id,
        unarchived_count => scalar @$campaigns,
    );
    if ($camp_limit_error) {
        $vr_main->add_generic(error_LimitExceeded(iget(
                'Кампании не могут быть скопированы. Превышено допустимое количество кампаний')));
    }

    return $vr_main;
}

=head2 validate_cpm_frequency

    Валидируем частоту показов и период

    my $defect = validate_cpm_frequency($frequency, $period);

=cut

sub validate_cpm_frequency {
    my ($frequency, $period) = @_;

    my $vr = Direct::ValidationResult->new();

    if (
        $frequency
        && (   $frequency < $Direct::Validation::Campaigns::Constraints::CPM_RF_MIN
            || $frequency > $Direct::Validation::Campaigns::Constraints::CPM_RF_MAX)
      ) {
        $vr->add(
            frequency => error_InvalidField(
                iget(
                    'Значение в поле #field# должно быть от %s до %s',
                    $Direct::Validation::Campaigns::Constraints::CPM_RF_MIN,
                    $Direct::Validation::Campaigns::Constraints::CPM_RF_MAX
                )
            )
        );
    }


    if (
        $period
        && (   $period < $Direct::Validation::Campaigns::Constraints::CPM_RF_RESET_MIN
            || $period > $Direct::Validation::Campaigns::Constraints::CPM_RF_RESET_MAX)
      ) {
        $vr->add(
            period => error_InvalidField(
                iget(
                    'Значение в поле #field# должно быть от %s до %s',
                    $Direct::Validation::Campaigns::Constraints::CPM_RF_RESET_MIN,
                    $Direct::Validation::Campaigns::Constraints::CPM_RF_RESET_MAX
                )
            )
        );
    }

    return $vr;
}

=head2

    Helpers

=cut

sub _is_valid_sms_time {
    my ( $hours, $minutes ) = @_;
    return if not ( defined( $hours )   && $hours   =~ /^\d+$/ );
    return if not ( defined( $minutes ) && $minutes =~ /^\d+$/ );
    return ( $hours >= 0 && $hours <= 23 && $minutes >= 0 && $minutes <= 59 ) || ( $hours == 24 && $minutes == 0 );
}

=head2 validate_strategy_id

Проверка ID стратегии

=cut

sub validate_strategy_id {
    my ($strategy_id, $client_id, $package_strategy_is_changed, $wallet_cid, $cid, $camp_type) = @_;

    my $where = {ClientID => $client_id};

    if ($package_strategy_is_changed && !$wallet_cid){
        return error_InconsistentState(iget('Стратегию без кошелька запрещено делать публичной'));
    }

    my $strategies = get_strategies($client_id, $where);
    my $strategy = $strategies->{$strategy_id};

    if ($package_strategy_is_changed){
        my $types = get_campaigns_type_by_strategy_id($strategy_id, $cid);
        my $campaigns_count = @$types;
        if (uniq(@$types, $camp_type) > 1){
            return error_InconsistentState(iget('Запрещено добавлять в пакет кампании разных типов'));
        }

        if ($campaigns_count >= $MAX_NUMBER_OF_CAMPAIGNS_IN_STRATEGY) {
            return error_InconsistentState(iget('Превышено допустимое количество кампаний в стратегии'));
        }

        if ($strategy->{is_public} eq 'No'){
            my $strategies_count = grep {$_->{is_public} eq 'Yes'} values %$strategies;

            if ($strategies_count >= $MAX_STRATEGIES_COUNT){
                return error_LimitExceeded(iget(
                    'Превышено допустимое количество стратегий'))
            }

            my $unarc_strategies_count = grep {$_->{archived} eq 'No' && $_->{is_public} eq 'Yes'} values %$strategies;

            if ($unarc_strategies_count >= $MAX_UNARC_STRATEGIES_COUNT){
                return error_LimitExceeded(iget(
                    'Превышено допустимое количество активных стратегий'))
            }
        }
    }

    if ($package_strategy_is_changed && $strategy->{type} eq 'default'){
        return error_InconsistentState(iget('Ручную стратегию запрещено делать публичной'));
    }

    if ($strategy && $strategy->{archived} eq 'No'){
        return;
    }

    return error_NotFound_Strategy();
}

1;
