package Direct::Validation::Client;
use strict;
use warnings;

use Common qw//;
use Client;
use Direct::Validation::Errors;
use Direct::ValidationResult;
use Yandex::I18n;

use Scalar::Util qw/blessed/;
use List::MoreUtils qw/any uniq/;
use Try::Tiny qw/try catch/;
use utf8;


=head2 check_country_and_currency

    Проверяет dataset клиента на корректность комбинации страна/валюта/клиент/агенство.
    Параметры позиционные:
        $client_data - hashref, dataset клиента
        $agency_id - клиентский идентификатор агенства, которому принадлежит клиент
    Параметры именованные:
        is_direct - 1/0, делается ли проверка для Директа (ограничения на фишки) или для Баяна/Геоконтекста (фишки можно)

        В $client_data для проверки используются следующие поля:
            ClientID - идентификатор клиента,
            initial_country || country - код страны из геобазы,
            initial_currency || currency - код валюты.

    Возвращает экземпляр Direct::ValidationResult.
=cut

sub check_country_and_currency {
    my ($client_data, $agency_id, %O) = @_;

    my $vr = Direct::ValidationResult->new();

    my $client_id = $client_data->{ClientID};

    try {
        my $country = $client_data->{initial_country} || $client_data->{country};
        my $currency = $client_data->{initial_currency} || $client_data->{currency};

        if ($client_id) {
            _throw($vr, currency => error_InvalidField(iget('Клиент уже использует другую валюту')))
                unless _check_balance_currency($client_id, $agency_id, $currency);

            _throw($vr, currency => error_InvalidField(iget('Клиент уже использует другую рабочую валюту')))
                if (Client::is_client_exists($client_id) && !_check_work_currency($client_id, $currency));
        }

        my $allowed = _get_allowed_countries_and_currencies($client_id, $agency_id, $client_data->{uid}, is_direct => $O{is_direct});

        _throw($vr, country  => error_InvalidField(iget('Недопустимая страна'))) unless _check_country($country, $allowed);
        _throw($vr, currency => error_InvalidField(iget('Недопустимая валюта'))) unless _check_currency($currency, $country, $allowed);
    }
    catch {
        my $error = shift;
        #_throw делает die '', если поймали непустую ошибку - она внешняя, отправляем дальше
        die $error unless __PACKAGE__ eq blessed $error;
    };

    return $vr;
}

sub _check_balance_currency {
    my ($client_id, $agency_id, $currency) = @_;

    my $existing = Common::get_country_and_currency_from_balance($client_id, $agency_id);
    my $currencies = $existing->{currencies};
    if ( $currency eq 'YND_FIXED' ){
        #Если проверяемая валюта YND_FIXED - в выдаче баланса ее нет.
        #В этом случае интерпретируем наличие множества валют в выдаче как отсутствие привязки (см. DIRECT-54708)
        return 1 unless $existing->{currency};
    }

    return @$currencies ? any {$_ eq $currency} @$currencies : 1;
}

sub _check_work_currency {
    my ($client_id, $currency) = @_;

    my $work_currency = Client::get_client_currencies($client_id)->{work_currency};

    return 1 unless defined $work_currency;
    return $work_currency eq $currency;
}

sub _check_country {
    my ($country, $allowed) = @_;
    return defined $country ? any { $_->{region_id} == $country } @{$allowed->{countries}} : 1;
}

sub _check_currency {
    my ($currency, $country, $allowed) = @_;

    my $allowed_currencies;
    my $countries_currencies = $allowed->{countries_currencies};

    if ($country) {
        $allowed_currencies = $countries_currencies->{$country} || $countries_currencies->{other};
    }
    else {
        $allowed_currencies = [ uniq map {@$_} values %{$countries_currencies}];
    }
    return any { $_ eq $currency } @$allowed_currencies;
}

sub _get_allowed_countries_and_currencies {
    my ($client_id, $agency_id, $uid, %O) =  @_;

    my $firm_country_currency_data = Common::get_cached_balance_country_currency($client_id, $agency_id);

    my $allowed_country_currency_data = Common::get_client_allowed_country_currency(
        ($agency_id ?
            ( is_for_agency => 1,
              agency_client_id => $agency_id,
            ) : ( is_for_agency => 0 )
        ),
        firm_country_currency_data => $firm_country_currency_data,
        is_direct => $O{is_direct},
        uid => $agency_id ? undef : $uid,
    );

    return $allowed_country_currency_data;
}

sub _throw {
    my ($vr, $field, $error ) = @_;
    $vr->add($field => $error);
    return die bless {}, __PACKAGE__;
}

1;
