package Direct::Validation::Creative;

use Direct::Modern;

use base qw(Exporter);

our @EXPORT = qw//;

our @EXPORT_OK = qw/
    validate_banner_creative
/;

use Direct::Validation::Errors;
use Direct::Model::VideoAddition;
use Direct::Model::CanvasCreative;
use Settings;

use Yandex::I18n qw/iget/;
use List::MoreUtils qw/none/;


our $TGO_VIDEO_DURATION //= 15;
our $MOBILE_CONTENT_VIDEO_MIN_DURATION //= 6;
our $MOBILE_CONTENT_VIDEO_MAX_DURATION //= 60;

=head2 validate_banner_creative

Проверка соответствия креатива баннеру

Возвращает объект ошибки или undef, если ошибки нет

=cut

sub validate_banner_creative {
    my ($adgroup, $banner) = @_;

    return unless $banner->banner_type =~ /(text|cpc_video|cpm_banner)/;

    return undef unless $banner->has_creative;

    return undef unless $banner->creative->has_creative;

    my $creative = $banner->creative->creative;

    my $layout_id = $creative->layout_id;
    return undef unless defined $layout_id;

    if ($banner->banner_type =~ /(text|cpc_video)/) {
        return validate_video_creative($banner, $creative, $layout_id);
    }
    if ($banner->banner_type eq 'cpm_banner') {
        return validate_cpm_banner_creative($adgroup, $layout_id);
    }
}

=head2 validate_cpm_banner_creative

Проверка соответствия креатива баннеру для cpm баннеров

Возвращает объект ошибки или undef, если ошибки нет

=cut
sub validate_cpm_banner_creative {
    my ($adgroup, $layout_id) = @_;

    my $range = $Direct::Model::CanvasCreative::COMPATIBLE_LAYOUT_IDS{$adgroup->adgroup_type};
        
    if (defined $range && none {$layout_id == $_} @$range) {
        return error_InvalidField(iget('Указан несовместимый шаблон креатива'));
    }

    return undef;
}

=head2 validate_video_creative

Проверка соответствия видео креатива баннеру

Возвращает объект ошибки или undef, если ошибки нет

=cut
sub validate_video_creative {
    my ($banner, $creative, $layout_id) = @_;

    my $range = $Direct::Model::VideoAddition::COMPATIBLE_LAYOUT_IDS{$banner->banner_type};
    return undef unless defined $range;

    if (none {$layout_id == $_} @$range) {
        if ($banner->banner_type eq 'cpc_video') {
            return error_InvalidField(iget('Выбранный креатив не подходит к типу объявления Видео'));
        } else {
            return error_InvalidField(iget('Указан несовместимый шаблон видео-дополнения'));
        }
    }

    if (UNIVERSAL::isa($banner->{adgroup},"Direct::Model::AdGroupMobileContent")) {
        if ($banner->banner_type =~ /(mobile_content|cpc_video)/ && $creative->creative_type eq 'video_addition'
            && ($creative->duration < $MOBILE_CONTENT_VIDEO_MIN_DURATION || $creative->duration > $MOBILE_CONTENT_VIDEO_MAX_DURATION)) {
            if ($banner->banner_type eq 'cpc_video') {
                return error_BadUsage(iget('Длина видеообъявления должна быть от %s до %s секунд', $MOBILE_CONTENT_VIDEO_MIN_DURATION, $MOBILE_CONTENT_VIDEO_MAX_DURATION));
            } else {
                return error_BadUsage(iget('Длина видеодополнения текстово-графических объявлений должна быть от %s до %s секунд',
                    $MOBILE_CONTENT_VIDEO_MIN_DURATION, $MOBILE_CONTENT_VIDEO_MAX_DURATION));
            }
        }
    } else {
        if ($banner->banner_type =~ /(text|cpc_video)/ && $creative->creative_type eq 'video_addition' && $creative->duration != $TGO_VIDEO_DURATION) {
            if ($banner->banner_type eq 'cpc_video') {
                return error_BadUsage(iget('Длина видеообъявления должна быть %s секунд', $TGO_VIDEO_DURATION));
            } else {
                return error_BadUsage(iget('Длина видеодополнения текстово-графических объявлений должна быть %s секунд', $TGO_VIDEO_DURATION));
            }
        }
    }

    return undef;
}

1;


