package Direct::Validation::DayBudget;

use Direct::Modern;

use List::MoreUtils qw/any none/;
use JSON;

use Settings;

use Yandex::HashUtils;
use Yandex::I18n;
use Yandex::Validate;

use Campaign;
use Currencies qw/get_currency_constant/;
use Currency::Format qw/format_const/;
use Direct::ValidationResult;
use Direct::Validation::Errors;
use Tools;

use base qw(Exporter);
our @EXPORT = qw//;
our @EXPORT_OK = qw/
    validate_camp_day_budget
    validate_wallet_day_budget
    validate_struct_day_budget
/;

=head2 $MAX_DAY_BUDGET_DAILY_CHANGE_COUNT

    Максимальное количество изменений дневного бюджета для кампании в течении дня.
    После указанного количества изменений суммы за сутки, до наступления следующих суток
    дневной бюджет можно только отключитью, но не изменить.

=cut

our $MAX_DAY_BUDGET_DAILY_CHANGE_COUNT = 3;


=head2 validate_camp_day_budget

    strategy => 'default', # название стратегии, соответствущее возвращаемому значению функции Primitives::detect_strategy
    new_day_budget_data => { # данные об устанавливаемом дневном бюджете
        day_budget => 123, # сумма дневного бюджета
        day_budget_show_mode => 'stretched', # режим показа объявлений при установленном дневном бюджете ('default', 'stretched')
    },
    old_day_budget_data => { # текущие данные о дневном бюджете на изменяемой кампании (только при !new_camp)
        day_budget => 123, # сумма дневного бюджета
        day_budget_daily_change_count => 2, # количество изменений дневного бюджета за сегодня (без учёта валидируемого изменения) 
    }
    currency => 'YND_FIXED', # валюта кампании, для которой устанавливается дневной бюджет; обязательный параметр!
    new_camp => 1|0, # новая ли кампания? для новой кампании не проверяется количество изменений дневного бюджета
    wallet_day_budget => { # дневной бюджет общего счета кампании - для выдачи warning, если настройки на кампанию будут перекрыты настройками на общем счету
                           # если у кампании нет общего счета, нужно передать тут undef
        day_budget => 100,
        day_budget_show_mode => 'default',
    }

=cut
sub validate_camp_day_budget {
    my (%O) = @_;

    return _validate_day_budget(%{hash_cut(\%O, qw/strategy
                                                    new_day_budget_data
                                                    old_day_budget_data
                                                    currency
                                                    new_camp
                                                    wallet_day_budget
                                                    is_autobudget
                                                /)});
}

=head2 validate_wallet_day_budget

    new_day_budget_data => { # данные об устанавливаемом дневном бюджете
        day_budget => 123, # сумма дневного бюджета
        day_budget_show_mode => 'stretched', # режим показа объявлений при установленном дневном бюджете ('default', 'stretched')
    },
    old_day_budget_data => { # текущие данные о дневном бюджете на изменяемой кампании (только при !new_camp)
        day_budget => 123, # сумма дневного бюджета
        day_budget_daily_change_count => 2, # количество изменений дневного бюджета за сегодня (без учёта валидируемого изменения) 
    }
    currency => 'YND_FIXED', # валюта кампании, для которой устанавливается дневной бюджет; обязательный параметр!

=cut
sub validate_wallet_day_budget {
    my (%O) = @_;

    return _validate_day_budget(%{hash_cut(\%O, qw/new_day_budget_data old_day_budget_data currency/)}, is_wallet => 1);
}

=head2 _validate_day_budget

    Проверяет корректность параметров дневного ограничения бюджета.
    Принимает именованные параметры:
        strategy => 'default', # название стратегии, соответствущее возвращаемому значению функции Primitives::detect_strategy
        new_day_budget_data => { # данные об устанавливаемом дневном бюджете
            day_budget => 123, # сумма дневного бюджета
            day_budget_show_mode => 'stretched', # режим показа объявлений при установленном дневном бюджете ('default', 'stretched')
        },
        old_day_budget_data => { # текущие данные о дневном бюджете на изменяемой кампании (только при !new_camp)
            day_budget => 123, # сумма дневного бюджета
            day_budget_daily_change_count => 2, # количество изменений дневного бюджета за сегодня (без учёта валидируемого изменения) 
        }
        currency => 'YND_FIXED', # валюта кампании, для которой устанавливается дневной бюджет; обязательный параметр!
        new_camp => 1|0, # новая ли кампания? для новой кампании не проверяется количество изменений дневного бюджета
        wallet_day_budget => { # дневной бюджет общего счета кампании - для выдачи warning, если настройки на кампанию будут перекрыты настройками на общем счету
            day_budget => 100,
            day_budget_show_mode => 'default',
        }
    Возвращает либо текст ошибки, либо undef.

    $err = validate_day_budget(strategy => 'default', day_budget => 123, day_budget_show_mode => 'stretched');

=cut

sub _validate_day_budget {
    my (%O) = @_;

    my $vr = Direct::ValidationResult->new();

    if (! $O{is_wallet}) {
        die 'strategy required' unless $O{strategy};

        if ($O{is_autobudget} || none { $O{strategy} eq $_ } @Campaign::MANUAL_PRICE_STRATEGIES) {
            $vr->add_generic(error_InconsistentState(iget('Дневной бюджет можно использовать только совместно с ручными стратегиями')));
        }
    }

    unless (defined $O{new_day_budget_data}{day_budget} && is_valid_float($O{new_day_budget_data}{day_budget}) && $O{new_day_budget_data}{day_budget} >= 0) {
        $vr->add(day_budget => error_InvalidField(iget('Неверно указана сумма дневного бюджета')));
    }

    if ($O{new_day_budget_data}{day_budget} > 0) {
        unless ($O{new_day_budget_data}{day_budget_show_mode} && any { $O{new_day_budget_data}{day_budget_show_mode} eq $_ } qw/default stretched/) {
            $vr->add(day_budget_show_mode => error_InvalidField(iget('Неверно указан режим показа объявлений')));
        }

        my $currency = $O{currency};

        die 'no currency given' unless $currency;

        my $min_day_budget_const_name = 'MIN_DAY_BUDGET';
        if ($O{is_wallet}) {
            $min_day_budget_const_name = 'MIN_WALLET_DAY_BUDGET';
        }
        if ($O{new_day_budget_data}{day_budget} < get_currency_constant($currency, $min_day_budget_const_name)) {
            $vr->add(day_budget => error_InvalidField(iget('Минимальная сумма дневного бюджета %s', format_const($currency, $min_day_budget_const_name))));
        } elsif ($O{new_day_budget_data}{day_budget} > get_currency_constant($currency, 'MAX_DAILY_BUDGET_AMOUNT')) {
            $vr->add(day_budget => error_InvalidField(iget('Максимальная сумма дневного бюджета %s', format_const($currency, 'MAX_DAILY_BUDGET_AMOUNT'))));
        }

        if ($O{wallet_day_budget} && ($O{wallet_day_budget}{day_budget} - 0) > $Currencies::EPSILON) { # на ОС есть дневной бюджет
            if ($O{new_day_budget_data}{day_budget} > $O{wallet_day_budget}{day_budget}) {
                $vr->add(day_budget => warning_NoEffect_WalletDayBudget(iget('Дневной бюджет на общем счёте меньше, чем дневной бюджет на кампании, так что расходы будут ограничены дневным бюджетом на общем счёте')));
            }
            if ($O{new_day_budget_data}{day_budget_show_mode} ne $O{wallet_day_budget}{day_budget_show_mode}) {
                $vr->add(day_budget_show_mode => warning_NoEffect_WalletDayBudgetShowMode(iget('Режим показов общего счета перекрывает режим показов кампании')));
            }
        }
    }

    if (!$O{new_camp}) {
        my $can_set_day_budget_err = _can_day_budget_be_set(before => $O{old_day_budget_data}, after => $O{new_day_budget_data});
        if ($can_set_day_budget_err) {
            $vr->add_generic($can_set_day_budget_err);
        }
    }

    return $vr;
}

=head2 validate_struct_day_budget

    Проверяет структуру хеша с дневным бюджетом из фронта.
    Проверяется наличие ключей set, sum, show_mode и типы их значений

=cut

sub validate_struct_day_budget {

    return !validate_structure(shift, {
        set => JSON::true,
        map { $_ => 1 } qw/sum show_mode/
    });
}


=head2 _can_day_budget_be_set

    Проверяет возможность изменения параметров дневного бюджета.
    Принимает обязательные именованные параметры before и after, описывающие текущие и желаемые значения для дневного бюджета соответственно.
    Возвращает либо текст ошибки, либо undef.

    $err = can_day_budget_be_set(
        before => {
            day_budget => 123, # текущая сумма дневного бюджета (или 0, если дневной бюджет отключен)
            day_budget_daily_change_count => 2, # количество изменений суммы дневного бюджета за текущие сутки
        },
        after => {
            day_budget => 456, # желаемая сумма дневного бюджета (или 0, если дневной бюджет отключен)
        },
    );

=cut

sub _can_day_budget_be_set {
    my (%O) = @_;

    die 'no after data' unless $O{after} && %{$O{after}} && defined $O{after}{day_budget};
    die 'day_budget_daily_change_count not specified' unless defined $O{before}{day_budget_daily_change_count};

    my $before_change_count = $O{before}{day_budget_daily_change_count};

    # если достигнут предел числа изменений суммы в день, то дневной бюджет можно только отключить
    if ($O{after}{day_budget} > 0 && ($O{before}{day_budget} || 0) != $O{after}{day_budget} && $before_change_count >= $MAX_DAY_BUDGET_DAILY_CHANGE_COUNT) {
        return error_CantPerform(iget('Дневной бюджет можно менять не более %d раз в день.', $MAX_DAY_BUDGET_DAILY_CHANGE_COUNT));
    }

    return undef;
}

1;
