package Direct::Validation::Domains;

use Direct::Modern;

use base qw/ Exporter /;

our @EXPORT = qw/
    validate_disabled_domains
    validate_disabled_video_placements
/;

use Direct::Validation::Errors;
use Direct::ValidationResult;

use Cache::SizeAwareMemoryCache;
use List::MoreUtils qw/uniq/;
use List::Util qw/ first /;

use Yandex::DBTools;
use Yandex::I18n qw/ iget /;
use Yandex::IDN qw/ idn_to_ascii is_valid_domain /;
use Yandex::MirrorsTools::Hostings qw/ strip_www /;

use Direct::Validation::InternalPagesDomains;

use Settings;


our $MAX_DOMAIN_LENGTH = 255;

my $SSP_CACHE = Cache::SizeAwareMemoryCache->new({namespace => 'SSP_Platforms', default_expires_in => 60});

=head2 validate_disabled_video_placements

    Проверка списка площадок РСЯ, на которых запрещены показы видео объявлений в кампании

    Позиционные параметры:
        $placements - список площадок
        $client_limits - лимиты клиента

=cut

sub validate_disabled_video_placements {
    my ($placements, %options) = @_;

    my $vr = validate_disabled_domains($placements, skip_ssp => 1, %options);
    my $video_blacklist_limit = $options{blacklist_size_limit} || $Settings::DEFAULT_VIDEO_BLACKLIST_SIZE_LIMIT;
    if ( scalar( @$placements ) > $video_blacklist_limit ) {
        $vr->add_generic(
            error_ReachLimit(iget('Размер списка #field# превышает максимально допустимый размер %d', $video_blacklist_limit))
        );
    }
    return $vr;
}



=head2 validate_disabled_domains

    Проверка доменов - площадок РСЯ, на которых запрещены показы объявлений в кампании

    Позиционные параметры:
        $domains - список доменов

    Именованные параметры %options:
        check_duplicates            - предупреждение если есть повторяющиеся значения
        skip_ssp                    - не искать в списке ssp-площадки, все значения полагать доменами
        disable_any_domains_allowed - возможность запрещать любые домены
        disable_mail_ru_domain_allowed - возможность запрещать домены mail.ru
        disable_number_id_and_short_bundle_id_allowed - возможность запрещать числовые и короткие app_id
        blacklist_size_limit - ограничение сверху на количество запрещенных доменов
        skip_blacklist_size_limit - не использовать ограничение сверхуц

    аналог Common::validate_domains + новые модели

=cut

sub validate_disabled_domains {
    my ( $domains, %options ) = ( shift, @_ );

    my $vresults = Direct::ValidationResult->new;

    if (!$options{skip_blacklist_size_limit}) {
        my $blacklist_limit = $options{blacklist_size_limit} || $Settings::DEFAULT_GENERAL_BLACKLIST_SIZE_LIMIT;
        # Раньше была такая логика: проверялось что не превышено кол-во отключенных площадок, но
        # только для случае если набор площадок отличается от набора в БД

        if ( scalar( @$domains ) > $blacklist_limit ) {
            $vresults->add_generic(
                error_ReachLimit( iget('Размер списка #field# превышает максимально допустимый размер %d', $blacklist_limit) )
            );
        }
    }
    my %seen_domains;
    my $valid_ssp;
    for my $domain ( @$domains ) {

        # вычленяем ssp-площадки
        if (!$options{skip_ssp}) {
            $valid_ssp ||= +{reverse %{get_known_ssp_platforms()}};
            my $key = _get_ssp_key($domain);
            next  if $valid_ssp->{$key};
        }

        # предполагаем, что в названии площадки домен - только до первого пробела, а дальше могут быть пояснения
        $domain =~ s/ \s .+ $ //xms;

        my $domain_ascii = idn_to_ascii($domain);

        # удаляем www. для всех доменов, кроме доменов общего пользования
        if ( $domain_ascii !~ /(^|\.)(pp|ac|boom|msk|spb|nnov)\.ru|(net|org|com|int|edu)\.[-a-z]+$/i ) {
            $domain_ascii = strip_www( $domain_ascii );
        }

        my $is_valid_domain = is_valid_domain($domain_ascii);
        my $is_valid_app_id = is_valid_app_id($domain, $options{disable_number_id_and_short_bundle_id_allowed});

        # если уже есть ошибка о том, что площадку запрещать нельзя, то не нужно добавлять аналогичное предупреждение
        my $has_cant_disable_error = 0;

        # Площадка может быть доменом, а может быть идентификатором мобильного приложения
        if (!$is_valid_domain && !$is_valid_app_id) {
            $vresults->add_generic(error_InvalidFormat(
                    iget('Элемент %s списка #field# - неправильный формат домена или идентификатора мобильного приложения', $domain)
                ));
        # часть яндексовских доменов запрещать нельзя (DIRECT-11313 & DIRECT-24204). Но некоторым можно - DIRECT-109732
        } elsif (   ($domain_ascii =~ /^((m|www)\.)?((direct)\.)?(yandex|ya)\.[a-z]+$/i
                 || lc( $domain_ascii ) eq 'xn--d1acpjx3f.xn--p1ai'
                 || ( !$options{disable_mail_ru_domain_allowed} &&  lc( $domain_ascii ) =~ /^((go|www|)\.)?mail\.ru/) )
            && !$options{disable_any_domains_allowed}
        ) {
            $vresults->add_generic( error_BadUsage( iget('Элемент %s списка #field# - отключать показы на этом домене нельзя', $domain) ) );
            $has_cant_disable_error = 1;
        } elsif ($domain_ascii =~ /^(\*\.)?((pp|ac|boom|msk|spb|nnov)\.ru|(net|org|com|int|edu)\.[a-z]{2})$/i) {
            $vresults->add_generic( error_BadUsage( iget('Элемент %s списка #field# - для этого домена можно блокировать только третий уровень', $domain) ) );
        } elsif ( length( $domain ) >= $MAX_DOMAIN_LENGTH ) {
            $vresults->add_generic( error_MaxLength( iget('Элемент %s списка #field# - превышена максимально допустимая длина %s', $domain, $MAX_DOMAIN_LENGTH) ) );
        }

        if ( $options{check_duplicates} && exists $seen_domains{ lc( $domain ) } ) {
            $vresults->add_generic( warning_DuplicatedItems( iget('Элемент %s списка #field# - повторяющееся значение', $domain) ) );
        }

        if ( $options{show_internal_pages_warning} && $Direct::Validation::InternalPagesDomains::INTERNAL_PAGES_DOMAINS{ lc( $domain ) } && !$has_cant_disable_error ) {
            $vresults->add_generic( warning_NoEffect_DisableInternalPageDomain( iget('Элемент %s списка #field# является площадкой Яндекса. Запрет на площадки Яндекса и турбо-сайты начнет применяться в сетях с 10 февраля.', $domain) ) );
        }

        $seen_domains{ lc( $domain ) }++;
    }

    return $vresults;
}


=head2 is_valid_app_id

Проверка валидности идентификатора мобильного приложения

$app_id - app_id для валидации
$disable_number_id_and_short_bundle_id_allowed - разрешать ли числовые и короткие app_id

=cut

sub is_valid_app_id {
    my ($app_id, $disable_number_id_and_short_bundle_id_allowed) = @_;

    state $first_part_re = qr/[A-Za-z][A-Za-z0-9_\-]*/;
    state $rest_part_re = qr/[A-Za-z0-9][A-Za-z0-9_\-]*/;
    state $app_re = qr/^$first_part_re(?:\.$rest_part_re)+$/;

    state $short_app_id = qr/^$rest_part_re$/;
    state $number_id = qr/^[1-9][0-9]*$/;

    if ($disable_number_id_and_short_bundle_id_allowed) {
        if ($app_id =~ $number_id || $app_id =~ $short_app_id) {
            return 1;
        }
    }

    return $app_id =~ $app_re;
}


=head2 get_known_ssp_platforms

Получить из базы список известных площадок (кешируется)
Результат: хеш { $ssp => lc($ssp), ... }

=cut

sub get_known_ssp_platforms {
    my $data = $SSP_CACHE->get('ssp');

    if (!$data) {
        my $ssps = get_one_column_sql(PPCDICT, 'SELECT title FROM ssp_platforms');
        $data = +{ map {($_ => _get_ssp_key($_))} @$ssps };
        $SSP_CACHE->set(ssp => $data);
    }

    return $data;
}


sub _get_ssp_key {
    my ($ssp) = @_;
    return lc $ssp;
}



=head2 split_disabled_platforms

Разделяет список на известные SSP и всё остальное.

    my $split = split_disabled_platforms( [ 'Rubicon', 'facebook.com' ],  10000);
    ## $split = { ssps => [ 'Rubicon' ], rest => ['facebook.com' ] }

Внутри ssps названия приведены к эталонному регистру.
Дубли не удаляет.

=cut

sub split_disabled_platforms {
    my ($candidates, %options) = @_;

    my ( @ssps, @rest );
    my $valid_ssp;
    for my $candidate ( @$candidates ) {
        $valid_ssp ||= +{reverse %{get_known_ssp_platforms()}};

        my $key = _get_ssp_key($candidate);
        if ( exists $valid_ssp->{$key} ) {
            push @ssps, $valid_ssp->{$key};
        }
        # площадки, совпадающие с доменами, помещаем и в домены тоже
        push @rest, $candidate  if validate_disabled_domains([$candidate], skip_ssp => 1, %options)->is_valid;
    }

    # т.к. validate_disabled_domains предполагает, что в названии площадки домен - только до первого пробела,
    # а дальше могут быть пояснения то и мы отбрасываем первый пробел и все что после него
    @rest = map {s/\s.+$//rms} @rest;
    unless (validate_disabled_domains(\@rest, skip_ssp => 1, %options)->is_valid) {
        return { ssps => [uniq @ssps], rest => [] };
    }

    return { ssps => [uniq @ssps], rest => [uniq @rest] };
}


=head2 merge_disabled_platforms

Склеивает список отключенных платформ, удаляет дубли

=cut

sub merge_disabled_platforms {
    my ($platforms) = @_;

    my %is_ssp = map {_get_ssp_key($_) => 1} @{$platforms->{ssps}};
    my @domains = grep {!$is_ssp{_get_ssp_key($_)}} @{$platforms->{rest}};

    return [@domains, @{$platforms->{ssps}}];
}

1;
