package Direct::Validation::DynamicConditions;

use Direct::Modern;

use base qw(Exporter);

use Settings;

use Encode;
use Yandex::I18n;
use List::MoreUtils qw/any part/;

use Currencies qw/get_currency_constant/;
use Currency::Format qw/format_const/;
use FilterSchema;

use Direct::ValidationResult;
use Direct::Errors::Messages;
use PrimitivesIds qw/get_clientid/;

use Direct::Validation::HRef qw/validate_href/;

our @EXPORT = qw/
    validate_dynamic_conditions_for_adgroup
/;

# Максимальное число условий нацеливания в группе
our $MAX_DYNAMIC_CONDITIONS_IN_ADGROUP = 50;

# Максимальное число правил в условии нацеливания
our $MAX_DYNAMIC_CONDITION_RULES = 10;

# Максимальное число аргументов в одном правиле
our $MAX_ARGUMENTS_IN_DYNAMIC_RULE = 10;

# Максимальная длина названия условия нацеливания
our $MAX_CONDITION_NAME_LENGTH = 100;

# Разрешенные символы для использования в текстовых полях
our $DISALLOW_LETTER_RE = qr/[^\Q${Settings::ALLOW_BANNER_LETTERS_STR}|\E]/i;

=head2 validate_dynamic_conditions_for_adgroup($checked_dyn_conds, $remaining_dyn_conds, $campaign)

Валидация условий нацеливания на группу.
Параметры:
    $checked_dyn_conds   -> список проверяемых (новых или изменившихся) условий нацеливания
    $remaining_dyn_conds -> список оставшихся (не изменившихся) условий нацеливания
    $campaign            -> хеш кампании (не модель(!); Models::Campaign::get_user_camp_gr($uid, $cid, {no_groups=>1}))

=cut

sub validate_dynamic_conditions_for_adgroup {
    my ($checked_dyn_conds, $remaining_dyn_conds, $campaign) = @_;
    $remaining_dyn_conds //= [];

    my $vr_main = validate_dynamic_conditions($checked_dyn_conds, $campaign);

    my $dyn_conds_count = @$checked_dyn_conds + @$remaining_dyn_conds;

    if ($dyn_conds_count > $MAX_DYNAMIC_CONDITIONS_IN_ADGROUP) {
        $vr_main->add_generic(error_LimitExceeded(
            iget("Количество условий нацеливания в группе не может быть больше %s", $MAX_DYNAMIC_CONDITIONS_IN_ADGROUP)
        ));
    }

    # Если на этом этапе есть ошибки, то дальнейшие проверки не делаем.
    # Т.к. далее будет проверяться уникальность и прочее, и нужны как минимум корректно заполненные структуры.
    return $vr_main unless $vr_main->is_valid;

    my (%dyn_cond_uhash, %duplicated);

    # Добавим в хеш существующие условия нацеливания
    for my $dyn_cond (@$remaining_dyn_conds) {
        my $uhash = $dyn_cond->get_condition_uhash;
        if (defined $dyn_cond_uhash{ $uhash }) {
            croak "conditions in adgroup " . $dyn_cond->adgroup_id . " already duplicated";
        }
        $dyn_cond_uhash{ $uhash } = -1;
    }

    my $i = 0;
    for my $dyn_cond (@$checked_dyn_conds) {
        my $uhash = $dyn_cond->get_condition_uhash;
        if (defined $dyn_cond_uhash{ $uhash }){
            $duplicated{$i} = 1;
            $duplicated{$dyn_cond_uhash{ $uhash }} //= 1;
        }
        else{
            $dyn_cond_uhash{ $uhash } = $i;
        }
        $i++;
    }

    for (my $i = 0; $i < @$checked_dyn_conds; $i++) {
        my $dyn_cond = $checked_dyn_conds->[$i];
        my $vr = $vr_main->get_objects_results->[$i];
        
        my $length = length(encode( utf8 => $dyn_cond->_condition_json));
        $vr->add_generic(
                error_MaxLength(iget("Превышен максимально допустимый размер списка условий (%s байт при лимите в %s байт)",
                    $length, $Settings::MAX_CONDITION_JSON_LENGTH)
            )) if $length > $Settings::MAX_CONDITION_JSON_LENGTH;

        if ($duplicated{$i}) {
            my $error_text;
            if ($dyn_cond->adgroup->feed_id) {
                $error_text = 'Фильтры в группе объявлений должны быть уникальны';
            } else {
                $error_text = 'Условия нацеливания в группе объявлений должны быть уникальны';
            }
            
            $vr->add_generic(error_InconsistentState(iget($error_text)));
        }
    }

    # Проверим, что цель с правилом `Все страницы` единственная в группе
    my ($all_pages_rule, $other_rules) = 
        part { $_->condition->[0]->type eq 'any' ? 0 : 1 } 
        grep { @{$_->condition} }
        (@$checked_dyn_conds, @$remaining_dyn_conds);
    if ($all_pages_rule && $other_rules) {
        $vr_main->add_generic(error_InconsistentState(
            iget("Условие нацеливания с правилом `Все страницы` не может комбинироваться с другими условиями в группе")
        ));
    }

    return $vr_main;
}

=head2 validate_dynamic_conditions($dyn_conds, $campaign)

Валидация списка условий нацеливания (без какой-либо привязки к группе) в пределах одной кампании.

=cut

sub validate_dynamic_conditions {
    my ($dyn_conds, $campaign) = @_;

    my $client_id = get_clientid(cid => $campaign->{cid});
    my $vr_main = Direct::ValidationResult->new();

    for my $dyn_cond (@$dyn_conds) {
        my $vr = $vr_main->next;

        #
        # condition_name ("Название условия нацеливания")
        #
        my $condition_name = $dyn_cond->has_condition_name ? $dyn_cond->condition_name : undef;
        if (!defined $condition_name || $condition_name !~ /\S/) {
            $vr->add(condition_name => error_ReqField());
        } else {
            if ($condition_name =~ $DISALLOW_LETTER_RE) {
                $vr->add(condition_name => error_InvalidChars_AlphaNumPunct());
            }
            if (length($condition_name) > $MAX_CONDITION_NAME_LENGTH) {
                $vr->add(condition_name => error_MaxLength(undef, length => $MAX_CONDITION_NAME_LENGTH));
            }
        }

        #
        # condition ("Правила условия нацеливания")
        #
        if (!$dyn_cond->has_condition) {
            $vr->add(condition => error_ReqField(iget("Необходимо задать список правил условия нацеливания")));
        } else {
            # все товары
            next unless scalar @{$dyn_cond->condition};
            
            my $adgroup = $dyn_cond->adgroup;
            my $filter_type = '';
            if ($adgroup->has_feed_id && $adgroup->feed_id) {
                my $feed = Direct::Feeds->get_by($client_id, id => $adgroup->feed_id)->items()->[0];
                unless ($feed) {
                    die "feed id = @{[$adgroup->feed_id]} not found";
                }
                $filter_type = $feed->business_type . '_' . $feed->feed_type;
            }
            $vr->add(condition => validate_dynamic_condition_rules($dyn_cond->condition, $filter_type));
        }

        #
        # price ("Цена на поиске")
        # price_context ("Цена на сети")
        # autobudget_priority ("Приоритет автобюджета")
        #
        # Цены валидируем только при наличии стратегии (ее может не быть, если валидируется снапшот данных без привязки к кампании)
        #
        if (exists $campaign->{strategy}) {
            my $is_autobudget = $campaign->{strategy}->{is_autobudget};
            my $is_search_stop = $campaign->{strategy}->{is_search_stop};
            my $is_net_stop = $campaign->{strategy}->{is_net_stop};
            my $is_different_places = $campaign->{strategy}->{name} eq 'different_places';

            if (!$is_autobudget) {
                my $currency = $campaign->{currency};
                my $min_price = get_currency_constant($currency, 'MIN_PRICE');
                my $max_price = get_currency_constant($currency, 'MAX_PRICE');

                for my $price_info (
                    {name => 'price',         is_required => !$is_search_stop},
                    {name => 'price_context', is_required => $is_different_places && !$is_net_stop},
                ) {
                    my $price = $price_info->{name};
                    my $has_price = "has_${price}";

                    if ($price_info->{is_required} && !$dyn_cond->$has_price) {
                        $vr->add($price => error_ReqField());
                        next;
                    }

                    next if !$dyn_cond->$has_price || ($dyn_cond->$price == 0 && !$price_info->{is_required});

                    $vr->add($price => error_InvalidField(
                        iget("Значение в поле #field# не может быть меньше %s", format_const($currency, 'MIN_PRICE'))
                    )) if $dyn_cond->$price < $min_price;

                    $vr->add($price => error_InvalidField(
                        iget("Значение в поле #field# не может быть больше %s", format_const($currency, 'MAX_PRICE'))
                    )) if $dyn_cond->$price > $max_price;
                }
            } else {
                my $autobudget_priority = $dyn_cond->has_autobudget_priority ? $dyn_cond->autobudget_priority : undef;

                if (!defined $autobudget_priority) {
                    $vr->add(autobudget_priority => error_ReqField());
                } elsif ($autobudget_priority !~ /^(?:1|3|5)$/) {
                    $vr->add(autobudget_priority => error_InvalidField());
                }
            }
        }
    }

    return $vr_main;
}

=head2 validate_dynamic_condition_rules($rules)

Валидация правил (`condition` структуры) условия нацеливания.

=cut

sub validate_dynamic_condition_rules {
    my ($rules, $filter_type) = @_; 

    $filter_type ||= 'dynamic_conditions';

    my $validator = FilterSchema->new(filter_type => $filter_type);

    return $validator->check([ map {$_->to_hash} @$rules ]);
}

1;
