package Direct::Validation::Feeds;
use Direct::Modern;

use base qw/Exporter/;
our @EXPORT_OK = qw/
    validate_feed
    is_valid_feed_filename
    is_valid_feed_business_type
    is_valid_feed_type
/;

use Yandex::IDN qw/is_valid_email/;

use Direct::Validation::Errors;
use Direct::Validation::HRef qw/validate_href/;
use Direct::ValidationResult;
use TextTools qw/smartstrip2/;

use Direct::Model::Feed;


=head2 validate_feed

=cut
sub validate_feed {
    my ($feed) = @_;
    my $vr = Direct::ValidationResult->new;

    if (!$feed->has_name) {
        $vr->add(name => error_ReqField());
    } elsif (length($feed->name) == 0) {
        $vr->add(name => error_EmptyField());
    }

    if ($feed->has_source && $feed->source eq 'url') {
        if (!$feed->has_url) {
            $vr->add(url => error_ReqField());
        } elsif (!defined($feed->url) || !length($feed->url)) {
            $vr->add(url => error_EmptyField());
        }
    }

    if ($feed->has_source && $feed->source eq 'file') {
        if (!$feed->has_filename) {
            $vr->add(filename => error_ReqField());
        } elsif (!defined($feed->filename) || !length($feed->filename)) {
            $vr->add(filename => error_EmptyField());
        } elsif (!is_valid_feed_filename($feed->filename)) {
            $vr->add(filename => error_InvalidField());
        }
    }

    if ($feed->has_url) {
        $vr->add(url => validate_href($feed->url, allow_ftp => 1, allow_ip => 1));
    }

    if ($feed->has_email && defined $feed->email) {
        if (!is_valid_email($feed->email)) {
            $vr->add(email => error_InvalidField())
        }
    }

    if ($feed->has_login && defined $feed->login) {
        if (!length(smartstrip2($feed->login))) {
            $vr->add(login => error_EmptyField());
        }
    }

    return $vr;
}


=head2 is_valid_feed_filename

Проверяет имя загруженного файла с фидом на валидность (xml, gz, zip и т.п.).

=cut

sub is_valid_feed_filename {
    my ($filename) = @_;
    state $valid_feed_ext_re = join q{|} => qw/gz zip xml csv tsv yml xls xlsx/;
    state $check_feed_ext_re = qr/\.(?:$valid_feed_ext_re)$/;
    return $filename =~ $check_feed_ext_re;
}


=head2 is_valid_feed_business_type

Проверяет соответствие полученного business_type списку известных типов из Direct::Model::Feed::BUSINESS_TYPE_VALUES

=cut

sub is_valid_feed_business_type {
    my ($value) = @_;
    state $is_valid_business_type = +{ map {($_ => 1)} @{Direct::Model::Feed::BUSINESS_TYPE_VALUES()} };

    return !!$value && !!$is_valid_business_type->{$value};
}

=head2 is_valid_feed_type

Проверяет соответствие полученного feed_type списку известных типов из Direct::Model::Feed::FEED_TYPE_VALUES

=cut

sub is_valid_feed_type {
    my ($value) = @_;
    state $is_valid_feed_type = +{ map {($_ => 1)} @{Direct::Model::Feed::FEED_TYPE_VALUES()} };

    return !!$value && !!$is_valid_feed_type->{$value};
}


1;



