package Direct::Validation::HierarchicalMultipliers;

=head1 NAME

Direct::Validation::HierarchicalMultipliers - валидация корректировок цены (или их наборов), которые можно
привязывать к кампании и/или группе.

=cut
use Direct::Modern;

use base qw(Exporter);
our @EXPORT_OK = qw/validate_hierarchical_multipliers
                    validate_mobile_multiplier
                    validate_desktop_multiplier
                    validate_video_multiplier
                    validate_demography_multiplier_condition
                    validate_demography_multiplier
                    validate_retargeting_multiplier
                    validate_geo_multiplier
                    validate_geo_multiplier_regions
                    validate_ab_segment_multiplier
                    validate_ab_segment_multiplier_segments
                    validate_multiplier_pct
                    validate_performance_tgo_multiplier
                    normalize_age
                    normalize_gender
                    normalize_os_type
                    $DEMOGRAPHY_MULTIPLIER_AGES
                    $ALLOWED_DEMOGRAPHY_MULTIPLIER_AGES
                    $DEMOGRAPHY_MULTIPLIER_GENDERS
                    %DEPRECATED_DEMOGRAPHY_MULTIPLIER_AGE
                    $MOBILE_MULTIPLIER_OS_TYPES
                    get_metadata
                   /;

use Storable qw/dclone/;
use List::MoreUtils qw/all any/;

use Yandex::HashUtils;
use Yandex::ListUtils qw/xminus/;
use Yandex::I18n qw/iget/;

use Settings;
use Direct::ValidationResult;
use Direct::Errors::Messages;
use Yandex::DBTools;
use RBACElementary qw /rbac_get_client_uids_by_clientid/;
use HierarchicalMultipliers::Base;

use Retargeting qw/get_metrika_ab_segments/;
use JavaIntapi::ValidateBidModifiers;

=head1 CONSTANTS

=head2 $DEMOGRAPHY_MULTIPLIER_AGES

Допустимые диапазоны возрастов для условий по демографическому признаку

=cut

our $DEMOGRAPHY_MULTIPLIER_AGES = ['0-17', '18-24', '25-34', '35-44', '45-54', '55-', '45-', 'unknown'];
our %DEPRECATED_DEMOGRAPHY_MULTIPLIER_AGE = (
    '45-' => [ '45-54', '55-' ],
);
our $ALLOWED_DEMOGRAPHY_MULTIPLIER_AGES = xminus($DEMOGRAPHY_MULTIPLIER_AGES, [keys %DEPRECATED_DEMOGRAPHY_MULTIPLIER_AGE]);

=head2 $DEMOGRAPHY_MULTIPLIER_GENDERS

Список известных нам полов.

=cut
our $DEMOGRAPHY_MULTIPLIER_GENDERS = [ 'male', 'female' ];
our $DEMOGRAPHY_MULTIPLIER_GENDER_RE = qr/^(male|female)$/;

=head2 $MOBILE_MULTIPLIER_OS_TYPES

Список известных нам мобильных ОС.

=cut
our $MOBILE_MULTIPLIER_OS_TYPES = [ 'ios', 'android' ];
our $MOBILE_MULTIPLIER_OS_TYPES_RE = qr/^(ios|android)$/;

=head2 $INVENTORY_MULTIPLIER_TYPES

    Список известных типов инвентаря

=cut
our $INVENTORY_MULTIPLIER_TYPES = [qw/
    instream_web
    inpage
    interstitial
    inbanner
    rewarded
/];

=head2 $BANNER_TYPE_MULTIPLIER_TYPES

    Список известных типов баннеров для корректировок цен по типу баннера

=cut
our $BANNER_TYPE_MULTIPLIER_TYPES = ['cpm_banner'];

=head2 $WEATHER_MULTIPLIER_CONDITION_TYPES

    Список типов погодных корректировок

=cut
our $WEATHER_MULTIPLIER_CONDITION_TYPES = ['temp', 'cloudness', 'prec_type', 'prec_strength'];

=head2 $WEATHER_MULTIPLIER_MAX_CONDITION_COUNT

    Максимальное количество корректировок

=cut
our $WEATHER_MULTIPLIER_MAX_CONDITION_COUNT = 20;

=head2 $WEATHER_MULTIPLIER_CONSTRAINTS

   Допустимые значения параметров для каждого типа погодных корректировок

=cut
our $WEATHER_MULTIPLIER_VALUES = {
    temp => {
        range => [-50, 50]
    },
    cloudness => {
        values => [0, 25, 50 , 75, 100]
    },
    prec_type => {
        values => [0, 1, 3]
    },
    prec_strength => {
        values => [0, 25, 50 , 75, 100]
    }
};

=head2 %HIERARCHICAL_MULTIPLIERS

Настройки различных видов коэффициентов в зависимости от типа кампании. Полные настройки по умолчанию прописаны
с ключом __default__, для конкретных типов кампании переопределяется только нужное.

=cut
our %HIERARCHICAL_MULTIPLIERS = (
    __default__ => {
        mobile_multiplier => {
            pct_min => 50,
            pct_max => 1300,
            enabled => 1,
        },
        mobile_multiplier_with_feature => {
            pct_min => 0,
            pct_max => 1300,
            enabled => 1,
        },
        tablet_multiplier => {
            pct_min => 0,
            pct_max => 1300,
            enabled => 1,
        },
        desktop_multiplier => {
            pct_min => 0,
            pct_max => 1300,
            enabled => 1,
        },
        desktop_only_multiplier => {
            pct_min => 0,
            pct_max => 1300,
            enabled => 1,
        },
        smarttv_multiplier => {
            pct_min => 0,
            pct_max => 1300,
            enabled => 0,
        },
        desktop_multiplier_with_mobile_os_feature => {
            pct_min => 0,
            pct_max => 400,
            enabled => 1,
        },
        video_multiplier => {
            pct_min => 50,
            pct_max => 1300,
            enabled => 1,
        },
        demography_multiplier => {
            pct_min => 0,
            pct_max => 1300,
            enabled => 1,
            max_conditions => 14,
        },
        retargeting_multiplier => {
            pct_min => 0,
            pct_max => 1300,
            enabled => 1,
            max_conditions => 100,
        },
        geo_multiplier => {
            pct_min => 10,
            pct_max => 1300,
            enabled => 1,
        },
        ab_segment_multiplier => {
            pct_min => 0,
            pct_max => 1300,
            enabled => 1,
        },
        performance_tgo_multiplier => {
            pct_min => 20,
            pct_max => 1300,
            enabled => 0,
        },
        banner_type_multiplier => {
            pct_min => 0,
            pct_max => 1300,
            enabled => 0,
        },
        inventory_multiplier => {
            pct_min => 0,
            pct_max => 1300,
            enabled => 0,
        },
        weather_multiplier => {
            pct_min => 0,
            pct_max => 1300,
            enabled => 0,
        },
        express_traffic_multiplier => {
            pct_min => 0,
            pct_max => 1300,
            max_conditions => 20,
            enabled => 0,
        },
        trafaret_position_multiplier => {
            pct_min => 100,
            pct_max => 1300,
            enabled => 0,
        },
        prisma_income_grade_multiplier => {
            pct_min => 100,
            pct_max => 1300,
            max_conditions => 4,
            enabled => 0,
        },
        retargeting_filter => {
            pct_min => 0,
            pct_max => 0,
            max_conditions => 50,
            enabled => 1,
        },
    },
    dynamic => {
        video_multiplier => {
            enabled => 0,
        },
        prisma_income_grade_multiplier => {
            enabled => 1
        },
    },
    mobile_content => {
        mobile_multiplier => {
            enabled => 0,
        },
        tablet_multiplier => {
            enabled => 0,
        },
        mobile_multiplier_with_feature => {
            enabled => 0,
        },
        desktop_multiplier => {
            enabled => 0,
        },
        desktop_only_multiplier => {
            enabled => 0,
        },
        video_multiplier => {
            enabled => 0,
        },
        prisma_income_grade_multiplier => {
            enabled => 1
        }
    },
    mcbanner => {
        mobile_multiplier => {
            enabled => 0,
        },
    },
    cpm_banner => {
        mobile_multiplier_with_feature => {
            enabled => 0,
        },
        desktop_multiplier => {
            enabled => 0,
        },
        tablet_multiplier => {
            enabled => 0,
        },
        desktop_only_multiplier => {
            enabled => 0,
        },
        banner_type_multiplier => {
            enabled => 1,
        },
        inventory_multiplier => {
            enabled => 1,
        },
        weather_multiplier => {
            enabled => 1,
        },
        express_traffic_multiplier => {
            enabled => 1,
        },
    },
    performance => {
        performance_tgo_multiplier => {
            enabled => 1,
        },
        prisma_income_grade_multiplier => {
            enabled => 1
        },
    },
    text => {
        prisma_income_grade_multiplier => {
            enabled => 1
        },
    },
);


my %VALIDATION_RULES = (
    mobile_multiplier => sub { my ($campaign_type, $ClientID, $data) = @_; validate_mobile_multiplier($campaign_type, $ClientID, $data); },
    desktop_multiplier => sub { my ($campaign_type, $ClientID, $data) = @_; validate_desktop_multiplier($campaign_type, $ClientID, $data); },
    demography_multiplier => sub { my ($campaign_type, $ClientID, $data) = @_; validate_demography_multiplier($campaign_type, $data); },
    retargeting_multiplier => \&validate_retargeting_multiplier,
    geo_multiplier  => sub { my ($campaign_type, $ClientID, $data) = @_; validate_geo_multiplier($campaign_type, $data); },
    ab_segment_multiplier  => sub { my ($campaign_type, $ClientID, $data) = @_; validate_ab_segment_multiplier($campaign_type, $ClientID, $data); },
    video_multiplier => sub { my ($campaign_type, $ClientID, $data) = @_; validate_video_multiplier($campaign_type, $data); },
    performance_tgo_multiplier => sub { my ($campaign_type, $ClientID, $data) = @_; validate_performance_tgo_multiplier($campaign_type, $data); },
);

=cut

=head1 FUNCTIONS

=head2 normalize_age

Принимает на вход строку с перечислением диапазонов возврастов, возвращает её в том виде, как она будет
выбираться из базы: в порядке, указанном в 'CREATE TABLE' для этого поля.

Специальный случай для 'all' - из-за требований интерфейса, модели там не терпят null в текстовых полях.

=cut
sub normalize_age {
    my ($age) = @_;
    return undef unless $age;
    return undef if $age eq 'all';

    for my $valid_age (@$DEMOGRAPHY_MULTIPLIER_AGES) {
        return $valid_age if $age eq $valid_age;
    }

    return undef;
}

=head2 normalize_gender

Нормализует пол до состояния хранимого в базе. Непонятные данные превращаем в "любой пол"(undef)

Специальный случай для 'all' - из-за требований интерфейса, модели там не терпят null в текстовых полях.

=cut
sub normalize_gender {
    my ($gender) = @_;
    return $gender if defined($gender) && $gender =~ $DEMOGRAPHY_MULTIPLIER_GENDER_RE;
    return undef if defined($gender) && $gender eq 'all';
    return undef;
}

=head2 normalize_os_type

Нормализует тип мобильной ОС до состояния хранимого в базе. Непонятные данные превращаем в undef

=cut
sub normalize_os_type {
    my ($os_type) = @_;
    return $os_type if defined($os_type) && $os_type =~ $MOBILE_MULTIPLIER_OS_TYPES_RE;
    return undef;
}

=head2 get_metadata

Метаданные о коэффициентах (границы, допустимость использования) по типу кампании, для передачи в вёрстку.

    my $result = get_metadata($campaign_type);

Где $result по структуре совпадает с $HIERARCHICAL_MULTIPLIERS{__default__}.

=cut
sub get_metadata {
    my ($campaign_type) = @_;
    state $cache = {};

    if (!exists $cache->{$campaign_type}) {
        my $data = dclone($HIERARCHICAL_MULTIPLIERS{__default__});
        if (exists $HIERARCHICAL_MULTIPLIERS{$campaign_type}) {
            my $type_options = $HIERARCHICAL_MULTIPLIERS{$campaign_type};
            for my $coef_name (keys %$type_options) {
                hash_merge $data->{$coef_name}, $type_options->{$coef_name},
            }
        }
        $cache->{$campaign_type} = $data;
    }
    return $cache->{$campaign_type};
}

=head2 validate_mobile_multiplier

Валидация отдельно взятой корректировки для мобильного

    my $validation_result = validate_mobile_multiplier($campaign_type, { multiplier_pct => 133 });

Возвращает Direct::ValidationResult.

=cut
sub validate_mobile_multiplier {
    my ($campaign_type, $ClientID, $data) = @_;

    my $client_has_cpc_devices_feature = Client::ClientFeatures::has_cpc_device_modifiers_allowed_feature($ClientID);
    my $client_has_mobile_os_feature = Client::ClientFeatures::is_mobile_os_bid_modifier_enabled($ClientID);

    my $result = Direct::ValidationResult->new;

    my $meta;

    if ($client_has_cpc_devices_feature || $client_has_mobile_os_feature) {
        $meta = get_metadata($campaign_type)->{mobile_multiplier_with_feature};
    } else {
        $meta = get_metadata($campaign_type)->{mobile_multiplier};
    }

    if ($meta->{enabled}) {
        if ($client_has_cpc_devices_feature) {
            $result->add_generic(validate_multiplier_pct($data->{multiplier_pct},
                $meta->{pct_min}, $meta->{pct_max}));
        } elsif ($client_has_mobile_os_feature) {
            #Для такого случая минимальная корректировка на смартфоны без указания ОС — 50.
            $result->add_generic(validate_multiplier_pct($data->{multiplier_pct},
                !$data->{os_type} ? 50 : $meta->{pct_min}, $meta->{pct_max}));
        }
        else {
            $result->add_generic(validate_multiplier_pct($data->{multiplier_pct}, $meta->{pct_min}, $meta->{pct_max}));
        }
    }
    else {
        $result->add_generic(error_NotSupported_Multiplier());
    }

    if (!$client_has_cpc_devices_feature && !$client_has_mobile_os_feature && $data->{os_type}) {
        $result->add_generic(error_InvalidField(iget('Тип операционной системы нельзя указывать в корректировках ставок для данного типа кампании или группы')));
    }

    if (defined $data->{desktop_multiplier_pct} && $data->{desktop_multiplier_pct} == 0 && $data->{multiplier_pct} == 0 && !$data->{os_type}){
        $result->add_generic(error_InvalidField(iget('Нельзя устанавливать корректировку -100% на все устройства')));
    }

    return $result;
}

=head2 validate_desktop_multiplier

Валидация отдельно взятой корректировки для desktop

    my $validation_result = validate_desktop_multiplier($campaign_type, { multiplier_pct => 133 });

Возвращает Direct::ValidationResult.

=cut
sub validate_desktop_multiplier {
    my ($campaign_type, $ClientID, $data) = @_;

    my $result = Direct::ValidationResult->new;
    my $client_has_cpc_devices_feature = Client::ClientFeatures::has_cpc_device_modifiers_allowed_feature($ClientID);
    my $client_has_mobile_os_feature = Client::ClientFeatures::is_mobile_os_bid_modifier_enabled($ClientID);

    my $meta;
    if ($client_has_cpc_devices_feature) {
        $meta = get_metadata($campaign_type)->{desktop_multiplier};
    } elsif ($client_has_mobile_os_feature) {
        $meta = get_metadata($campaign_type)->{desktop_multiplier_with_mobile_os_feature};
    }

    if (defined $meta && $meta->{enabled}) {
        $result->add_generic(validate_multiplier_pct($data->{multiplier_pct}, $meta->{pct_min}, $meta->{pct_max}));
    }
    else {
        $result->add_generic(error_NotSupported_Multiplier());
    }

    if (defined $data->{mobile_multiplier_pct} && $data->{mobile_multiplier_pct} == 0 && $data->{multiplier_pct} == 0){
        $result->add_generic(error_InvalidField(iget('Нельзя устанавливать корректировку -100% на все устройства')));
    }

    return $result;
}

=head2 validate_performance_tgo_multiplier($campaign_type, $ClientID, $data)

    валидация Смарт-ТГО корректировки
    Возвращает Direct::ValidationResult.

=cut

sub validate_performance_tgo_multiplier {
    my ($campaign_type, $data) = @_;

    return _validate_simple_multiplier($campaign_type, $data, 'performance_tgo_multiplier');
}

=head2 validate_video_multiplier

Валидация отдельно взятой видео-корректировки

    my $validation_result = validate_video_multiplier($campaign_type, { multiplier_pct => 133 });

Возвращает Direct::ValidationResult.

=cut
sub validate_video_multiplier {
    my ($campaign_type, $data) = @_;

    return _validate_simple_multiplier($campaign_type, $data, 'video_multiplier');
}

sub _validate_simple_multiplier {
    my ($campaign_type, $data, $multiplier_type) = @_;

    my $result = Direct::ValidationResult->new;
    my $meta = get_metadata($campaign_type)->{$multiplier_type};

    if ($meta->{enabled}) {
        $result->add_generic(validate_multiplier_pct($data->{multiplier_pct}, $meta->{pct_min}, $meta->{pct_max}));
    } else {
        $result->add_generic(error_NotSupported_Multiplier());
    }
    return $result;
}

=head2 validate_multiplier_pct

Валидация коэффициента цены, как отдельно взятого числа.

=cut
sub validate_multiplier_pct {
    my ($multiplier, $min, $max) = @_;

    my @errors_arr;
    if (defined $multiplier) {
        for ($multiplier) {
            if (/\D/) {
                push @errors_arr, error_InvalidField_NotPositive(iget('Значение коэффициента должно быть целым положительным числом'));
                last;
            }
            my $looks_like_number = /^[-+]?\d+(?:\.\d+)?$/a;
            $looks_like_number and $_ < $min and push @errors_arr, error_InvalidField(iget('Значение коэффициента не может быть меньшe %d', $min));
            $looks_like_number and $_ > $max and push @errors_arr, error_InvalidField(iget('Значение коэффициента не может быть больше %d', $max));
        }
    } else {
        push @errors_arr, error_ReqField(iget('Не указано значение коэффициента'));
    }

    return \@errors_arr;
}

=head2 validate_demography_multiplier_condition

Валидация отдельно взятой корректировки по соцдему.

    my $vr = validate_demography_multiplier_condition($campaign_type, {
        age => ...,
        gender => ...,
        multiplier_pct => 135,
    });

=cut
sub validate_demography_multiplier_condition {
    my ($campaign_type, $data, %opts) = @_;
    my $result = $opts{target_validation_result} || Direct::ValidationResult->new;
    my $meta = get_metadata($campaign_type)->{demography_multiplier};

    for my $field (qw/age gender/) {
        $result->add($field => error_ReqField()) unless exists $data->{$field};
    }

    $result->add(age => validate_demography_age($data->{age}));
    $result->add(gender => validate_demography_gender($data->{gender}));
    $result->add(multiplier_pct => validate_multiplier_pct($data->{multiplier_pct}, $meta->{pct_min}, $meta->{pct_max}));
    if (!normalize_age($data->{age}) && !normalize_gender($data->{gender})) {
        $result->add_generic(error_RequiredAtLeastOneOfFields(iget('Обязательно должен быть указан пол или возраст')));
    }
    return $result;
}

=head2 validate_demography_age

Валидация строки, где перечислены диапазоны возрастов.

=cut
sub validate_demography_age {
    my ($age) = @_;
    my @errors;
    if (defined $age) {
        if ($age ne 'all') {
            my $is_valid = grep { $_ eq $age } @{$DEMOGRAPHY_MULTIPLIER_AGES};
            push @errors, error_InvalidField(iget('Неверно указан возраст')) unless $is_valid;
        }
    }
    return \@errors;
}

=head2 validate_demography_gender

Валидация строки, указывающей пол.

=cut
sub validate_demography_gender {
    my ($gender) = @_;
    my @errors;
    if (defined($gender) && $gender ne 'all') {
        push @errors, error_InvalidField(iget('Неверно указан пол')) unless $gender =~ $DEMOGRAPHY_MULTIPLIER_GENDER_RE;
    }
    return \@errors;
}

=head2 validate_demography_multiplier

Валидация всего набора корректировок по соцдему.

=cut
sub validate_demography_multiplier {
    my ($campaign_type, $data) = @_;

    my $result = Direct::ValidationResult->new;
    my $meta = get_metadata($campaign_type)->{demography_multiplier};
    if (!$meta->{enabled}) {
        $result->add_generic(error_NotSupported_Multiplier());
        return $result;
    }
    if (scalar(@{$data->{conditions}}) > $meta->{max_conditions}) {
        $result->add_generic(error_ReachLimit(iget('Количество корректировок ставок для демографических сегментов аудитории в наборе должно быть от %s до %s', 1, $meta->{max_conditions})));
    }
    my @conditions;
    for my $condition (@{$data->{conditions}}) {
        if (defined $condition->{age} && $condition->{age} eq '45-') {
            my $condition_45_54 = dclone($condition);
            $condition_45_54->{age} = "45-54";
            my $condition_55 = dclone($condition);
            $condition_55->{age} = "55-";
            push @conditions, $condition_45_54, $condition_55;
        } else {
            push @conditions, $condition;
        }
    }
    my %coverage_map;
    for my $condition (@conditions) {
        my $cond_result = $result->next;
        validate_demography_multiplier_condition($campaign_type, $condition, target_validation_result => $cond_result);
        my @signatures = expanded_demography_condition_signatures($condition);
        push @{$coverage_map{$_}}, $condition for @signatures;
    }

    my @signatures_with_intersection = grep { @{$coverage_map{$_}} > 1 } keys %coverage_map;
    if (@signatures_with_intersection) {
        $result->add_generic(error_InconsistentState(iget('Пересекаются условия корректировок в наборе')));
    }

    return $result;
}


=head2 expanded_demography_condition_signatures

Раскрывает демографическое условие до списка характерных строк, перечисляющих что именно этим условием
покрывается. Используется для проверки условий на пересечение.

    my @expanded = expanded_demography_condition_signatures({
        age => '0-17,45-',
        gender => undef,
    });

->

    my @expanded = ( 'male:0-17', 'male:45-', 'female:0-17', 'female:45-' );

=cut
sub expanded_demography_condition_signatures {
    my ($condition) = @_;
    my %valid_ages = map { $_ => undef } @{$DEMOGRAPHY_MULTIPLIER_AGES};
    my $normalized_age = ($condition->{age} && $condition->{age} eq 'all') ? undef : $condition->{age};
    my $normalized_gender = ($condition->{gender} && $condition->{gender} eq 'all') ? undef : $condition->{gender};
    my @ages = $normalized_age ? ($normalized_age) : keys %valid_ages;
    my @genders = defined $normalized_gender ? ($normalized_gender) : @$DEMOGRAPHY_MULTIPLIER_GENDERS;
    my %result;
    for my $gender (@genders) {
        for my $age (@ages) {
            $result{"$gender:$age"} = undef;
        }
    }
    return keys %result;
}

=head2 validate_retargeting_multiplier

Проверяет набор корректировок про условиям ретаргетинга.

=cut
sub validate_retargeting_multiplier {
    my ($campaign_type, $ClientID, $data) = @_;
    my $result = Direct::ValidationResult->new();
    my $meta = get_metadata($campaign_type)->{retargeting_multiplier};
    if (!$meta->{enabled}) {
        $result->add_generic(error_NotSupported_Multiplier());
        return $result;
    }

    if (keys %{$data->{conditions}} > $meta->{max_conditions}) {
        $result->add_generic(error_ReachLimit(iget('В наборе должно быть от %s до %s корректировок, основанных на условиях подбора аудитории', 1, $meta->{max_conditions})));
    }

    my $ret_conds = get_hashes_hash_sql(PPC(ClientID => $ClientID), [
        "select ret_cond_id, is_deleted, ClientID from retargeting_conditions",
        where => { ret_cond_id => [keys %{$data->{conditions}}]}
    ]);

    while (my($ret_cond_id, $multiplier_data) = each %{$data->{conditions}}) {
        my $cond_result = Direct::ValidationResult->new;
        my $is_ret_cond_id_valid = defined($ret_cond_id) && $ret_cond_id =~ /^[1-9][0-9]*$/; # positive integer
        if ($is_ret_cond_id_valid) {
            if (  !exists $ret_conds->{$ret_cond_id}
               || $ret_conds->{$ret_cond_id}{ClientID} != $ClientID
               || $ret_conds->{$ret_cond_id}{is_deleted}
            ) {
                $cond_result->add_generic(error_NotFound_RetargetingConditionId(undef, id => $ret_cond_id));
            }
        } else {
            $cond_result->add_generic(error_InvalidField_NotPositive(iget("Значение поля #ret_cond_id_field# должно быть целым положительным числом")));
        }
        $cond_result->add(multiplier_pct =>
              validate_multiplier_pct($multiplier_data->{multiplier_pct}, $meta->{pct_min}, $meta->{pct_max}));
        $result->add($ret_cond_id, $cond_result);
    }

    return $result;
}

=head2 validate_geo_multiplier

Валидация набора корректировок по георегионам.

=cut
sub validate_geo_multiplier {
    my ($campaign_type, $data) = @_;

    my $result = Direct::ValidationResult->new;
    my $meta = get_metadata($campaign_type)->{geo_multiplier};
    if (!$meta->{enabled}) {
        $result->add_generic(error_NotSupported_Multiplier());
        return $result;
    }

    $result = validate_geo_multiplier_regions($campaign_type, $data->{regions}, $result);

    return $result;
}

=head2 validate_geo_multiplier_regions

Валидация корректировок по георегионам.

=cut

sub validate_geo_multiplier_regions {
    my ($campaign_type, $regions, $external_vr) = @_;

    my $result = $external_vr // Direct::ValidationResult->new;
    my $meta = get_metadata($campaign_type)->{geo_multiplier};

    my $existing_regions;
    my $has_intersections = 0;
    for my $region (@$regions) {
        my $region_id = $region->{region_id};
        $has_intersections ||= $existing_regions->{$region_id}++;
        my $region_result = $result->next;
        $region_result->add(multiplier_pct => validate_multiplier_pct($region->{multiplier_pct}, $meta->{pct_min}, $meta->{pct_max}));
    }

    $result->add_generic(error_InconsistentState(iget('Пересекаются регионы корректировок в наборе'))) if $has_intersections;

    return $result;
}

=head2 validate_ab_segment_multiplier

        Валидация набора корректировок по аб-сегментам.

=cut
sub validate_ab_segment_multiplier {
    my ($campaign_type, $ClientID, $data) = @_;

    my $result = Direct::ValidationResult->new;
    my $meta = get_metadata($campaign_type)->{ab_segment_multiplier};
    if (!$meta->{enabled}) {
        $result->add_generic(error_NotSupported_Multiplier());
        return $result;
    }

    $result = validate_ab_segment_multiplier_segments($campaign_type, $ClientID, $data->{ab_segments}, $result);

    return $result;
}

=head2 validate_ab_segment_multiplier_segments

Валидация корректировок по аб-сегментам.

=cut

sub validate_ab_segment_multiplier_segments {
    my ($campaign_type, $ClientID, $segments, $external_vr) = @_;

    my $result = $external_vr // Direct::ValidationResult->new;
    my $meta = get_metadata($campaign_type)->{ab_segment_multiplier};

    my $existing_segments = Retargeting::get_metrika_ab_segments(rbac_get_client_uids_by_clientid($ClientID));
    my %section_id_by_segment_id = map {$_->{segment_id} => $_->{section_id}} @$existing_segments;
    my $data_segments;
    my $has_intersections = 0;
    for my $segment (@$segments) {
        my $existing_section_id = $section_id_by_segment_id{$segment->{segment_id}};
        if (!defined $existing_section_id) {
            $result->add_generic(error_NotFound(iget("Сегмент №%d в архиве", $segment->{segment_id})));
        }
        if (defined $existing_section_id && $existing_section_id != $segment->{section_id}) {
            $result->add_generic(error_NotFound(iget("Эксперимент №%d в архиве", $segment->{section_id})));
        }

        $has_intersections ||= $data_segments->{$segment->{segment_id}}++;
        my $segment_vr = $result->next;
        $segment_vr->add(multiplier_pct => validate_multiplier_pct($segment->{multiplier_pct}, $meta->{pct_min}, $meta->{pct_max}));
    }

    $result->add_generic(error_InconsistentState(iget('Пересекаются корректировки сегментов в наборе'))) if $has_intersections;

    return $result;
}



=head2 validate_hierarchical_multipliers

    Проверяет весь набор корректировок разных типов (на группе или кампании), отдавая валидацию на откуп
    специфичной для каждого типа функции.

    my $validation_result = validate_hierarchical_multipliers($campaign_type, $ClientID, $data, %O);
    Именованные параметры.
        adgroup_type => тип группы, если корректировки выставляются на группу.
        dont_forward_to_java => не передавать управление в java-intapi, нужно для юнит тестов

=cut
sub validate_hierarchical_multipliers {
    my ($campaign_type, $ClientID, $data, %O) = @_;
    my $adgroup_type = delete $O{adgroup_type};
    my $dont_forward_to_java = delete $O{dont_forward_to_java};
    my $vr = Direct::ValidationResult->new;

    # пробрасываем в java-intapi валидацию корректировок на кампанию,
    # а также РМП групп (исторически они валидируются там...)
    if (!$dont_forward_to_java && (!defined($adgroup_type) || $adgroup_type eq 'mobile_content')) {
        my %params_data = (client_id=>$ClientID, campaign_type=>$campaign_type);

        # список types_allowed_on_camp включает в себя все разрешённые корректировки для РМП группы
        foreach my $type (HierarchicalMultipliers::Base::types_allowed_on_camp()) {
            $params_data{$type} = $data->{$type} if defined $data->{$type};
        }
        $vr = JavaIntapi::ValidateBidModifiers->new(%params_data)->call();
    } else  {
        while (my ($type, $data_for_type) = each %$data) {
            if (exists $VALIDATION_RULES{$type}) {
                if ($type eq 'mobile_multiplier' && exists $data->{desktop_multiplier}) {
                    $data_for_type->{desktop_multiplier_pct} = $data->{desktop_multiplier}->{multiplier_pct};
                }
                if ($type eq 'desktop_multiplier' && exists $data->{mobile_multiplier} && !$data->{mobile_multiplier}->{os_type}) {
                    $data_for_type->{mobile_multiplier_pct} = $data->{mobile_multiplier}->{multiplier_pct};
                }
                $vr->add($type, $VALIDATION_RULES{$type}->($campaign_type, $ClientID, $data_for_type));
            } else {
                my $bad_usage = Direct::ValidationResult->new;
                $bad_usage->add_generic(error_BadUsage());
                $vr->add($type, $bad_usage);
            }
        }
    }
    return $vr;
}


1;
