package Direct::Validation::Image;

use Direct::Modern;
use Yandex::I18n;
use Encode;
use HashingTools qw/md5_base64ya/;
use Image::ExifTool qw/:Public/;

use base qw(Exporter);

our @EXPORT = qw/
    validate_images
/;

our @EXPORT_OK = qw//;

use List::MoreUtils qw/any/;

use Direct::Validation::Errors;
use Direct::ValidationResult;

# Форматы изображений для ретины
our @VALID_RETINA_SIZES = qw/
    640x100
    640x200
    640x960
    960x640
    /;

our @VALID_RETINA_SIZES_ORIGIN = qw/
    320x50
    320x100
    320x480
    480x320
    /;

our @VALID_SIZES = (
    map { [split /x/] } qw/
    240x400
    728x90
    300x250
    300x600
    336x280
    300x500
    970x250
    /, @VALID_RETINA_SIZES
);

our @VALID_SIZES_ORIGIN = (
    map { [split /x/] } qw/
    240x400
    728x90
    300x250
    300x600
    336x280
    300x500
    970x250
    /, @VALID_RETINA_SIZES_ORIGIN
);

our @MCBANNER_VALID_SIZES = (
    map { [split /x/] } qw/
    240x400
    /);

=head2 check_image

=cut

sub check_image
{
    my ($data) = @_;

    my $encoded_data = Encode::is_utf8($data) ? Encode::encode_utf8($data) : $data;
    my $md5 = md5_base64ya($encoded_data);

    my $iinfo = eval { ImageInfo(\$data) };
    if ($@) {
        return { error => iget("Неверный формат") };
    }

    my $res = {
        width       => $iinfo->{ImageWidth}//0,
        height      => $iinfo->{ImageHeight}//0,
        contentType => $iinfo->{MIMEType},
        md5         => $md5,
        size        => length($data),
        img         => $data,
    };
    $res->{error} = $iinfo->{error} if defined $iinfo->{error};

    unless ($res->{contentType} and $res->{contentType} =~ m!image/(jpeg|png|gif)! ) {
        return { error => iget("Недопустимый тип файла изображения, используйте графические форматы GIF, JPEG, PNG") }
    }

    return $res;
}

=head2 is_image_size_valid_for_image_ad($width, $height)

=cut

sub is_image_size_valid_for_image_ad {
    my ($width, $height) = @_;
    return any { $_->[0] == $width && $_->[1] == $height } @VALID_SIZES;
}

=head2 is_image_size_valid_for_mcbanner($width, $height)
       Валидация размеров изображения для графических объявлений на поиске (mcbanner)

=cut

sub is_image_size_valid_for_mcbanner {
    my ($width, $height) = @_;
    return any { $_->[0] == $width && $_->[1] == $height } @MCBANNER_VALID_SIZES;
}



=head2 validate_images([$images], %O)

Опции:

    type => $type - тип картинки (imagead | text)

=cut

sub validate_images {
    my ($images, %opt) = @_;

    my $type = $opt{type} // 'image_ad';

    my $vr_main = Direct::ValidationResult->new();

    for my $image (@$images) {
        my $vr = $vr_main->next();

        my $info = check_image($image);
        if ($info->{error} || (!$info->{width} && $info->{height})) {
            $vr->add_generic(error_InvalidFormat(iget("Неверный формат изображения")));
            next;
        }
        my $width = $info->{width};
        my $height = $info->{height};
        if ($type eq "image_ad" && !is_image_size_valid_for_image_ad($width, $height)) {
            $vr->add_generic(error_InvalidFormat(iget("Неверный размер изображения")));
        } elsif ($type eq "mcbanner" && !is_image_size_valid_for_mcbanner($width, $height)) {
            $vr->add_generic(error_InvalidFormat(iget('Размер изображения должен быть 240х400px')));
        }
    }

    return $vr_main;
}

1;

