package Direct::Validation::Ips;

use Direct::Modern;

use base qw/ Exporter /;

our @EXPORT = qw/
    validate_disabled_ips
/;

use Direct::Validation::Errors;
use Direct::ValidationResult;

use IpTools qw/ is_ip_in_list /;

use Yandex::I18n qw/ iget /;

our $MAX_IP_QTY = 25;

=head2 validate_disabled_ips

    Проверка списка заблокированных IP-адресов

    Позиционные параметры:
        $ips - список IP-адресов

    Именованные параметры %options:
        check_duplicates - предупреждение если есть повторяющиеся значения

    аналог Common::validate_disabledIps_arrayref + новые модели (отличие в том,
    что функция из Common возвращает все ошибки сконкатенированными в строку)

=cut

sub validate_disabled_ips {
    my ( $ips, %options ) = ( shift, @_ );

    my $vresults = Direct::ValidationResult->new;

    if ( scalar( @$ips ) > $MAX_IP_QTY ) {
        $vresults->add_generic(
            error_ReachLimit( iget('Размер списка #field# превышает максимально допустимый размер %d', $MAX_IP_QTY) )
        );
    }

    my %seen_ips;
    for my $ip ( @$ips ) {

        if ( $ip !~ /^([0-9]{1,3})\.([0-9]{1,3})\.([0-9]{1,3})\.([0-9]{1,3})$/ ) {
            $vresults->add_generic( error_InvalidFormat( iget('Элемент %s списка #field# - неправильный формат IP-адреса', $ip) ) );
        } else {
            my $ok = 1;
            for my $part ( $1, $2, $3, $4 ) {
                $ok = $ok && ( $part <= 255 ) && !( ( $part > 0 ) && ( $part =~ /^0/ ) );
            }

            if ( ! $ok ) {
                $vresults->add_generic( error_InvalidFormat( iget('Элемент %s списка #field# - неправильный формат IP-адреса', $ip) ) );
            } elsif ( is_ip_in_list( $ip, $Settings::PRIVATE_NETWORKS ) ) {
                $vresults->add_generic( error_BadUsage( iget('Элемент %s списка #field# - нельзя запрещать IP-адреса из частных подсетей', $ip) ) );
            } elsif ( is_ip_in_list( $ip, $Settings::UNBLOCKABLE_IPS ) ) {
                $vresults->add_generic( error_BadUsage( iget('Элемент %s списка #field# - нельзя запрещать IP-адреса из подсети Яндекса (показы и клики с этих адресов не учитываются)', $ip) ) );
            }

            if ( $options{check_duplicates} && exists $seen_ips{ $ip } ) {
                $vresults->add_generic( warning_DuplicatedItems( iget('Элемент %s списка #field# - повторяющееся значение', $ip) ) );
            }

            $seen_ips{ $ip }++;
        }
    }

    return $vresults;
}

1;
