package Direct::Validation::Keywords;

use Direct::Modern;

use base qw(Exporter);

use Settings;

use Yandex::MyGoodWords;
use Yandex::ListUtils;
use Yandex::I18n;
use PhraseText;
use TextTools qw/get_word_for_digit/;
use List::MoreUtils qw/ any each_array uniq /;
use MinusWords;

use Currencies qw/get_currency_constant/;
use Currency::Format qw/format_const format_currency format_const_millis/;

use Direct::Validation::Errors;
use Direct::Validation::MinusWords;
use Direct::ValidationResult;

our @EXPORT = qw/
    validate_keywords_for_adgroup
    validate_add_keywords
    validate_add_keywords_oldstyle

    validate_advq_keywords
    validate_forecast_keywords
    base_validate_keywords
    validate_keywords_for_api_forecasts
/;

our $MAX_KEYWORD_ERROR_LENGTH = 200;
our $MAX_WORD_LENGTH = 35; # size of bids.norm_phrase / $Settings::MAX_WORDS_IN_KEYPHRASE => 255 / 7 ~ 35

=head2 validate_keywords_for_adgroup($keywords, $adgroup, $campaign)

Валидация полного набора ключевых фраз на группу.

Параметры:
    $checked_keywords   -> набор изменившихся ключевых фраз
    $remaining_keywords -> набор не изменившихся ключевых фраз
    $adgroup            -> общая группа для $keywords (Direct::Model::AdGroup)
    $campaign           -> общая кампания для $keywords (HashRef)
    $client             -> Direct::Model::Client для персональных лимитов

=cut

sub validate_keywords_for_adgroup {
    my ($checked_keywords, $remaining_keywords, $adgroup, $campaign, $client) = @_;
    $remaining_keywords //= [];

    my $vr_main = _validate_keywords_only($checked_keywords, $campaign);

    my $keywords_limit = $client->keywords_limit;
    my $keywords_count = @$checked_keywords + @$remaining_keywords;

    if ($keywords_count > $keywords_limit) {
        $vr_main->add_generic(
            error_LimitExceeded(iget('Группа объявлений может содержать не более %d ключевых фраз', $keywords_limit))
        );
    }

    return $vr_main;
}

=head2 validate_add_keywords($keywords, $adgroup, $campaign, $client, %options)

Валидация добавления ключевых фраз на группу.

Параметры:
    $keywords   -> массив ключевых фраз ([Direct::Model::Keyword])
    $adgroup    -> общая группа для $keywords (Direct::Model::AdGroup)
    $campaign   -> общая кампания для $keywords (HashRef)
    $client     -> Direct::Model::Client для персональных лимитов
    %options:
        keywords_count_to_delete -> число удаляемых ключевых фраз (используется при расчете ограничений на группу)
        остальные опции пробрасываются _validate_keywords_only

=cut

sub validate_add_keywords {
    my ($keywords, $adgroup, $campaign, $client, %options) = @_;

    my $to_delete_count = exists $options{keywords_count_to_delete} ? delete  $options{keywords_count_to_delete} : 0;
    my $vr_main = _validate_keywords_only($keywords, $campaign, %options);

    my $keywords_limit = $client->keywords_limit;
    my $keywords_count_to_delete = $to_delete_count;

    if ($adgroup->keywords_count - $keywords_count_to_delete + @$keywords > $keywords_limit) {
        $vr_main->add_generic(
            error_ReachLimit(iget('Группа объявлений может содержать не более %d ключевых слов', $keywords_limit))
        );
    }

    return $vr_main;
}

=head2 validate_add_keywords_oldstyle($phrases)
    Валидация добавления ключевых фраз в старом стиле, когда КФ передаются хешом, а не моделями

    Параметры:
         $phrases - массив ключевых фраз, [{phrase => 'ключевая фараза -слово'}]
                    элемент массива - хеш с обязательным ключом phrase
    Результат:
        $validation_result - результат валидации (объект Direct::ValidationResult)
                    ошибки валидации записываются пообъектно, поле содержащее ошибки - keyword
=cut

sub validate_add_keywords_oldstyle {
    my $phrases = shift;
    return base_validate_keywords([map {$_->{phrase}} @$phrases]);
}

=head2 _validate_keywords_only($keywords, $campaign, %options)

    Валидация только самих объектов фраз, без валидации связанных объектов

    Параметры:
        $keywords   -> массив ключевых фраз ([Direct::Model::Keyword])
        $campaign   -> общая кампания для $keywords (HashRef)
        %options:
            deny_quotes     -> запретить использование кавычек
            advq_phrase     -> валидация ключевых слов для запроса статистики по ADVQ
            brackets_errors -> не выдавать ошибку о наличии запрещенных скобок, например, т.к. в фразе были ошибки в использовании скобок
            sum_in_millis   -> суммы возвращаются целыми числами умноженными на миллион (как представляются в API)

=cut

sub _validate_keywords_only {
    my ($keywords, $campaign, %options) = @_;

    my $vr_main = Direct::ValidationResult->new();

    for my $keyword (@$keywords) {
        my $vr = $vr_main->next;

        $vr->add(text => _validate_keyword_text($keyword->text, %options));

        #
        # Валидация ставок
        #
        # price ("Цена на поиске")
        # price_context ("Цена на сети")
        # autobudget_priority ("Приоритет автобюджета")
        #
        # Цены валидируем только при наличии стратегии (ее может не быть, если валидируется снапшот данных без привязки к кампании)
        #
        if (exists $campaign->{strategy}) {
            my $is_autobudget = $campaign->{strategy}->{is_autobudget};
            my $is_search_stop = $campaign->{strategy}->{is_search_stop};
            my $is_net_stop = $campaign->{strategy}->{is_net_stop};
            my $is_different_places = $campaign->{strategy}->{name} eq 'different_places';
            my $is_cpm_banner_campaign = $campaign->{type} eq 'cpm_banner' ? 1 : 0;
            my $is_cpm_deals_campaign = $campaign->{type} eq 'cpm_deals' ? 1 : 0;

            if (!$is_autobudget) {
                my $currency = $campaign->{currency};
                my $constant_min = ($is_cpm_banner_campaign || $is_cpm_deals_campaign) ? 'MIN_CPM_PRICE' : 'MIN_PRICE';
                my $constant_max = ($is_cpm_banner_campaign || $is_cpm_deals_campaign) ? 'MAX_CPM_PRICE' : 'MAX_PRICE';
                my $min_price = get_currency_constant($currency, $constant_min);
                my $max_price = get_currency_constant($currency, $constant_max);

                for my $price_info (
                    {name => 'price',         is_required => !$is_search_stop},
                    {name => 'price_context', is_required => $is_different_places && !$is_net_stop},
                ) {
                    my $price = $price_info->{name};
                    my $has_price = "has_${price}";

                    if ($price_info->{is_required} && !$keyword->$has_price) {
                        $vr->add($price => error_ReqField());
                        next;
                    }

                    next if !$keyword->$has_price || ($keyword->$price == 0 && !$price_info->{is_required});

                    $vr->add($price => error_InvalidField(
                        iget('Значение ставки в поле #field# для валюты "%s" должно быть не меньше %s',
                            format_currency($currency, {full => 1}),
                            $options{sum_in_millis} ? format_const_millis($currency, $constant_min) : format_const($currency, $constant_min))
                    )) if $keyword->$price < $min_price;

                    $vr->add($price => error_InvalidField(
                        iget('Значение ставки в поле #field# для валюты "%s" должно быть не больше %s',
                            format_currency($currency, {full => 1}),
                            $options{sum_in_millis} ? format_const_millis($currency, $constant_max) : format_const($currency, $constant_max))
                    )) if $keyword->$price > $max_price;
                }
            } else {
                my $autobudget_priority = $keyword->has_autobudget_priority ? $keyword->autobudget_priority : undef;

                if (!defined $autobudget_priority) {
                    $vr->add(autobudget_priority => error_ReqField());
                } elsif ($autobudget_priority !~ /^(?:1|3|5)$/) {
                    $vr->add(autobudget_priority => error_InvalidField());
                }
            }
        }

        #
        # Валидация подстановочных параметров (href_param1/href_param2)
        #
        for my $param_name (qw/href_param1 href_param2/) {
            my $has_param = "has_${param_name}";
            my $param_value = $keyword->$has_param ? $keyword->$param_name : undef;

            next if !defined $param_value;

            if (length($param_value) > $Settings::MAX_HREF_PARAM_LENGTH) {
                $vr->add($param_name => error_MaxLength());
            }

            if ($param_value =~ $Settings::DISALLOW_BANNER_LETTER_RE) {
                $vr->add($param_name => error_InvalidChars_AlphaNumPunct());
            }
        }
    }

    return $vr_main;
}

sub _validate_keyword_text {
    my ($phrase, %options) = @_;

    my @errors;

    # Если в фразе были ошибки в использовании скобок, то не выдаем ошибку о запрещенных символах "(", ")", т.к. уже есть ошибка о скобках
    # TODO: упразднить этот параметр (проверять на наличие недопустимых скобок всегда)
    my $ALLOW_LETTERS_LOCAL = $Settings::ALLOW_LETTERS;
    $ALLOW_LETTERS_LOCAL .= "\Q()|\E" if $options{brackets_errors};

    my $allow_letters_err = iget('В тексте ключевых фраз разрешается использовать только буквы английского, турецкого, казахского, русского, украинского или белорусского алфавита, %s. #phrase_extended#');

    if (!defined $phrase || $phrase !~ /\S/) {
        push @errors, error_ReqField();
        return \@errors;
    }

    # При подсчете длины фразы не учитываются `!` и `+` в минус-словах
    if (length($phrase =~ s/-[!+]/-/gr) > $Settings::MAX_PHRASE_LENGTH) {
        push @errors, error_MaxLength(iget('Превышена допустимая длина строки в %s символов в ключевой фразе #phrase#', $Settings::MAX_PHRASE_LENGTH));
    }

    if ($options{deny_quotes}) {
        if ($phrase !~ /^[${ALLOW_LETTERS_LOCAL}\- !\+]+$/ || $phrase =~ /''/) {
            push @errors, error_InvalidChars(sprintf($allow_letters_err, iget('знаки "-", "+", "!", пробел')));
        }
    } elsif ($options{advq_phrase}) {
        if ($phrase !~ /^[${Settings::ALLOW_LETTERS}\- \"!\+\|\(\)]+$/ || $phrase =~ /''/) {
            push @errors, error_InvalidChars(sprintf($allow_letters_err, iget('кавычки, круглые скобки, знаки "-", "+", "!", "|", пробел')));
        }
    } else {
        if ($phrase !~ /^[${ALLOW_LETTERS_LOCAL}\- \"!\+]+$/ || $phrase =~ /''/) {
            push @errors, error_InvalidChars(sprintf($allow_letters_err, iget('кавычки, знаки "-", "+", "!", пробел')));
        }
    }

    if ($phrase =~ /(?:^|[\s"])\.+(?:$|[\s"])/) {
        push @errors, error_InvalidChars(iget('Ключевая фраза не может содержать отдельно стоящие точки'));
    }

    if ($phrase =~ m/"/ && $phrase !~ m/^"[^"]+"$/) {
        push @errors, error_InvalidChars(iget('Неправильное использование кавычек в ключевой фразе #phrase#'));
    }

    my @all_words = uniq map { s/!//r } split(/\s+/, $phrase);
    # заменяем Ё на Е т.к. в списке стоп-слов слов с Ё нет, но такая логика есть в нормализаторе
    my (@plus_words, %original_words);
    for my $original (@all_words) {
        next if $original =~ /^\-/;
        $original =~ s/[\+\!\[\]\"]//g;
        my $res_word = $original;
        $res_word =~ tr/Ёё/Eе/;
        $original_words{$res_word} = $original;
        push @plus_words, $res_word;
    }
    if (@plus_words == 1 && Yandex::MyGoodWords::is_stopword($plus_words[0])) {
        push @errors, error_StopWords(iget('Ключевая фраза не может состоять только из стоп-слов: союзов, предлогов, частиц #phrase#'));
        return _postprocess_keyword_text_errors(\@errors, $phrase, %options);
    } elsif (!scalar @plus_words && scalar @all_words) {
        push @errors, error_MinusWords(iget('Ключевая фраза не может состоять только из минус-слов #phrase#'));
        return _postprocess_keyword_text_errors(\@errors, $phrase, %options);
    } elsif (my @too_long_words = grep { length > $MAX_WORD_LENGTH } @plus_words) {
        # NB: потенциально возможно превышение максимальной длины плюс-слова,
        # если при нормализации длина слова увеличится. Пренебрегаем такой
        # вероятностью, так как ограничение длины выбрано с запасом.
        my @too_long_orignal_words = @original_words{@too_long_words};
        push @errors, error_MaxKeywordLength(iget('Превышена допустимая длина отдельного ключевого слова в %s символов. Ошибки в словах: %s.', $MAX_WORD_LENGTH, join(', ' => @too_long_orignal_words)));
    }

    if (my @words = PhraseText::split_phrase_with_normalize($phrase)) {
        if (@words > $Settings::MAX_WORDS_IN_KEYPHRASE) {
            push @errors, error_MaxWords(iget('Ключевая фраза не может состоять более чем из %s слов #phrase#', $Settings::MAX_WORDS_IN_KEYPHRASE));
        }
    } else {
        push @errors, error_StopWords(iget('Ключевая фраза не может состоять только из стоп-слов: союзов, предлогов, частиц #phrase#'));
    }

    # Валидация квадратных скобок
    if (
        @{[$phrase =~ m/\[/g]} != @{[$phrase =~ m/\]/g]}
    ) {
        push @errors, error_InvalidChars(iget('Неправильное использование скобок [] в ключевой фразе #phrase#'));
    }
    if (
        $phrase =~ /\[\s*\]/ || @{[$phrase =~ m/\[([^\[\]]+)\]/g]} != @{[$phrase =~ m/\[/g]}
    ) {
        push @errors, error_InvalidChars(iget('В ключевой фразе #phrase# квадратные скобки [] не могут быть пустыми и вложенными'));
    }
    if (
        $phrase =~ /\[/ && ($phrase !~ /\[[^\+\"]+\]/ || !_validate_inside_brackets($phrase =~ m/\[([^\]]+)\]/g))
    ) {
        push @errors, error_InvalidChars(iget('В ключевой фразе #phrase# присутствуют недопустимые модификаторы +-"" внутри скобок []'));
    }

    if ($phrase =~ m/![\s.'"\-+!\[\]]/ || $phrase =~ m/!$/ || $phrase =~ m/[^-!\"\s\[]![^-!\s]/) {
        push @errors, error_InvalidChars(iget('Неправильное использование знака "!" в ключевой фразе #phrase#'));
    }
    if (
        $phrase =~ m/--/    || $phrase =~ m/-\s/      || $phrase =~ m/-$/  || $phrase =~ m/!-/ ||
        $phrase =~ m/\S-!/  || $phrase =~ m/[-!]{3,}/ || $phrase =~ m/\+-/ ||
        $phrase =~ m/\S-\+/ || $phrase =~ m/[-+]{3,}/
    ) {
        push @errors, error_InvalidChars(iget('Неправильное использование знака "-" в ключевой фразе #phrase#'));
    }
    if ($phrase =~ m/\+\+/ || $phrase =~ m/\+\s/ || $phrase =~ m/\+$/ || $phrase =~ m/[^-+\"\s]\+[^-+\s]/) {
        push @errors, error_InvalidChars(iget('Неправильное использование знака "+" в ключевой фразе #phrase#'));
    }
    if ($phrase =~ /(^|\s)\-.*?\s[^\-\s]/ || $phrase =~ /(^|\s)\-.*?\S-/) {
        push @errors, error_MinusWords(iget('Из ключевой фразы могут вычитаться только отдельные слова, а не словосочетания #phrase#'));
    }

    my @minus_words = map { s/^-//r } grep { /^-/ } (split /\s+/, $phrase);
    # Содержит точку, но не число.число
    # Кажется это условие никогда не должно срабатывать из-за предварительных склеек и чисток минус-слов.
    if (any { m/\./ && /[^0-9\.\!\+]/ } @minus_words) {
        push @errors, error_MinusWords(iget('Из ключевой фразы нельзя вычитать словосочетания, содержащие точку #phrase#'));
    }

    if ($phrase =~ m/[\.\[\]\'\-\+\!]{2,}/ && $phrase !~ m/-[!+]/ && $phrase !~ m/\[!/) {
        push @errors, error_MinusWords(iget('Неправильное сочетание специальных символов в ключевой фразе #phrase#'));
    }

    my $word_start_re = qr/^|[\s"\.\[\]\'\-\+\!]|-[!+]|\[!/;
    if ($phrase =~ /(?:$word_start_re)[\.\']/) {
        push @errors, error_MinusWords(iget('Слова не могут начинаться с точек и апострофов #phrase#'));
    }

    if ($phrase =~ m/^".+"$/ && $phrase =~ m/(?:^\"|\s)-/) {
        push @errors, error_MinusWords(iget('Словосочетание #phrase# в кавычках не может состоять из минус-слов'));
    }

    my $minus_words_vr = validate_keyword_minus_words(\@minus_words);
    if (!$minus_words_vr->is_valid) {
        push @errors, @{$minus_words_vr->get_errors};
    }

    # Учет словоформ при вычитании фраз
    # Ставим пробел между минус-словами, записанными, через дефис
    my $modified_phrase = $phrase =~ s/(\s\-[^\-\s]+)\-(?=[^\s])/$1 \-/gr;
    # Заменяем дефис между словами на пробел
    $modified_phrase =~ s/(?<=[^\s])\-(?=[^\s])/ /g;

    # Проверяем, не вычитаются ли ключевые слова, кроме стоп-слов.
    # "кредит авто -авто" -- нельзя, но "кредит в авто -в" -- можно

    my @mass_not_valid = @{PhraseText::validate_key_phrase($modified_phrase)};
    my $bad_word = join ", ", xuniq { $_ } @mass_not_valid;

    if ($bad_word) {
        push @errors, error_MinusWords(
            get_word_for_digit(scalar @mass_not_valid,
                iget('Нельзя вычитать слово - (%s), содержащееся в исходной ключевой фразе #phrase#', $bad_word),
                iget('Нельзя вычитать слова - (%s), содержащиеся в исходной ключевой фразе #phrase#', $bad_word),
                iget('Нельзя вычитать слова - (%s), содержащиеся в исходной ключевой фразе #phrase#', $bad_word),
            )
        );
    }

    _postprocess_keyword_text_errors(\@errors, $phrase, %options);
    return @errors ? \@errors : undef;
}

sub _postprocess_keyword_text_errors {
    my ($errors, $phrase, %options) = @_;

    if (defined $phrase) {
        # для обратной совместимости подставляем в ошибки сразу текст фразы, по которой собственно выявлена ошибка
        unless ($options{not_process_phrase_text_placeholder_in_errors}) {
            my $truncated_phrase = TextTools::truncate_text($phrase, $MAX_KEYWORD_ERROR_LENGTH);
            for my $err (@$errors) {
                $err->description(TextTools::process_text_template($err->description, phrase => qq/"$truncated_phrase"/,
                                                                                      phrase_extended => iget('Ошибка в ключевой фразе "%s"', $truncated_phrase)));
            }
        }
    }

    return $errors;
}

sub _validate_inside_brackets { for (@_) { return 0 if /(?<!\S)\-/; } 1; }

=head2 validate_advq_keywords($phrases)

Валидация ключевых слов для запроса статистики по ADVQ

Параметры:
    $phrases - массив текстов ключевых фраз, ['ключевая фараза -слово', 'двери межкомнатные -москва']

Результат:
    $validation_result - результат валидации (объект Direct::ValidationResult)
                            ошибки валидации записываются по объектно, поле содержащее ошибки - keyword

=cut

sub validate_advq_keywords {
    my $phrases = shift;
    return base_validate_keywords($phrases, {advq_phrase => 1});
}

=head2 validate_forecast_keywords($phrases)

Валидация ключевых фраз для прогноза бюджета

Параметры:
    $phrases - массив текстов ключевых фраз, ['ключевая фараза -слово', 'двери межкомнатные -москва']

Результат:
    $validation_result - результат валидации (объект Direct::ValidationResult)
                            ошибки валидации записываются по объектно, поле содержащее ошибки - keyword

=cut

sub validate_forecast_keywords {
    my $phrases = shift;
    return base_validate_keywords($phrases);
}

=head2 base_validate_keywords

    TODO: выпилить, перейти на _validate_keywords_only

=cut

sub base_validate_keywords {
    my ($phrases, $options) = @_;

    my $vr_main = Direct::ValidationResult->new();

    for my $phrase (@$phrases) {
        my $phrase_vr = $vr_main->next;

        $phrase_vr->add(keyword => _validate_keyword_text($phrase, %$options));
    }

    return $vr_main;
}

=head2 validate_keywords_for_api_forecasts(\@params)
    
    Валидация ключевых фраз с минус-фразами
    
    Параметры:
        @params - массив ключевых фраз
    
    Результат:
        $validation_result - результат валидации (объект Direct::ValidationResult)
                                ошибки валидации записываются по объектно, поле содержащее ошибки - keyword

=cut

sub validate_keywords_for_api_forecasts {
    my @phrases = @{ $_[0] };
    my $keywords_vr;
    my @phrases_without_minus_phrases = @phrases;
    if ( any { $_ =~ /-\(.+?\)/ } @phrases_without_minus_phrases ) {
        map {$_ =~ s/(\s-\(.+?\))//g} @phrases_without_minus_phrases;

        $keywords_vr = validate_advq_keywords(\@phrases_without_minus_phrases);
            
        for (my $i=0; $i <= $#phrases; $i++) {
            my $phrase = $phrases[$i];
            my $keyword_vr = $keywords_vr->get_objects_results->[$i];
            my @minus_phrases = $phrase =~ /-\(?(.+?)\)?\s(?=-)/g;
            my $minus_phrases_vr = validate_group_minus_words(\@minus_phrases, MinusWords::polish_minus_words_array(\@minus_phrases));
            $keyword_vr->add(text => $minus_phrases_vr->get_errors);
        }       
    } else {
        $keywords_vr = validate_advq_keywords(\@phrases);
    }
    return $keywords_vr;
}

1;
