
package Direct::Validation::MinusWords;
## no critic (TestingAndDebugging::RequireUseStrict, TestingAndDebugging::RequireUseWarnings)

use base qw(Exporter);
our @EXPORT = qw/
    validate_keyword_minus_words
    validate_group_minus_words
    validate_campaign_minus_words
/;

use strict;
use warnings;
use utf8;
use feature qw/state/;

use Direct::Validation::Errors;
use Direct::ValidationResult;
use Settings;
use PhraseText;
use MinusWordsTools;

use Yandex::I18n;
use List::MoreUtils qw/any uniq/;

our $MAX_MINUS_WORD_LENGTH = 35; # also see $Direct::Validation::Keywords::MAX_WORD_LENGTH

our %context;

=head2 validate_keyword_minus_words($minus_words)
    
    Валидация минус слов в ключевой фразе.
    Ключвые фразы не поддерживают минус-фразы, а только минус-слова, поэтому используется старая валидация,
    отличная от минус-фраз на группу или кампанию.
    
    Параметры:
        $minus_words - массив текстов минус слов, ['минус', 'слово', ....]
    
    Результат:
        $validation_result - результат валидации (объект Direct::ValidationResult)
                                ошибки валидации содержаться в generic_errors

=cut

sub validate_keyword_minus_words {

    my $minus_words = shift;

    my $validation_result = new Direct::ValidationResult();

    my $non_count_spec_symbols_re = qr/[!+\"\[\]]/;

    my (@several_minus_words, @wrong_minus_words, @multipoint_minus_words, @too_long_minus_words);
    for my $minus_word (@$minus_words) {
        # Удалим пробелы в начале и в конце строки
        $minus_word =~ s/^\s+//;
        $minus_word =~ s/\s+$//;

        next unless length $minus_word;

        if ($minus_word =~ /\s/) {
            push @several_minus_words, $minus_word;
        } elsif ($minus_word !~ /^[\!\+]?[$Settings::ALLOW_MINUS_WORD_LETTERS]+$/) {
            push @wrong_minus_words, $minus_word;
        } elsif ($minus_word =~ /^[0-9\.\!\+]+$/ && scalar(grep { length $_ } split /\./, $minus_word) > 2) {
            push @multipoint_minus_words, $minus_word;
        } elsif (length($minus_word =~ s/$non_count_spec_symbols_re//gr) > $MAX_MINUS_WORD_LENGTH) {

            push @too_long_minus_words, $minus_word;
        }
    }

    $validation_result->add_generic(
        error_HoldPhrase(iget('Минус-слово не должно являться словосочетанием. Ошибки в словах: %s.', join(', ', @several_minus_words)))
    ) if @several_minus_words;

    $validation_result->add_generic(
        error_InvalidChars(iget('Минус-слово должно состоять из букв английского, руcского, украинского или белорусского алфавитов, цифр, в начале может быть указан один из символов: "+", "!". Ошибки в словах: %s.', join(', ', @wrong_minus_words)))
    ) if @wrong_minus_words;

    $validation_result->add_generic(
        error_MinusWords(iget('Минус-слово может содержать не более двух цифр подряд через точку. Ошибки в словах: %s.', join(', ', @multipoint_minus_words)))
    ) if @multipoint_minus_words;

    $validation_result->add_generic(
        error_MaxMinusWordLength(iget('Превышена допустимая длина отдельного минус-слова в %s символов. Ошибки в словах: %s.', $MAX_MINUS_WORD_LENGTH, join(', ', @too_long_minus_words)))
    ) if @too_long_minus_words;

    return $validation_result;
}

=head2 validate_group_minus_words($minus_words, $polished_minus_phrases)
    
    Валидация общих минус слов на группу объявлений
    
    Параметры:
        $minus_words - массив текстов минус слов, ['минус', 'слово', ....]
    
    Результат:
        $validation_result - результат валидации (объект Direct::ValidationResult)
                                ошибки валидации содержаться в generic_errors

=cut

sub validate_group_minus_words {

    my ($minus_words, $polished_minus_phrases, %opt) = @_;
    return base_validate_minus_words($minus_words, polished_minus_phrases => $polished_minus_phrases, %opt);
}

=head2 validate_campaign_minus_words($minus_words, $polished_minus_phrases)
    
    Валидация общих минус слов на кампанию
    
    Параметры:
        $minus_words - массив текстов минус слов, ['минус', 'слово', ....]
    
    Результат:
        $validation_result - результат валидации (объект Direct::ValidationResult)
                                ошибки валидации содержаться в generic_errors

=cut

sub validate_campaign_minus_words {

    my ($minus_words, $polished_minus_phrases) = @_;
    return base_validate_minus_words($minus_words, type => 'campaign', polished_minus_phrases => $polished_minus_phrases);
}

sub base_validate_minus_words {

    my $minus_phrases = shift;
    my %options = @_;

    # лимиты на разные виды минус-слов. -1 означает "нет ограничений"
    # не может быть my т.к. на переменные из Settings мы иногда делаем local $Settings::SOME_CONST = 'some_var'
    my $LIMIT = {
        campaign => $Settings::CAMPAIGN_MINUS_WORDS_LIMIT,
        default => $Settings::GROUP_MINUS_WORDS_LIMIT,
        phrase => -1,
    };
    my $MAX_MINUS_PHRASE_ERROR_LENGTH = 200;

    my $validation_result = new Direct::ValidationResult();

    my %errors;

    # Проверим длину минус-слов без учета возможных спецсимволов
    my $limit = $options{max_overall_length} // $LIMIT->{ $options{type} || 'default' } // $LIMIT->{default};
    my $limit_calc_minus_words = MinusWordsTools::minus_words_length_for_limit($minus_phrases, $options{polished_minus_phrases}); 
    if ($limit >= 0 && $limit_calc_minus_words > $limit) { 
        push @{$errors{max_length_all_phrase}}, $limit;
    }
    for my $minus_phrase (@$minus_phrases) {
        next unless length $minus_phrase;
        my $truncated_phrase = TextTools::truncate_text($minus_phrase, $MAX_MINUS_PHRASE_ERROR_LENGTH);

        foreach my $error_code (_validate_minus_phrase_text($minus_phrase)) {
            push @{$errors{$error_code}}, $truncated_phrase;
        }
    }

    foreach my $error (uniq keys %errors) {
        my $error_phrases_str = join(', ', @{$errors{$error}});


        $validation_result->add_generic(
            error_MaxLength(iget('Длина минус-фраз превышает %s символов.', $error_phrases_str))
        ) if $error eq 'max_length_all_phrase';

        $validation_result->add_generic(
            error_MaxLength(iget('Превышена допустимая длина строк в %s символов в минус-фразах "%s"', $Settings::MAX_MINUS_WORDS_LENGTH, $error_phrases_str))
        ) if $error eq 'max_length';

        $validation_result->add_generic(
            error_InvalidChars(iget('В минус-фразах разрешается использовать только буквы английского, турецкого, казахского, русского, украинского или белорусского алфавита, кавычки, квадратные скобки, знаки "-", "+", "!", пробел. Ошибки во фразах: %s', $error_phrases_str))
        ) if $error eq 'chars_base';

        $validation_result->add_generic(
            error_InvalidChars(iget('Минус-фразы могут содержать не более двух цифр подряд через точку. Ошибки во фразах: %s', $error_phrases_str))
        ) if $error eq 'multipoints';

        $validation_result->add_generic(
            error_MaxMinusWordLength(iget('Превышена допустимая длина слова в минус-фразе в %s символов. Ошибки во фразах: %s', $MAX_MINUS_WORD_LENGTH, $error_phrases_str))
        ) if $error eq 'too_long';

        $validation_result->add_generic(
            error_InvalidChars(iget('Минус-фразы не могут содержать отдельно стоящие точки: %s', $error_phrases_str))
        ) if $error eq 'chars_base_dots';

        $validation_result->add_generic(
            error_InvalidChars(iget('Неправильное использование кавычек в минус-фразах: %s', $error_phrases_str))
        ) if $error eq 'chars_base_quotes';

        $validation_result->add_generic(
            error_InvalidChars(iget('Минус-фраза не может состоять более чем из %s слов: %s', $Settings::MAX_WORDS_IN_MINUSPHRASE, $error_phrases_str))
        ) if $error eq 'too_many_words';

        $validation_result->add_generic(
            error_InvalidChars(iget('Неправильное использование скобок [] в минус-фразах: %s', $error_phrases_str))
        ) if $error eq 'square_brackets';

        $validation_result->add_generic(
            error_InvalidChars(iget('Квадратные скобки [] не могут быть пустыми и вложенными, минус-фразы: %s', $error_phrases_str))
        ) if $error eq 'square_brackets_empty';

        $validation_result->add_generic(
            error_InvalidChars(iget('Внутри скобок [] недопустимы символы +-"", минус-фразы: %s', $error_phrases_str))
        ) if $error eq 'square_brackets_mods';

        $validation_result->add_generic(
            error_InvalidChars(iget('Неправильное использование знака "!" в минус-фразах: %s', $error_phrases_str))
        ) if $error eq 'exclamation_mark';

        $validation_result->add_generic(
            error_InvalidChars(iget('Неправильное использование знака "-" в минус-фразах: %s. Знак "-" нельзя использовать в начале и в конце слов', $error_phrases_str))
        ) if $error eq 'start_finish_minus';

        $validation_result->add_generic(
            error_InvalidChars(iget('Неправильное использование знака "+" в минус-фразах: %s', $error_phrases_str))
        ) if $error eq 'chars_plus';

        $validation_result->add_generic(
            error_InvalidChars(iget('Неправильное сочетание специальных символов в минус-фразах: %s', $error_phrases_str))
        ) if $error eq 'chars_spec';

        $validation_result->add_generic(
            error_InvalidChars(iget('Слова не могут начинаться с точек и апострофов: %s', $error_phrases_str))
        ) if $error eq 'start_dots';

    }

    return $validation_result;
}

sub _validate_minus_phrase_text {
    my ($phrase, %options) = @_;
    my @errors;

    # При подсчете длины фразы не учитываются `!` и `+` в минус-словах

    my $word_start_re = qr/^|[\s"\.\[\]\'\-\+\!]|-[!+]|\[!/;
    my $word_finish_re = qr/[\s"\.\[\]\'\-\+\!]|$/;
    my $non_count_spec_symbols_re = qr/[!+\"\[\]]/;

    if ($phrase !~ /^[$Settings::ALLOW_MINUS_PHRASE_LETTERS \"\!\+\-]+$/ || $phrase =~ /''/) {
        # В минус-фразах разрешается использовать только буквы английского, турецкого, казахского, русского или украинского алфавита, кавычки, квадратные скобки, знаки "-", "+", "!", пробел
        push @errors, 'chars_base';
    
    } elsif (length($phrase =~ s/$non_count_spec_symbols_re//gr) > $Settings::MAX_MINUS_WORDS_LENGTH) {
        # Превышена допустимая длина строк в %s символов в минус-фразах
        push @errors, 'max_length';

    } elsif ($phrase =~ /(?:^|[\s"])\.+(?:$|[\s"])/) {
        # Минус-фразы не могут содержать отдельно стоящие точки
        push @errors, 'chars_base_dots';

    } elsif ($phrase =~ m/"/ && $phrase !~ m/^"[^"]+"$/) {
        # Неправильное использование кавычек в минус-фразах
        push @errors, 'chars_base_quotes';

    } elsif ( scalar (PhraseText::split_phrase_with_normalize($phrase, preserve_stopwords => 1)) > $Settings::MAX_WORDS_IN_MINUSPHRASE) {
        # Минус-фраза не может состоять более чем из %s слов
        push @errors, 'too_many_words';

    } elsif ($phrase =~ /[0-9\.\!\+]+/ && scalar(grep { length $_ } split /\./, $phrase) > 2) {
        # Минус-фразы могут содержать не более двух цифр подряд через точку.
        push @errors, 'multipoints';

    } elsif (any {length($_ =~ s/$non_count_spec_symbols_re//gr) > $MAX_MINUS_WORD_LENGTH} split /\s+/, $phrase ) {
        # Превышена допустимая длина слова в минус-фразе в %s символов
        push @errors, 'too_long';

    } elsif ( @{[$phrase =~ m/\[/g]} != @{[$phrase =~ m/\]/g]} ) {
        # Неправильное использование скобок [] в минус-фразах
        push @errors, 'square_brackets';

    } elsif ( $phrase =~ /\[\s*\]/ || @{[$phrase =~ m/\[([^\[\]]+)\]/g]} != @{[$phrase =~ m/\[/g]} ) {
        # Квадратные скобки [] не могут быть пустыми и вложенными
        push @errors, 'square_brackets_empty';

    } elsif ( $phrase =~ /\[/ && ($phrase !~ /\[[^\+\"]+\]/ || !_validate_inside_brackets($phrase =~ m/\[([^\]]+)\]/g)) ) {
        # Внутри скобок [] недопустимы символы +-"", минус-фразы
        push @errors, 'square_brackets_mods';

    } elsif ( $phrase =~ m/!!/ || $phrase =~ m/[^-!\"\s\[]![^-!\s]/ || $phrase =~ m/\!(?:$word_finish_re)/) {
        # Неправильное использование знака "!" в минус-фразах
        push @errors, 'exclamation_mark';

    } elsif ($phrase =~ /(?:$word_start_re)\-|\-(?:$word_finish_re)/) {
        # Неправильное использование знака "-" в минус-фразах
        push @errors, 'start_finish_minus';
 
    } elsif ( $phrase =~ m/\+\+/ || $phrase =~ m/[^-+\"\s]\+[^-+\s]/ || $phrase =~ m/\+(?:$word_finish_re)/) {
        # Неправильное использование знака "+" в минус-фразах
        push @errors, 'chars_plus';

    } elsif ($phrase =~ m/[\.\[\]\'\-\+\!]{2,}/ && $phrase !~ m/-[!+]/ && $phrase !~ m/\[!/) {
        # Неправильное сочетание специальных символов в минус-фразах
        push @errors, 'chars_spec';
    
    } elsif ($phrase =~ /(?:$word_start_re)[\.\']/) {
        # Слова не могут начинаться с точек и апострофов
        push @errors, 'start_dots';

    }

    return @errors;
}

sub _validate_inside_brackets { for (@_) { return 0 if /(?<!\S)\-/; } 1; }

1;
