package Direct::Validation::PerformanceFilters;

use Direct::Modern;

use base qw(Exporter);

use Settings;

use Encode;
use Yandex::I18n;
use Yandex::HashUtils qw/hash_cut/;
use List::Util qw/first/;

use Currencies qw/get_currency_constant/;
use Currency::Format qw/format_const/;
use FilterSchema;
use Retargeting qw//;
use PrimitivesIds qw/get_clientid/;

use Direct::Errors::Messages;
use Direct::ValidationResult;
use Direct::Feeds;

our @EXPORT_OK = qw/
    validate_performance_filters_for_adgroup
/;

# Максимальное число фильтров в группе
our $MAX_FILTERS_IN_ADGROUP ||= 50;

# Максимальное число правил в фильтре
our $MAX_RULES_PER_FILTER ||= 10;

# Максимальная длина названия фильтра
our $MAX_FILTER_NAME_LENGTH ||= 100;

# Разрешенные символы для использования в текстовых полях
our $DISALLOW_LETTER_RE = qr/[^\Q${Settings::ALLOW_BANNER_LETTERS_STR}|\E]/i;

=head2 validate_performance_filters_for_adgroup($filters, $campaign, %options)

    my $validation_result = validate_performance_filters_for_adgroup([$filter_1, ..., $filter_N], $campaign, %options)

%options:
    for_estimation  ->  булево значение, показывающее что фильтр будет использоваться только с целью оценки
                        количество попадающих под него товарных позиций (некоторые поля становятся необязательными).

=cut

sub validate_performance_filters_for_adgroup {
    my ($filters, $campaign, $adgroup, %options) = @_;

    my $vr_main = validate_performance_filters($filters, $campaign, $adgroup, %{hash_cut \%options, qw/for_estimation/});

    my $filters_count = @$filters;

    if ($filters_count > $MAX_FILTERS_IN_ADGROUP) {
        $vr_main->add_generic(error_LimitExceeded(
            iget("Количество фильтров в группе не может быть больше %s", $MAX_FILTERS_IN_ADGROUP)
        ));
    }

    return $vr_main;
}

=head2 validate_performance_filters($filters, $campaign, %options)

Валидация списка перфоманс фильтров в пределах одной кампании.

=cut

sub validate_performance_filters {
    my ($filters, $campaign, $adgroup, %options) = @_;

    my $vr_main = Direct::ValidationResult->new();
    my $client_id = get_clientid(cid => $campaign->{cid});
    my $exists_retargetings = Retargeting::are_retargeting_conditions_exists($client_id, [grep { $_ } map { $_->ret_cond_id } @$filters]);
    my (%filters_uhash, %duplicated_filters);
    my $i;
    foreach my $filter (@$filters){
        $i++;
        next unless $filter->has_condition;
        my $uhash = $filter->get_condition_uhash();
        if (defined $filters_uhash{$uhash}){
            $duplicated_filters{$i} = 1;
            $duplicated_filters{$filters_uhash{$uhash}} //= 1;
        }
        else {
            $filters_uhash{$uhash} = $i;
        }
    }

    $i = 0;
    for my $filter (@$filters) {
        my $vr = $vr_main->next;
        $i++;

        #
        # filter_name ("Название фильтра")
        #
        if (!$options{for_estimation}) {
            my $filter_name = $filter->has_filter_name ? $filter->filter_name : undef;
            if (!defined $filter_name || $filter_name !~ /\S/) {
                $vr->add(filter_name => error_ReqField());
            } else {
                if ($filter_name =~ $DISALLOW_LETTER_RE) {
                    $vr->add(filter_name => error_InvalidChars_AlphaNumPunct());
                }
                if (length($filter_name) > $MAX_FILTER_NAME_LENGTH) {
                    $vr->add(filter_name => error_MaxLength(undef, length => $MAX_FILTER_NAME_LENGTH));
                }
            }
        }

        #
        # condition ("Правила фильтра")
        #
        if (!$filter->has_condition) {
            $vr->add(condition => error_ReqField(iget("Необходимо задать список правил для фильтра")));
        } elsif ( $duplicated_filters{$i} ){
            $vr->add_generic(error_Duplicated(iget("Фильтры в группе объявлений должны быть уникальны")));
        }
        else {
            my $length = length(encode(utf8 => $filter->_condition_json));
            if ( $length > $Settings::MAX_CONDITION_JSON_LENGTH ) {
                $vr->add(
                    condition => error_MaxLength(iget("Превышен максимально допустимый размер списка условий (%s байт при лимите в %s байт)",
                            $length, $Settings::MAX_CONDITION_JSON_LENGTH)
                ));
            } else {
                # все товары
                next unless scalar @{$filter->condition};

                my $feed = Direct::Feeds->get_by($client_id, id => $adgroup->feed_id)->items()->[0];
                unless ($feed) {
                    die "feed id = @{[$adgroup->feed_id]} not found";
                }
                my $filter_type = $feed->business_type . '_' . $feed->feed_type;
                my $rules_vr = validate_performance_filter_rules($filter->condition, $filter_type);
                $vr->add(condition => $rules_vr) if !$rules_vr->is_valid || $rules_vr->has_only_warnings;
            }

        }

        #
        # price_cpc/price_cpa ("Цены CPC/CPA")
        # autobudget_priority ("Приоритет автобюджета")
        #
        # Стратегия на кампании должны быть всегда.
        #
        if (exists $campaign->{strategy}) {
            my $strategy = $campaign->{strategy};
            my $check_prices = $strategy->{name} ne 'autobudget' && $strategy->{name} ne 'autobudget_roi' && $strategy->{name} ne 'autobudget_crr';

            if ($check_prices) {
                my $currency = $campaign->{currency};
                my $min_price = get_currency_constant($currency, 'MIN_CPC_CPA_PERFORMANCE');
                my $max_price = get_currency_constant($currency, 'MAX_PRICE');
                
                my $filter_avg_bid = $strategy->{search}->{filter_avg_bid} || $strategy->{net}->{filter_avg_bid};
                my $filter_avg_cpa = $strategy->{search}->{filter_avg_cpa} || $strategy->{net}->{filter_avg_cpa};
                for my $price_info (
                    {name => 'price_cpc', is_required => $strategy->{name} =~ /_cpc_per_filter$/ && !$filter_avg_bid},
                    {name => 'price_cpa', is_required => $strategy->{name} =~ /_cpa_per_filter$/ && !$filter_avg_cpa},
                ) {
                    my $price = $price_info->{name};
                    my $has_price = "has_${price}";

                    if ($price_info->{is_required} && !$filter->$has_price) {
                        $vr->add($price => error_ReqField());
                        next;
                    }

                    next if !$filter->$has_price || ($filter->$price == 0 && !$price_info->{is_required});

                    $vr->add($price => error_InvalidField(
                        iget("Значение в поле #field# не может быть меньше %s", format_const($currency, 'MIN_PRICE'))
                    )) if $filter->$price < $min_price;
                    $vr->add($price => error_InvalidField(
                        iget("Значение в поле #field# не может быть больше %s", format_const($currency, 'MAX_PRICE'))
                    )) if $filter->$price > $max_price;
                }
            } else {
                my $autobudget_priority = $filter->has_autobudget_priority ? $filter->autobudget_priority : undef;

                if (!defined $autobudget_priority) {
                    $vr->add(autobudget_priority => error_ReqField());
                } elsif ($autobudget_priority !~ /^(?:1|3|5)$/) {
                    $vr->add(autobudget_priority => error_InvalidField());
                }
            }
        } else {
            croak "Cannot find strategy for campaign";
        }

        #
        # target_funnel ("Воронка целей")
        #
        if (!$filter->has_target_funnel) {
            $vr->add(target_funnel => error_ReqField());
        }

        #
        # retargeting
        #
        if (defined $filter->ret_cond_id) {
            if (!exists $exists_retargetings->{$filter->ret_cond_id}) { 
                $vr->add(retargeting => error_NotFound_RetargetingConditionId(undef, id => $filter->ret_cond_id));
            } elsif ($filter->has_target_funnel && $filter->target_funnel ne 'same_products') {
                $vr->add(retargeting => error_BadUsage(iget('Условие подбора можно задавать только для целевой аудитории "Обе группы"')));
            }
        }
    }

    return $vr_main;
}

=head2 validate_performance_filter_rules($rules)

Валидация правил (поле `condition`) одного фильтра.

=cut

sub validate_performance_filter_rules {
    my ($rules, $filter_type) = @_;

    my $validator = FilterSchema->new(filter_type => $filter_type);

    return $validator->check([ map {$_->to_hash} @$rules ]);
}

1;
