package Direct::Validation::RetargetingConditions;

use Direct::Modern;

use base qw(Exporter);

use Settings;

use Yandex::I18n;
use List::MoreUtils qw/all any/;

use Direct::Errors::Messages;
use Direct::ValidationResult;

our @EXPORT_OK = qw/
    validate_retargeting_conditions
    validate_retargeting_conditions_for_client
/;

# Максимальное количество условий ретаргетинга на клиента
our $MAX_RET_CONDS_ON_CLIENT ||= $Settings::MAX_RETARGETINGS_ON_CLIENT;

# Максимальное число правил (групп) в условии ретаргетинга
our $MAX_RULES_IN_RET_COND ||= $Settings::MAX_RETARGETING_GROUPS;

# Максимальное число целей в условии ретаргетинга
our $MAX_GOALS_IN_RET_COND_RULE ||= $Settings::MAX_RETARGETING_GOALS_IN_GROUP;

# Максимальное время достижения цели
our $MAX_GOAL_REACH_TIME_DAYS ||= $Settings::MAX_RETARGETING_GOAL_TIME_DAYS;

# Максимальная длина названия условия ретаргетинга
our $MAX_CONDITION_NAME_LENGTH ||= 250;
our $MAX_CONDITION_DESC_LENGTH ||= 4096;

# Разрешенные символы для использования в текстовых полях
our $DISALLOW_LETTER_RE = qr/[^\Q${Settings::ALLOW_BANNER_LETTERS_STR}|\E]/i;

# Какие тексты ошибок отдавать: обычные (undef, с термином "условия подбора аудитории") или
# для APIv4 ('api4', с термином "условие ретаргетинга")
our $ERROR_TEXT_VARIANT = undef;


=head3 _lc_trim($name)

    Привести название ретаргетинга к стандартизированному виду, в котором можно будет сравниивать его на уникальность:

    1. Перевести в нижний регистр
    2. Убрать пустые символы с начала и конца
    3. Поменять все ё на е

    Все это нужно для того, чтобы позже, при сохранении условия ретаргетинга, не возникла ошибка в валидации
    уникальности через MySQL, так как в директе он не учитывает регистр, пустые символы и разницу в е и ё

=cut

sub _lc_trim {
    my ($name) = @_;
    $name = lc($name);
    $name =~ s/^\s+|\s+$//g;
    $name =~ s/ё/е/g;
    return $name;
}


=head2 validate_retargeting_conditions_for_client($checked_ret_conds, $remaining_ret_conds, $metrika_goals_by_id)

Валидация условий ретаргетинга в пределах одного клиента.

Параметры:
    $checked_ret_conds   -> список проверяемых (новых или изменившихся) условий ретаргетинга
    $remaining_ret_conds -> список оставшихся (не изменившихся) условий ретаргетинга
    $metrika_goals_by_id -> (опционально) хеш вида { goal_id => {goal_data}, ... }

=cut

sub validate_retargeting_conditions_for_client {
    my ($checked_ret_conds, $remaining_ret_conds, $metrika_goals_by_id) = @_;
    $remaining_ret_conds //= [];

    my $vr_main = validate_retargeting_conditions($checked_ret_conds, $metrika_goals_by_id);

    my $ret_conds_count = @$checked_ret_conds + @$remaining_ret_conds;
    if ($ret_conds_count > $MAX_RET_CONDS_ON_CLIENT) {
        $vr_main->add_generic(error_ReachLimit(iget("Количество условий подбора аудитории не должно превышать %s", $MAX_RET_CONDS_ON_CLIENT)));
    }
    
    return $vr_main if !$vr_main->is_valid;
    
    #
    # Проверка на уникальность
    #

    my (%used_ret_cond_names, %used_ret_cond_jsons);
    for (@$checked_ret_conds) {
        $used_ret_cond_names{ _lc_trim($_->condition_name) }++;
        $used_ret_cond_jsons{ $_->_condition_json }++;
    }

    # Проверим CHECKED условия ретаргетинга на уникальность по имени
    for (my $i = 0; $i < @$checked_ret_conds; $i++) {
        my $ret_cond = $checked_ret_conds->[$i];
        my $vr = $vr_main->get_objects_results->[$i];

        if ($used_ret_cond_names{ _lc_trim($ret_cond->condition_name) } > 1) {
            $vr->add_generic(error_Duplicated(iget("Названия условий подбора аудитории должны быть уникальны")));
        }
    }

    # Проверим CHECKED условия ретаргетинга на уникальность по условию (json)
    for (my $i = 0; $i < @$checked_ret_conds; $i++) {
        my $ret_cond = $checked_ret_conds->[$i];
        my $vr = $vr_main->get_objects_results->[$i];

        if ($used_ret_cond_jsons{ $ret_cond->_condition_json } > 1) {
            $vr->add_generic(error_Duplicated(iget("Наборы правил в условиях подбора аудитории должны быть уникальны")));
        }
    }
    return $vr_main if !$vr_main->is_valid;

    # Проверим CHECKED условия ретаргетинга на уникальность по имени с REMAINING
    %used_ret_cond_names = ();
    for (@$remaining_ret_conds) {
        if ($used_ret_cond_names{ _lc_trim($_->condition_name) }++) {
            #croak "ret_cond names on client #".$_->client_id." already duplicated";
        }
    }
    for (my $i = 0; $i < @$checked_ret_conds; $i++) {
        my $ret_cond = $checked_ret_conds->[$i];
        my $vr = $vr_main->get_objects_results->[$i];

        if ($used_ret_cond_names{ _lc_trim($ret_cond->condition_name) }++) {
            $vr->add_generic(error_InconsistentState(iget("Условие подбора аудитории с таким именем уже существует")));
        }
    }

    %used_ret_cond_jsons = ();
    for (@$remaining_ret_conds) {
        if ($used_ret_cond_jsons{ $_->_condition_json }++) {
            #croak "ret_cond jsons on client #".$_->client_id." already duplicated";
        }
    }
    for (my $i = 0; $i < @$checked_ret_conds; $i++) {
        my $ret_cond = $checked_ret_conds->[$i];
        my $vr = $vr_main->get_objects_results->[$i];

        if ($used_ret_cond_jsons{ $ret_cond->_condition_json }++) {
            $vr->add_generic(error_InconsistentState(iget("Условие подбора аудитории с таким набором правил уже существует")));
        }
    }

    return $vr_main;
}

=head2 validate_retargeting_conditions($ret_conds, $metrika_goals_by_id)

Валидация произвольного списка условий ретаргетинга.

=cut

sub validate_retargeting_conditions {
    my ($ret_conds, $metrika_goals_by_id) = @_;

    my $vr_main = Direct::ValidationResult->new();

    for my $ret_cond (@$ret_conds) {
        my $vr = $vr_main->next;

        #
        # condition_name ("Название условия ретаргетинга")
        #
        my $condition_name = $ret_cond->has_condition_name ? $ret_cond->condition_name : undef;
        if (!defined $condition_name) {
            $vr->add(condition_name => error_ReqField());  
        } else {
            if ($condition_name !~ /\S/) {
                $vr->add(condition_name => error_EmptyField());
            }
            if ($condition_name =~ $DISALLOW_LETTER_RE) {
                $vr->add(condition_name => error_InvalidChars_AlphaNumPunct());
            }
            if (length($condition_name) > $MAX_CONDITION_NAME_LENGTH) {
                $vr->add(condition_name => error_MaxLength(undef, length => $MAX_CONDITION_NAME_LENGTH));
            }
        }
        
        #
        # condition_desc ("Описание условия ретаргетинга")
        #
        my $condition_desc = $ret_cond->has_condition_desc ? $ret_cond->condition_desc : undef;
        if (defined $condition_desc)  {
            if ($condition_desc =~ $DISALLOW_LETTER_RE) {
                $vr->add(condition_desc => error_InvalidChars_AlphaNumPunct());
            }
            if (length($condition_desc) > $MAX_CONDITION_DESC_LENGTH) {
                $vr->add(condition_desc => error_MaxLength(undef, length => $MAX_CONDITION_DESC_LENGTH));
            }
        }

        #
        # condition
        #
        $vr->add(condition => validate_ret_cond_rules($ret_cond->condition, $metrika_goals_by_id));

        #
        # properties
        # Проверим, что свойство `negative` не изменяется
        #
        if ($ret_cond->has_old && ($ret_cond->old->is_negative ^ $ret_cond->is_negative)) {
            if ($ret_cond->old->is_negative) {
                if ($ERROR_TEXT_VARIANT && $ERROR_TEXT_VARIANT eq 'api4') {
                    $vr->add_generic(error_BadUsage(iget('В условии ретаргетинга нельзя изменять свойство "Не выполнено ни одного"')));
                } else {
                    $vr->add_generic(error_BadUsage(iget('В условии подбора аудитории нельзя изменять свойство "Не выполнено ни одного"')));
                }
            } else {
                if ($ERROR_TEXT_VARIANT && $ERROR_TEXT_VARIANT eq 'api4') {
                    $vr->add_generic(error_BadUsage(iget('В условии ретаргетинга нельзя изменить какое-либо из свойств на "Не выполнено ни одного"')));
                } else {
                    $vr->add_generic(error_BadUsage(iget('В условии подбора аудитории нельзя изменить какое-либо из свойств на "Не выполнено ни одного"')));
                }
            }
        }
    }

    return $vr_main;
}

=head2 validate_ret_cond_rules($rules, $metrika_goals_by_id)

Валидация правил одного условия ретаргетинга.

=cut

sub validate_ret_cond_rules {
    my ($rules, $metrika_goals_by_id) = @_;

    my $vr_main = Direct::ValidationResult->new();

    if (!@$rules || @$rules > $MAX_RULES_IN_RET_COND) {
        if ($ERROR_TEXT_VARIANT && $ERROR_TEXT_VARIANT eq 'api4') {
            $vr_main->add_generic(error_LimitExceeded(iget("Количество наборов в условии ретаргетинга должно быть от %s до %s", 1, $MAX_RULES_IN_RET_COND)));
        } else {
            $vr_main->add_generic(error_LimitExceeded(iget("Количество наборов в условии подбора аудитории должно быть от %s до %s", 1, $MAX_RULES_IN_RET_COND)));
        }
    }

    for my $rule (@$rules) {
        my $vr = $vr_main->next;

        my $vr_goals = Direct::ValidationResult->new();
        {;
            for my $goal (@{$rule->goals}) {
                my $vr_goal = $vr_goals->next;

                # Существование цели
                if ($metrika_goals_by_id && !$metrika_goals_by_id->{ $goal->goal_id }) {
                    $vr_goal->add(goal_id => error_NotFound_Goal());
                }

                # Время достижения цели
                if ( ($goal->goal_type eq 'goal' || $goal->goal_type eq 'segment') &&
                     ($goal->time < 1 || $goal->time > $MAX_GOAL_REACH_TIME_DAYS)
                ) {
                    $vr_goal->add(time => error_InvalidField(iget("Время достижения цели должно быть от %s до %s дней", 1, $MAX_GOAL_REACH_TIME_DAYS)));
                }
            }

            # Максимальное количество целей
            if (!@{$rule->goals} || @{$rule->goals} > $MAX_GOALS_IN_RET_COND_RULE) {
                if ($ERROR_TEXT_VARIANT && $ERROR_TEXT_VARIANT eq 'api4') {
                    $vr_goals->add_generic(error_LimitExceeded(iget("Количество целей в правиле условия ретаргетинга должно быть от %s до %s", 1, $MAX_GOALS_IN_RET_COND_RULE)));
                } else {
                    $vr_goals->add_generic(error_LimitExceeded(iget("Количество #elements_of_field# должно быть от %s до %s", 1, $MAX_GOALS_IN_RET_COND_RULE)));
                }
            }
        }
        $vr->add(goals => $vr_goals) if !$vr_goals->is_valid;
    }

    return $vr_main if !$vr_main->is_valid;

    # Условие состоит только из `not` правил // в этом случае могут быть только цели
    if ((all { $_->type eq 'not' } @$rules) && (any { $_->goal_type ne 'goal' } map { @{$_->goals} } @$rules)) {
        $vr_main->add_generic(error_InconsistentState(iget('Условие, состоящее только из наборов с типом "Не выполнено ни одной", может использовать только цели Яндекс.Метрики')));
    }

    return $vr_main;
}

1;
