package Direct::Validation::Retargetings;

use Direct::Modern;

use base qw(Exporter);

use Settings;

use Yandex::I18n;
use List::MoreUtils qw/any/;

use Currencies qw/get_currency_constant/;
use Currency::Format qw/format_const/;

use Direct::ValidationResult;
use Direct::Errors::Messages;

our @EXPORT = qw/
    validate_retargetings_for_adgroup
    validate_retargetings_bids
/;

# Максимальное число ретаргетингов в группе
our $MAX_RETARGETINGS_IN_ADGROUP = 50;

=head2 validate_retargetings_for_adgroup($checked_rets, $remaining_rets, $adgroup, $campaign)

Валидация списка ретаргетингов на группу.
Параметры:
    $checked_rets   -> список проверяемых (новых или изменившихся) ретаргетингов
    $remaining_rets -> список оставшихся (не изменившихся) ретаргетингов
    $adgroup        -> общая группа для $retargetings (Direct::Model::AdGroup)
    $campaign       -> общая кампания для $retargetings (HashRef)

=cut

sub validate_retargetings_for_adgroup {
    my ($checked_rets, $remaining_rets, $adgroup, $campaign) = @_;
    $remaining_rets //= [];

    my $vr_main = validate_retargetings($checked_rets, $campaign);

    my $retargetings_limit = $MAX_RETARGETINGS_IN_ADGROUP;
    my $retargetings_count = @$checked_rets + @$remaining_rets;

    if ($retargetings_count > $retargetings_limit) {
        $vr_main->add_generic(error_LimitExceeded(
            iget("Группа объявлений может содержать не более %d условий подбора аудитории", $retargetings_limit)
        ));
    }

    # Если на этом этапе есть ошибки, то дальнейшие проверки не делаем.
    # Т.к. далее будет проверяться уникальность и прочее, и нужны как минимум корректно заполненные структуры.
    return $vr_main unless $vr_main->is_valid;

    # Проверка на уникальность проверяемых условий ретаргетинга
    my %checked_ret_cond_ids;
    for (@$checked_rets) {
        if ($checked_ret_cond_ids{ $_->ret_cond_id }++) {
            $vr_main->add_generic(error_Duplicated(iget("Условия подбора аудитории должны быть уникальны")));
        }
    }
    return $vr_main unless $vr_main->is_valid;

    my %used_ret_cond_ids;

    # Добавим в хеш существующие условия ретаргетинга
    for my $ret (@$remaining_rets) {
        if ($used_ret_cond_ids{ $ret->ret_cond_id }++) {
            croak "ret_cond_id @{[$ret->ret_cond_id]} in adgroup @{[$ret->adgroup_id]} already duplicated";
        }
    }

    for (my $i = 0; $i < @$checked_rets; $i++) {
        my $ret = $checked_rets->[$i];
        my $vr = $vr_main->get_objects_results->[$i];

        if ($used_ret_cond_ids{ $ret->ret_cond_id }++) {
            $vr->add_generic(error_AlreadyExists(iget("Такое условие подбора аудитории уже существует в группе")));
        }
    }

    return $vr_main;
}

=head2 validate_retargetings($retargetings, $campaign)

Валидация списка ретаргетингов (без какой-либо привязки к группе) в пределах одной кампании.

=cut

sub validate_retargetings {
    my ($retargetings, $campaign) = @_;

    my $vr_main = Direct::ValidationResult->new();

    for my $ret (@$retargetings) {
        my $vr = $vr_main->next;

        #
        # ret_cond (должно быть привязано условие без признака `negative`)
        #
        if ($ret->ret_cond->is_negative) {
            $vr->add(ret_cond_id => error_InconsistentState(iget("Это условие подбора аудитории можно использовать только для корректировки ставок")));
        }
        if (any { $_ eq 'interest'} @{ $ret->ret_cond->properties }) {
            $vr->add(ret_cond_id => error_InconsistentState(iget("Это условие подбора аудитории можно использовать только для таргетинга по интересам")));
        }

        #
        # price_context ("Цена на сети")
        # autobudget_priority ("Приоритет автобюджета")
        #
        # Цены валидируем только при наличии стратегии (ее может не быть, если валидируется снапшот данных без привязки к кампании)
        #
        if (exists $campaign->{strategy}) {
            _validate_retargetings_bid($vr, $ret, $campaign);
        } 
    }

    return $vr_main;
}

=head2 validate_retargetings_bids($vr, $retargetings, $campaign)

Валидация ставок для списка ретаргетингов.

=cut

sub validate_retargetings_bids {
    my ($retargetings, $campaign) = @_;

    my $vr_main = Direct::ValidationResult->new();

    for my $ret (@$retargetings) {
        my $vr = $vr_main->next;
        #
        # price_context ("Цена на сети")
        # autobudget_priority ("Приоритет автобюджета")
        #
        # Цены валидируем только при наличии стратегии (ее может не быть, если валидируется снапшот данных без привязки к кампании)
        #
        if (exists $campaign->{strategy}) {
            _validate_retargetings_bid($vr, $ret, $campaign);
        }
    }

    return $vr_main;
}

sub _validate_retargetings_bid {
    my ($vr, $retargeting, $campaign) = @_;
    my $is_autobudget = $campaign->{strategy}->{is_autobudget};
    my $is_search_stop = $campaign->{strategy}->{is_search_stop};
    my $is_net_stop = $campaign->{strategy}->{is_net_stop};
    my $is_different_places = $campaign->{strategy}->{name} eq 'different_places';
    my $is_cpm_banner_campaign = ($campaign->{type} && $campaign->{type} eq 'cpm_banner') ? 1 : 0;
    my $is_cpm_deals_campaign = ($campaign->{type} && $campaign->{type} eq 'cpm_deals') ? 1 : 0;
    my $is_cpm_yndx_frontpage_campaign = ($campaign->{type} && $campaign->{type} eq 'cpm_yndx_frontpage') ? 1 : 0;
    my $is_cpm_price_campaign = ($campaign->{type} && $campaign->{type} eq 'cpm_price') ? 1 : 0;

    if (!$is_autobudget) {
        my $currency = $campaign->{currency};
        my $min_constant;
        if ($is_cpm_yndx_frontpage_campaign) {
            $min_constant = 'MIN_CPM_FRONTPAGE_PRICE';
        } else {
            $min_constant = ($is_cpm_banner_campaign || $is_cpm_deals_campaign || $is_cpm_price_campaign) ? 'MIN_CPM_PRICE' : 'MIN_PRICE';
        }
        my $max_constant = ($is_cpm_banner_campaign || $is_cpm_deals_campaign || $is_cpm_yndx_frontpage_campaign || $is_cpm_price_campaign) ? 'MAX_CPM_PRICE' : 'MAX_PRICE';
        my $min_price = get_currency_constant($currency, $min_constant);
        my $max_price = get_currency_constant($currency, $max_constant);

        for my $price_info (
            {name => 'price_context', is_required => $is_different_places && !$is_net_stop},
        ) {
            my $price = $price_info->{name};
            my $has_price = "has_${price}";

            if ($price_info->{is_required} && !$retargeting->$has_price) {
                $vr->add($price => error_ReqField());
                next;
            }

            next if !$retargeting->$has_price || ($retargeting->$price == 0 && !$price_info->{is_required});

            $vr->add($price => error_InvalidField(
                iget("Значение в поле #field# не может быть меньше %s", format_const($currency, $min_constant))
            )) if $retargeting->$price < $min_price;

            $vr->add($price => error_InvalidField(
                iget("Значение в поле #field# не может быть больше %s", format_const($currency, $max_constant))
            )) if $retargeting->$price > $max_price;
        }
    } else {
        my $autobudget_priority = $retargeting->has_autobudget_priority ? $retargeting->autobudget_priority : undef;

        if (!defined $autobudget_priority) {
            $vr->add(autobudget_priority => error_ReqField());
        } elsif ($autobudget_priority !~ /^(?:1|3|5)$/) {
            $vr->add(autobudget_priority => error_InvalidField());
        }
    }
}

1;
