package Direct::Validation::SitelinksSets;

use Direct::Modern;

use base qw(Exporter);

our @EXPORT_OK = qw/
    validate_sitelinks_sets
    validate_sitelinks_set_with_banners
    sitelinks_valid_domains_list

    $SITELINKS_NUMBER
    $SITELINKS_MAX_LENGTH
    $ONE_SITELINK_MAX_LENGTH
    $ONE_SITELINK_DESC_MAX_LENGTH
/;

use Direct::Validation::Errors;
use Direct::ValidationResult;

use Direct::Validation::HRef qw/validate_href/;
use Direct::Validation::TurboLandings qw/is_valid_turbolanding_href check_turbolandings_ownership/;

use TextTools qw/space_cleaner/;
use Settings;
use Yandex::I18n;
use List::Util qw/sum/;
use Yandex::ListUtils qw/xuniq/;
# use Yandex::URL qw/get_host/;
use List::MoreUtils qw/any all uniq/;

=head2 $SITELINKS_NUMBER

    Ограничение на число сайтлинков

=cut

our $SITELINKS_NUMBER = 8;

=head2 $SITELINKS_MAX_LENGTH

    Ограничение на общую длину заголовков

=cut

our $SITELINKS_MAX_LENGTH = 66;

=head2 $ONE_SITELINK_MAX_LENGTH

    Ограничение на длину одного заголовка

=cut

our $ONE_SITELINK_MAX_LENGTH = 30;

=head2 $ONE_SITELINK_DESC_MAX_LENGTH

    Ограничение на длину одного описания

=cut

our $ONE_SITELINK_DESC_MAX_LENGTH = 60;

=head2 $ALLOW_SAME_TURBOLANDINGS

Можно использовать одинаковые турболендинги внутри набора сайтлинков

=cut

our $ALLOW_SAME_TURBOLANDINGS = 0;


=head2 validate_sitelinks_sets($sitelinks_sets)

    Базовая проверка сайтлинк-сетов.

    Параметры:
        $sitelinks_sets - ссылка на массив сайтлинк-сетов [Direct::Model::SitelinksSet, ...]

    Результат:
        Результат проверки (ValidationResult)
        Ошибки хранятся по сайтлинкам (field) по сайтлинк-сетам (field and generic)

=cut

sub validate_sitelinks_sets {
    my ($sitelinks_sets) = @_;

    my $validation = Direct::ValidationResult->new();

    for my $sitelinks_set (@$sitelinks_sets) {
        my $sitelinks_set_vr = $validation->next;

        my $sitelinks = $sitelinks_set->links;
        my $validation_sitelinks = Direct::ValidationResult->new();

        my $wrong_turbolandings;
        my @turbolandings = map {$_->turbolanding} grep {$_->isa('Direct::Model::Sitelink') && $_->has_turbolanding} @$sitelinks;
        if (@turbolandings) {
            my $wrong = check_turbolandings_ownership($sitelinks_set->client_id => \@turbolandings);
            $wrong_turbolandings->{$_->id} = 1 foreach @$wrong;
        }
        
        for my $sitelink (@$sitelinks) {
            
            my $sitelink_vr = $validation_sitelinks->next;

            #
            #turbolanding
            #
            if ($sitelink->isa('Direct::Model::Sitelink') && $sitelink->has_turbolanding && $wrong_turbolandings->{$sitelink->turbolanding->id}) {
                $sitelink_vr->add(turbolanding => error_InconsistentState(iget('Для быстрой ссылки задана несуществующая Турбо-страница')));
            }

            #
            # title
            #
            my $title = $sitelink->title;
            if (!defined $title || $title !~ /\S/) {
                $sitelink_vr->add(title => error_ReqField(
                    iget("Не указан текст быстрой ссылки")
                ));
            } else {
                if (length($title) > $ONE_SITELINK_MAX_LENGTH) {
                    $sitelink_vr->add(title => error_MaxLength(
                        iget("Превышена допустимая длина текста одной быстрой ссылки в %s символов", $ONE_SITELINK_MAX_LENGTH)
                    ));
                }
                if ($title =~ $Settings::DISALLOW_BANNER_LETTER_RE || $title =~ /[!?]/) {
                    $sitelink_vr->add(title => error_InvalidChars(
                        iget('В тексте быстрых ссылок можно использовать только буквы латинского, турецкого, русского, украинского, казахского или белорусского алфавита, знаки пунктуации, за исключением !,?')
                    ));
                }
            }

            #
            # description
            #
            my $description = $sitelink->description;
            if (defined $description) {
                if ($description =~ /^\s*$/) {
                    $sitelink_vr->add(description => error_EmptyField(
                        iget("Не указан текст описания быстрой ссылки")
                    ));
                }
                if (length($description) > $ONE_SITELINK_DESC_MAX_LENGTH) {
                    $sitelink_vr->add(description => error_MaxLength(
                        iget("Превышена допустимая длина текста одного описания быстрой ссылки в %s символов", $ONE_SITELINK_DESC_MAX_LENGTH)
                    ));
                }
                if ($description =~ $Settings::DISALLOW_BANNER_LETTER_RE || $description =~ /[!?]/) {
                    $sitelink_vr->add(description => error_InvalidChars(
                        iget('В описании быстрых ссылок можно использовать только буквы латинского, турецкого, русского, украинского, казахского или белорусского алфавита, знаки пунктуации, за исключением !,?')
                    ));
                }
            }

            #
            # href
            #
            my $href = $sitelink->href;
            if (!defined $href || $href !~ /\S/) {
                $sitelink_vr->add(href => error_ReqField(
                    iget("Не указан адрес быстрой ссылки")
                )) if !$sitelink->has_turbolanding;
            } else {
                $sitelink_vr->add(href => validate_href($href, context => 'sitelink'));
                if (is_valid_turbolanding_href($href)) {
                    $sitelink_vr->add(href => error_InconsistentState(iget('Ссылка сайтлинка не может указывать на Турбо-страницу')));
                }
            }
        }

        $sitelinks_set_vr->add(sitelinks => $validation_sitelinks) unless $validation_sitelinks->is_valid;

        if (@$sitelinks == 0 || @$sitelinks > $SITELINKS_NUMBER) {
            $sitelinks_set_vr->add_generic(error_LimitExceeded(
                iget("Количество быстрых ссылок должно быть от %s до %s", 1, $SITELINKS_NUMBER)
            ));
        }

        if (@$sitelinks) {
            my @titles = grep { defined $_ && /\S/ } map { $_->{title} } @$sitelinks;
            if (scalar @titles > 4) {
                # предполагаем, что у нас режим проверки блоков по 4 сайтлинка.
                my @first_block_titles = @titles[0 .. 3];
                my $second_block_end = $#titles > 7 ? 7 : $#titles;
                my @second_block_titles = @titles[4 .. $second_block_end];
                if (sum(map { length $_ } @first_block_titles) > $SITELINKS_MAX_LENGTH) {
                    $sitelinks_set_vr->add_generic(error_MaxLength(
                        iget("Суммарная длина текстов быстрых ссылок № 1-4 превышает %s символов", $SITELINKS_MAX_LENGTH)
                    ));
                }
                if (sum(map { length $_ } @second_block_titles) > $SITELINKS_MAX_LENGTH) {
                    $sitelinks_set_vr->add_generic(error_MaxLength(
                        iget("Суммарная длина текстов быстрых ссылок № 5-8 превышает %s символов", $SITELINKS_MAX_LENGTH)
                    ));
                }
            } elsif (@titles && sum(map { length $_ } @titles) > $SITELINKS_MAX_LENGTH) {
                $sitelinks_set_vr->add_generic(error_MaxLength(
                    iget("Превышена суммарная длина текста быстрых ссылок — максимум %s символов", $SITELINKS_MAX_LENGTH)
                ));
            }

            my $title_uniq_count = xuniq { lc space_cleaner($_) } @titles;
            if (@titles != $title_uniq_count) {
                $sitelinks_set_vr->add_generic(error_DuplicateField(
                    iget("Тексты быстрых ссылок не должны быть одинаковыми")
                ));
            }

            my @descriptions = grep { defined $_ && /\S/ } map { $_->{description} } @$sitelinks;
            my $description_uniq_count = xuniq { lc space_cleaner($_) } @descriptions;
            if (scalar(@descriptions) != $description_uniq_count) {
                $sitelinks_set_vr->add_generic(error_DuplicateField(
                    iget("Описания быстрых ссылок не должны быть одинаковыми")
                ));
            }
        }
    }

    return $validation;
}

=head2 sitelinks_valid_domains_list

Список регулярок, описывающих домены, которые могут быть в сайтлинках
Кроме этого списка, в сайтлинках может быть домен из banner.href

Регулярки должны быть совместимые с js, т.к. отдаются в верстку

=cut

sub sitelinks_valid_domains_list
{
    my @re = map { '(?:[^\.]+\.)?'.quotemeta($_) } qw/
        ask.fm
        avaaz.org
        badoo.com
        change.org
        facebook.com
        flickr.com
        foursquare.com
        instagram.com
        last.fm
        linkedin.com
        livejournal.com
        mixi.jp
        myspace.com
        ok.ru
        pinterest.com
        plus.google.com
        renren.com
        soundcloud.com
        tagged.com
        tumblr.com
        twitter.com
        vine.co
        vk.com
        weibo.com
    /;
    push @re, 'market\.yandex\.(?:ru|ua|com\.tr|com|kz|by)';
    return \@re;
}

#sub _is_valid_sitelink_domain
#{
#    my $domain = shift;
#    state $re_list = sitelinks_valid_domains_list();
#    my $re = join '|', @$re_list;
#    return $domain =~ /^$re$/;
#}

1;
