package Direct::Validation::Strategy;

use Direct::Modern;

use Direct::ValidationResult;
use Direct::Validation::Errors;

use Yandex::I18n;
use Yandex::Validate;
use Yandex::DateTime qw/now date/;
use Yandex::TimeCommon qw/mysql2unix unix2human ts_round_day get_distinct_dates tomorrow today str_round_day/;
use List::Util qw/pairs max/;
use Try::Tiny;

use Campaign::Types;
use Currencies qw/get_currency_constant get_geo_list get_frontpage_min_price get_min_price/;
use Currency::Format qw/format_const format_sum_of_money format_currency/;
use CampaignTools;
use TextTools;

use base qw/Exporter/;
our @EXPORT = qw/
    validate_strategy_for_campaign
/;

our $MIN_STRATEGY_PERIOD_DAYS_COUNT = 1;
our $MAX_STRATEGY_PERIOD_DAYS_COUNT = 90;

our $CPM_STRATEGY_DAILY_CHANGE_COUNT = 4;

our $MIN_CRR_VALUE = 1;
our $MAX_CRR_VALUE = 500;

=head2 validate_strategy_for_campaign

Проверка возможности установить стратегию для кампании

=cut

{
my $check_sub_by_field = {
    sum                => \&validate_sum,
    bid                => \&validate_bid,
    avg_bid            => \&validate_avg_bid,
    avg_cpa            => \&validate_avg_cpa,
    avg_cpm            => \&validate_avg_cpm,
    avg_cpv            => \&validate_avg_cpv,
    avg_cpi            => \&validate_avg_cpi,
    limit_clicks       => \&validate_limit_clicks,
    filter_avg_bid     => \&validate_avg_bid,
    filter_avg_cpa     => \&validate_avg_cpa,
    goal_id            => \&validate_goal_id,
    crr                => \&validate_crr,
    roi_coef           => \&validate_roi_coef,
    reserve_return     => \&validate_roi_reserve_return,
    profitability      => \&validate_roi_profitability,
    date               => \&validate_autobudget_date,
    start              => \&validate_start,
    finish             => \&validate_finish,
    budget             => \&validate_budget,
    pay_for_conversion => \&validate_pay_for_conversion,
};

=head2 validate_strategy_for_campaign($strategy, $camp, %opts)

    %opts
        is_flat_cpc - включено ли совместное управление ставками
        meaningful_goals - ключевые цели из запроса
        has_mobile_app_goals_for_text_campaign_allowed - возможность указывать для ТГО-кампании специальные цели для РМП (не используется)
        has_mobile_app_goals_for_text_campaign_strategy_enabled - фича для использования целей РМП в стратегиях ТГО-кампаний
        has_disable_all_goals_optimization_for_dna_enabled - запрет использовать оптимизацию по всем целям
        has_disable_autobudget_week_bundle_feature - отключение стратегии `Пакет кликов`
        has_flat_cpc_disabled - отключение стратегии совместного управления ставками
        has_flat_cpc_adding_disabled - отключение добавления стратегии совместного управления ставками

=cut

sub validate_strategy_for_campaign {
    my ($strategy, $camp, %opts) = @_;
    my $vr = Direct::ValidationResult->new();

    $vr->add_generic(error_BadUsage(iget("Стратегия не поддерживается для данного типа кампаний")))
        if !$strategy->_is_campaign_type_supported($camp->supported_type);

    if ($opts{has_edit_avg_cpm_without_restart_enabled}) {
        my $update_prohibited_defect = validate_strategy_update_timeout($strategy, $camp);
        $vr->add_generic($update_prohibited_defect) if $update_prohibited_defect;
    }

    if ($strategy->name eq 'autobudget_week_bundle' && $opts{has_disable_autobudget_week_bundle_feature}) {
        # Если кампания создается или переходит на Пакет кликов с другой стратегии
        if ($opts{new_camp} || ($camp->{strategy} && $camp->{strategy}->name ne $strategy->name)) {
            $vr->add_generic(error_BadUsage(iget("Стратегия 'Пакет кликов' не поддерживается")));
        }
    }

    if (($camp->campaign_type eq 'text' || $camp->campaign_type eq 'dynamic' || $camp->campaign_type eq 'mobile_content')
        && $opts{is_flat_cpc}) {
        #Под одной фичей всегда выдаём ошибку, под другой - ошибка при добавлении и ворнинг при обновлении
        if ($opts{has_flat_cpc_disabled}) {
            $vr->add_generic(error_BadUsage(iget("Стратегия совместного управления ставками не поддерживается")));
        } elsif ($opts{has_flat_cpc_adding_disabled} && $opts{new_camp}) {
                $vr->add_generic(error_BadUsage(iget("Стратегия совместного управления ставками не поддерживается")));
        }
    }

    my $checks = $strategy->get_fields_to_check();
    for my $check (pairs @$checks) {
        my ($field, $check_params) = @$check;
        my $check_sub = $check_sub_by_field->{$field};
        croak "Don't know how to check <$field>" if !$check_sub;

        my $value = shift @$check_params;
        my $defect = $check_sub->($value, $camp, $strategy, @$check_params,
            new_camp                                     => $opts{new_camp},
            meaningful_goals                             => $opts{meaningful_goals},
            prefetched_goals                             => $opts{prefetched_goals},
            has_cpa_pay_for_conversions_extended_mode_allowed => $opts{has_cpa_pay_for_conversions_extended_mode_allowed},
            has_cpa_pay_for_conversions_mobile_apps_allowed => $opts{has_cpa_pay_for_conversions_mobile_apps_allowed},
            has_edit_avg_cpm_without_restart_enabled => $opts{has_edit_avg_cpm_without_restart_enabled},
            has_mobile_app_goals_for_text_campaign_allowed => $opts{has_mobile_app_goals_for_text_campaign_allowed},
            has_mobile_app_goals_for_text_campaign_strategy_enabled => $opts{has_mobile_app_goals_for_text_campaign_strategy_enabled},
            has_disable_all_goals_optimization_for_dna_enabled      => $opts{has_disable_all_goals_optimization_for_dna_enabled},
            has_increased_cpa_limit_for_pay_for_conversion       => $opts{has_increased_cpa_limit_for_pay_for_conversion},
            has_all_meaningful_goals_for_pay_for_conversion_strategies_allowed => $opts{has_all_meaningful_goals_for_pay_for_conversion_strategies_allowed}
        );
        next if !$defect;
        $vr->add($field => $defect);
    }

    return $vr;
}
}

=head2 validate_strategy_update_timeout

    Проверка стратегии на возможность обновления в текущий момент

=cut

sub validate_strategy_update_timeout {
    my ($strategy, $camp) = @_;
    my $camp_strategy = $camp->get_strategy();

    return if (!$camp_strategy ||
        $camp_strategy->name ne $strategy->name ||
        !$strategy->isa('Direct::Strategy::AutobudgetCpmCustomPeriodBase')
            && !$strategy->isa('Direct::Strategy::AutobudgetAvgCpvCustomPeriodBase'));
    return if !$camp_strategy->last_update_time || !$camp_strategy->daily_change_count;

    #если дата старта стратегии еще не наступила, то временное ограничение на редактировние параметров не накладываем.
    my $now_ts = ts_round_day(time());
    my $strategy_start_ts = ts_round_day(mysql2unix($strategy->start));
    return if ($now_ts < $strategy_start_ts);

    my $last_update_date = str_round_day($camp_strategy->last_update_time);
    if ($strategy->start eq $camp_strategy->start
        && $last_update_date eq str_round_day(today())
        && $camp_strategy->daily_change_count >= $CPM_STRATEGY_DAILY_CHANGE_COUNT) {
        return error_BadUsage(iget("Параметры стратегии можно менять не более %d раз в день.", $CPM_STRATEGY_DAILY_CHANGE_COUNT));
    }
    return;
}

=head2 validate_limit_clicks

Проверка количества кликов на неделю

Опции:
 * required

=cut

sub validate_limit_clicks {
    my ($limit_clicks, $camp, $strategy, %opt) = @_;
    my $currency = $camp->currency;

    return error_EmptyField(iget("Не указано количество кликов на неделю."))
        if !defined $limit_clicks && $opt{required};

    return if !defined $limit_clicks;

    return error_InvalidFormat(iget("Неверно указано количество кликов на неделю."))
        if !is_valid_int($limit_clicks);

    return error_InvalidField(iget('Количество кликов на неделю должно быть не менее %s', format_const($currency,'MIN_AUTOBUDGET_CLICKS_BUNDLE')))
        if $limit_clicks < get_currency_constant($currency, 'MIN_AUTOBUDGET_CLICKS_BUNDLE');

    return error_InvalidField(iget("Указано слишком большое количество кликов"))
        if $limit_clicks > get_currency_constant($currency, 'MAX_AUTOBUDGET_CLICKS_BUNDLE');

    return;
}

=head2 validate_sum

Проверка суммы недельного бюджета

Опции:
 * required

=cut

sub validate_sum {
    my ($sum, $camp, $strategy, %opt) = @_;
    my $currency = $camp->currency;

    return error_EmptyField(iget("Не указана сумма недельного бюджета."))
        if !defined $sum && $opt{required};

    return if !defined $sum;

    return error_InvalidFormat(iget("Неверно указан бюджет."))
        if !is_valid_float($sum);


    my $min_weekly_budget;
    if (camp_kind_in(type => $camp->campaign_type, 'cpm')) {
        $min_weekly_budget = round2s(get_currency_constant($currency, 'MIN_DAILY_BUDGET_FOR_PERIOD') * 7);
    } else {
        $min_weekly_budget = round2s(get_currency_constant($currency, 'MIN_AUTOBUDGET'));
    }
    return error_InvalidField(iget('Недельный бюджет не может быть меньше %s', format_sum_of_money($currency, $min_weekly_budget)))
        if $sum < $min_weekly_budget;

    return error_InvalidField(iget("Недельный бюджет не может быть больше %s", format_const($currency, 'MAX_AUTOBUDGET')))
        if $sum > get_currency_constant($currency, 'MAX_AUTOBUDGET');

    return;
}

=head2 validate_budget

Проверка суммы бюджета на фиксированный период

Опции:
 * required

=cut

sub validate_budget {
    my ($budget, $camp, $strategy, %opt) = @_;
    my $currency = $camp->currency;

    return error_EmptyField(iget("Не указана сумма бюджета на фиксированный период."))
        if !defined $budget && $opt{required};

    return if !defined $budget;

    return error_InvalidFormat(iget("Неверно указан бюджет на фиксированный период."))
        if !is_valid_float($budget);

    return if validate_start($strategy->start, $camp, $strategy, %opt) || validate_finish($strategy->finish, $camp, $strategy, %opt);

    #Если редактируем существующий период и меняем бюджет в меньшую сторону,
    # то дополнительно проверяем что новый бюджет больше, чем текущие траты с некоторым запасом и выдерживается
    # минимальный дневной бюджет на оставшейся период. (https://st.yandex-team.ru/DIRECT-106529)
    my $camp_strategy = $camp->get_strategy();
    if ($camp_strategy
        && $opt{has_edit_avg_cpm_without_restart_enabled}
        && ($strategy->isa('Direct::Strategy::AutobudgetCpmCustomPeriodBase')
                || $strategy->isa('Direct::Strategy::AutobudgetAvgCpvCustomPeriodBase'))
        && $strategy->name eq $camp_strategy->name
        && $strategy->start eq $camp_strategy->start
        && $strategy->is_period_strategy_in_progress()
        && ($budget ne $camp_strategy->budget || $strategy->finish ne $camp_strategy->finish)) {

        my $min_budget = CampaignTools::calc_min_available_budget_for_strategy_with_custom_period($camp_strategy, $camp->client_id, $currency, $camp->{bs_order_id}, $strategy->finish);

        if ($budget < $min_budget) {
            return error_InvalidField(iget('Бюджет на текущий период не может быть меньше %s', format_sum_of_money($currency, $min_budget)))
        }
    } else {

        my $days_count = scalar get_distinct_dates($strategy->start, $strategy->finish);
        my $min_sum = get_currency_constant( $currency, 'MIN_DAILY_BUDGET_FOR_PERIOD' ) * $days_count;
        my $max_sum = get_currency_constant( $currency, 'MAX_DAILY_BUDGET_FOR_PERIOD' ) * $days_count;


        return error_InvalidField(iget('Бюджет на фиксированный период не может быть меньше %s', format_sum_of_money($currency, $min_sum)))
            if $budget < $min_sum;

        return error_InvalidField(iget("Бюджет на фиксированный период не может быть больше %s", format_sum_of_money($currency, $max_sum)))
            if $budget > $max_sum;
    }

    return;
}



=head2 validate_bid

Проверка максимальной ставки

Опции:
 * required
 * sum

=cut

sub validate_bid {
    my ($bid, $camp, $strategy, %opt) = @_;
    my $currency = $camp->currency;

    return error_EmptyField(iget("Не указана максимальная ставка."))
        if !defined $bid && $opt{required};

    return if !defined $bid;

    return error_InvalidFormat(iget("Неверно указана максимальная ставка."))
        if !is_valid_float($bid);

    my $min_const = $camp->campaign_type eq 'performance' ? 'MIN_CPC_CPA_PERFORMANCE' : 'MIN_PRICE';
    return error_InvalidField(iget("Указана ставка меньше минимальной %s", format_const($currency, $min_const)))
        if $bid < get_currency_constant($currency, $min_const);

    # надо бы уточнить условие
    my $max_const = $strategy->name eq 'autobudget_week_bundle' ? 'MAX_PRICE' : 'MAX_AUTOBUDGET_BID';
    return error_InvalidField(iget("Указана ставка больше максимальной %s", format_const($currency, $max_const)))
        if $bid > get_currency_constant($currency, $max_const);

    return error_InvalidField(iget("Недельный бюджет должен быть больше максимальной ставки."))
        if $opt{sum} && $bid > $opt{sum};

    return error_InvalidFormat(iget("При оплате за конверсии ограничение максимальной ставки запрещено."))
        if (($strategy->name eq 'autobudget_avg_cpa'
            || $strategy->name eq 'autobudget_avg_cpa_per_camp'
            || $strategy->name eq 'autobudget_avg_cpa_per_filter')
            && $strategy->pay_for_conversion && $bid > 0);

    return;
}


=head2 validate_avg_bid

Проверка средней цены клика

Опции:
 * required
 * sum
 * bid

=cut

sub validate_avg_bid {
    my ($avg_bid, $camp, $strategy, %opt) = @_;
    my $currency = $camp->currency;

    return error_EmptyField(iget("Не указано значение средней цены клика."))
        if !defined $avg_bid && $opt{required};

    return if !defined $avg_bid;

    return error_InvalidFormat(iget("Неверно указано значение средней цены клика."))
        if !is_valid_float($avg_bid);

    my $min_const = $camp->campaign_type eq 'performance' ? 'MIN_CPC_CPA_PERFORMANCE' : 'MIN_AUTOBUDGET_AVG_PRICE';
    return error_InvalidField(iget('Средняя цена клика должна быть больше %s', format_const($currency, $min_const)))
        if $avg_bid < get_currency_constant($currency, $min_const);

    return error_InvalidField(iget("Средняя цена клика не может превышать %s", format_const($currency, 'MAX_AUTOBUDGET_BID')))
        if $avg_bid > get_currency_constant($currency, 'MAX_AUTOBUDGET_BID');

    return error_InvalidField(iget("Недельный бюджет должен быть больше средней цены клика."))
        if $opt{sum} && $avg_bid > $opt{sum};

    return error_InvalidField(iget("Нельзя одновременно задать и максимальную ставку и среднюю цену клика."))
        if $opt{bid};

    return;
}


=head2 validate_avg_cpa

Проверка средней цены конверсии

Опции:
 * required
 * sum

=cut

sub validate_avg_cpa {
    my ($avg_cpa, $camp, $strategy, %opt) = @_;
    my $currency = $camp->currency;

    return error_EmptyField(iget("Не указано значение средней цены конверсии."))
        if !defined $avg_cpa && $opt{required};

    return if !defined $avg_cpa;

    return error_InvalidFormat(iget("Неверно указано значение средней цены конверсии."))
        if !is_valid_float($avg_cpa);

    my $min_const = $camp->campaign_type eq 'performance' ? 'MIN_CPC_CPA_PERFORMANCE' : 'MIN_AUTOBUDGET_AVG_CPA';
    return error_InvalidField(iget('Средняя цена конверсии должна быть не менее %s', format_const($currency, $min_const)))
        if $avg_cpa < get_currency_constant($currency, $min_const);

    my $autobudget_pay_for_conversion_avg_cpa_warning = $opt{has_increased_cpa_limit_for_pay_for_conversion} ?
        'AUTOBUDGET_PAY_FOR_CONVERSION_AVG_CPA_WARNING_INCREASED' :
        'AUTOBUDGET_PAY_FOR_CONVERSION_AVG_CPA_WARNING';

    my $avg_cpa_restriction_name = (
        ($strategy->name eq 'autobudget_avg_cpa'
            || $strategy->name eq 'autobudget_avg_cpa_per_camp'
            || $strategy->name eq 'autobudget_avg_cpa_per_filter')
        && $strategy->pay_for_conversion)
        ? $autobudget_pay_for_conversion_avg_cpa_warning
        : 'AUTOBUDGET_AVG_CPA_WARNING';

    return error_InvalidField(iget('Средняя цена конверсии должна быть не более %s', format_const($currency, $avg_cpa_restriction_name)))
        if $avg_cpa > get_currency_constant($currency, $avg_cpa_restriction_name);

    return error_InvalidField(iget("Недельный бюджет должен быть больше средней цены конверсии."))
        if $opt{sum} && $avg_cpa > $opt{sum};

    return;
}


=head2 validate_avg_cpi

Проверка средней цены установки

Опции:
 * required
 * sum

=cut

sub validate_avg_cpi {
    my ($avg_cpi, $camp, $strategy, %opt) = @_;
    my $currency = $camp->currency;

    return error_EmptyField(iget("Не указано значение средней цены установки."))
        if !defined $avg_cpi && $opt{required};

    return if !defined $avg_cpi;

    return error_InvalidFormat(iget("Неверно указано значение средней цены установки."))
        if !is_valid_float($avg_cpi);

    return error_InvalidField(iget('Средняя цена установки должна быть не менее %s', format_const($currency, 'MIN_AUTOBUDGET_AVG_CPA')))
        if $avg_cpi < get_currency_constant($currency, 'MIN_AUTOBUDGET_AVG_CPA');

    my $autobudget_pay_for_conversion_avg_cpa_warning = $opt{has_increased_cpa_limit_for_pay_for_conversion} ?
        'AUTOBUDGET_PAY_FOR_CONVERSION_AVG_CPA_WARNING_INCREASED' :
        'AUTOBUDGET_PAY_FOR_CONVERSION_AVG_CPA_WARNING';

    return error_InvalidField(iget('Средняя цена установки должна быть не более %s', format_const($currency, $autobudget_pay_for_conversion_avg_cpa_warning)))
        if $strategy->pay_for_conversion && $avg_cpi > get_currency_constant($currency, $autobudget_pay_for_conversion_avg_cpa_warning);

    return error_InvalidField(iget("Недельный бюджет должен быть больше средней цены установки."))
        if $opt{sum} && $avg_cpi > $opt{sum};

    return;
}

=head2 validate_avg_cpm

Проверка средней цены за 1000 показов

Опции:
 * required
 * sum

=cut

sub validate_avg_cpm {
    my ($avg_cpm, $camp, $strategy, %opt) = @_;
    my $currency = $camp->currency;

    return error_EmptyField(iget("Не указано значение средней цены за тыс. показов."))
        if !defined $avg_cpm && $opt{required};

    return if !defined $avg_cpm;

    return error_InvalidFormat(iget("Неверно указано значение средней цены за тыс. показов."))
        if !is_valid_float($avg_cpm);

    if ($camp->campaign_type ne "cpm_yndx_frontpage") {
        return error_InvalidField(iget('Средняя цена за тыс. показов должна быть не менее %s', format_const($currency, 'MIN_AUTOBUDGET_AVG_CPM')))
            if $avg_cpm < get_currency_constant($currency, 'MIN_AUTOBUDGET_AVG_CPM');
    } else {
        my $error = validate_frontpage_avg_cpm($avg_cpm, $camp, $strategy, %opt);
        return $error if $error;
    }
    return error_InvalidField(iget('Средняя цена за тыс. показов должна быть не более %s', format_const($currency, 'MAX_CPM_PRICE')))
        if $avg_cpm > get_currency_constant($currency, 'MAX_CPM_PRICE');

    return;
}


=head2 validate_avg_cpv

Проверка средней цены за просмотр

Опции:
 * required
 * sum

=cut

sub validate_avg_cpv {
    my ($avg_cpv, $camp, $strategy, %opt) = @_;
    my $currency = $camp->currency;

    return error_EmptyField(iget("Не указано значение средней цены за просмотр."))
        if !defined $avg_cpv && $opt{required};

    return if !defined $avg_cpv;

    return error_InvalidFormat(iget("Неверно указано значение средней цены за просмотр."))
        if !is_valid_float($avg_cpv);

    return error_InvalidField(iget('Средняя цена за просмотр должна быть не менее %s', format_const($currency, 'MIN_AVG_CPV')))
        if $avg_cpv < get_currency_constant($currency, 'MIN_AVG_CPV');

    return error_InvalidField(iget('Средняя цена за просмотр должна быть не более %s', format_const($currency, 'MAX_AVG_CPV')))
        if $avg_cpv > get_currency_constant($currency, 'MAX_AVG_CPV');

    return error_EmptyField(iget("Оплата за показы возможна только в кампаниях с группами типа Видео"))
        if $camp->has_id && is_valid_id($camp->id) && !CampaignTools::is_cpv_strategies_enabled($camp->id);

    return;
}

=head2 validate_frontpage_avg_cpm

    Проверка средней цены за 1000 показов на Главной

    $error = validate_frontpage_avg_cpm($avg_cpm, $camp, $strategy, %opt);

=cut

sub validate_frontpage_avg_cpm {
    my ($avg_cpm, $camp, $strategy, %opt) = @_;
    my $currency = $camp->currency;

    my $min_avg_cpm = get_min_price($currency, $camp->geo, $camp->allowed_frontpage_types, $camp->client_id);

    if ($avg_cpm < $min_avg_cpm || $avg_cpm < 0) {
        return error_InvalidField(iget('Средняя цена за тыс. показов должна быть не менее %s %s', $min_avg_cpm, format_currency($currency)));
    }
    return;
}

=head2 validate_crr

Проверка доли рекламных расходов (ДРР)

Опции:
 * required

=cut

sub validate_crr {
    my ($crr, $camp, $strategy,  %opt) = @_;

    return error_EmptyField(iget("Не указано значение доли рекламных расходов (должно быть от 1% до 500%)"))
        if !defined $crr && $opt{required};

    return if !defined $crr;

    return error_InvalidFormat(iget("Значение доли рекламных расходов должно быть целым числом"))
        if !is_valid_int($crr);

    return error_InvalidField(iget("Значение доли рекламных расходов должно быть не меньше 1%"))
        if $crr < $MIN_CRR_VALUE;

    return error_InvalidField(iget("Значение доли рекламных расходов должно быть не больше 500%"))
        if $crr > $MAX_CRR_VALUE;

    return;
}

=head2 validate_roi_coef

Проверка коэффициента рентабельности

Опции:
 * required

=cut

sub validate_roi_coef {
    my ($roi_coef, $camp, $strategy, %opt) = @_;

    return error_EmptyField(iget("Не указано значение коэффициента рентабельности рекламы (должно быть больше -1)."))
        if !defined $roi_coef && $opt{required};

    return if !defined $roi_coef;

    return error_InvalidFormat(iget("Значение коэффициента рентабельности должно быть числом"))
        if !is_valid_float($roi_coef);

    return error_InvalidField(iget("Значение коэффициента рентабельности рекламы должно быть больше -1."))
        if $roi_coef <= -1;

    return;
}


=head2 validate_roi_reserve_return

Проверка процента возврата в рекламу от сэкономленного бюджета

Опции:
 * required

=cut

sub validate_roi_reserve_return {
    my ($reserve_return, $camp, $strategy, %opt) = @_;

    return error_EmptyField(iget("Не указано значение процента возврата в рекламу от сэкономленного бюджета."))
        if !defined $reserve_return && $opt{required};

    return if !defined $reserve_return;

    return error_InvalidField(iget("Процент возврата в рекламу от сэкономленного бюджета должен быть целым числом в диапазоне от 0 до 100 с шагом 10."))
        if !is_valid_int($reserve_return, 0, 100) || $reserve_return % 10;

    return;
}


=head2 validate_roi_profitability

Проверка процента доходов, являющихся себестоимостью товаров

Опции:
 * required

=cut

sub validate_roi_profitability {
    my ($profitability, $camp, $strategy, %opt) = @_;

    return error_EmptyField(iget("Не указано значение процента доходов, являющихся себестоимостью товаров или услуг."))
        if !defined $profitability && $opt{required};

    return if !defined $profitability;

    return error_InvalidFormat(iget("Процент доходов, являющихся себестоимостью товаров или услуг должен быть числом"))
        if !is_valid_float($profitability);

    return error_InvalidField(iget("Процент доходов, являющихся себестоимостью товаров или услуг должен быть указан в диапазоне от 0 до 100."))
        if $profitability < 0 || $profitability > 100;

    return;
}


=head2 validate_autobudget_date

Проверка длительности баяновых кампаний

Опции:
 * required

=cut

sub validate_autobudget_date {
    my ($date, $camp, $strategy, %opt) = @_;

    return error_EmptyField(iget("Не указана длительность кампании"))
        if !defined $date && $opt{required};

    return if !defined $date;

    return error_InvalidFormat(iget("Неверно указана длительность кампании"))
        if !is_valid_date($date);

    my @today = localtime;
    return error_InvalidField(iget("Неверно указана конечная дата показов"))
        if $date !~ /^(\d{4})-(\d{2})-(\d{2})$/ || "$1$2$3"+0 <= sprintf("%04d%02d%02d", $today[5]+1900, $today[4]+1, $today[3])+0;

    return;
}


=head2 validate_goal_id

Проверка цели метрики

Опции:
 * required
 * has_mobile_app_goals_for_text_campaign_allowed
 * has_mobile_app_goals_for_text_campaign_strategy_enabled

=cut

sub validate_goal_id {
    my ($goal_id, $camp, $strategy, %opt) = @_;

    return error_EmptyField(iget("Цель не задана"))
        if !defined $goal_id && $opt{required};

    return error_EmptyField(iget("Цель не задана"))
        if !defined $goal_id && (
            $strategy->isa('Direct::Strategy::AutobudgetAvgCpa')
            || $strategy->isa('Direct::Strategy::AutobudgetAvgCpaPerFilter')
            || $strategy->isa('Direct::Strategy::AutobudgetAvgCpaPerCamp')
            || $camp->campaign_type eq 'performance' && $strategy->isa('Direct::Strategy::AutobudgetWeekSum')
        );

    return if !defined $goal_id;

    return error_InvalidFormat(iget("Цель задана неверно"))
        if !($goal_id eq '0' || is_valid_id($goal_id));

    if (($strategy->isa('Direct::Strategy::AutobudgetAvgCpa')
        || $strategy->isa('Direct::Strategy::AutobudgetAvgCpaPerCamp')
        || $strategy->isa('Direct::Strategy::AutobudgetAvgCpaPerFilter')) && $strategy->pay_for_conversion) {
        if ($opt{has_cpa_pay_for_conversions_extended_mode_allowed}) {
            return error_NotFound(iget("Включена оплата за конверсии. Оптимизация по всем целям запрещена.")) if ($goal_id == 0);
        }
    }

    # кажется, тут нехорошее условие, цель всё равно нужно перепроверить
    my $old_strategy = $camp->get_strategy;
    if ($old_strategy && $old_strategy->name eq $strategy->name && $old_strategy->can('goal_id')) {
        return if $goal_id == $old_strategy->goal_id;
    }

    if ($goal_id == 0) {
        if ($strategy->can('pay_for_conversion')
            && $strategy->pay_for_conversion
            && $opt{has_cpa_pay_for_conversions_extended_mode_allowed}) {
            return error_NotFound(iget("Включена оплата за конверсии. Оптимизация по всем целям запрещена."))
        }

        if ($opt{has_disable_all_goals_optimization_for_dna_enabled}) {
            return error_NotFound(iget('Оптимизация по всем целям запрещена'));
        }
    }

    my $qcamp = {
        cid              => $camp->has_id ? $camp->id : undef,
        type             => $camp->campaign_type,
        strategy         => $camp->get_strategy_app_hash($strategy),
        strategy_name    => $strategy->name,
    };

    $qcamp->{metrika_counters} = join(',' => @{$camp->metrika_counters}) if $camp->has_metrika_counters;

    my $goal_result = CampaignTools::validate_autobudget_goal_id(
        $goal_id, $qcamp,
        pay_for_conversion        => $strategy->can('pay_for_conversion') && $strategy->pay_for_conversion,
        new_camp                  => $opt{new_camp},
        check_if_enough_stat      => 0,
        meaningful_goals => $opt{meaningful_goals},
        prefetched_goals => $opt{prefetched_goals},
        has_mobile_app_goals_for_text_campaign_allowed => $opt{has_mobile_app_goals_for_text_campaign_allowed},
        has_mobile_app_goals_for_text_campaign_strategy_enabled => $opt{has_mobile_app_goals_for_text_campaign_strategy_enabled},
        has_disable_all_goals_optimization_for_dna_enabled => $opt{has_disable_all_goals_optimization_for_dna_enabled},
        has_all_meaningful_goals_for_pay_for_conversion_strategies_allowed => $opt{has_all_meaningful_goals_for_pay_for_conversion_strategies_allowed}
    );

    return error_NotFound($goal_result) if $goal_result;

    return;
}

=head2 validate_pay_for_conversion

Проверка возможности переключиться на оплату за конверсии

=cut

sub validate_pay_for_conversion {
    my ($pay_for_conversion, $camp, $strategy, %opt) = @_;

    return if !defined $pay_for_conversion;

    if ($pay_for_conversion) {

        if ($camp->campaign_type eq "mobile_content") {
            return error_LimitAccess(iget("Оплата за конверсии в рекламе мобильных приложений недоступна"))
                if (!$opt{has_cpa_pay_for_conversions_mobile_apps_allowed});

        } else {
            if (!$opt{has_cpa_pay_for_conversions_extended_mode_allowed}) {
                return error_LimitAccess(iget("Оплата за конверсии недоступна"));
            }

            return error_BadUsage(iget("Оплата за конверсии возможна только в кампаниях типа ТГО, Смарт-баннеры, Динамические объявления"))
                if ($opt{has_cpa_pay_for_conversions_extended_mode_allowed} && $camp->campaign_type !~ m/^(text|performance|dynamic|cpm_banner)/);
        }

        my $avg_cpa_cpi = $camp->campaign_type eq 'mobile_content' ? $opt{avg_cpi} : $opt{avg_cpa};

        return error_InvalidField(iget("Необходимо указать среднюю цену конверсии"))
            if $opt{sum} && $opt{sum} > 0 && !$avg_cpa_cpi;
    }

    return;
}

=head2 validate_start

Проверка даты старта периода

Опции
 - has_edit_avg_cpm_without_restart_enabled - включена фича, позволяющая менять
      параметры стратегии (avg_bid, budget, finish) без рестарта

=cut

sub validate_start {
    my ($start, $camp, $strategy, %opt) = @_;
    if ( !defined $start ) {
        return error_ReqField();
    } elsif ( $start !~ /\S/ ) {
        return error_EmptyField(iget("Дата начала периода не задана"));
    } elsif ($start !~ /^\d{4}-\d{2}-\d{2}$/) {
        return error_InvalidFormat_IncorrectDate();
    } else {
        my ($error, $start_ts);
        try {
            $start_ts = ts_round_day( mysql2unix( $start ) );
        } catch {
            $error = 1;
        };
        if ( $error || ! $start_ts ) {
            return error_InvalidFormat_IncorrectDate();
        } else {
            my $now_ts = ts_round_day( time() );
            my $campaign_start_ts;
            try {
                $campaign_start_ts = ts_round_day( mysql2unix( $camp->start_date() ) );
            };
            my @exclude;
            if ($strategy->isa('Direct::Strategy::AutobudgetCpmCustomPeriodBase')) {
                push @exclude, 'auto_prolongation';
                if ($opt{has_edit_avg_cpm_without_restart_enabled}) {
                    push @exclude, ('avg_cpm', 'budget', 'finish');
                }
            }
            if ($strategy->isa('Direct::Strategy::AutobudgetAvgCpvCustomPeriodBase')) {
                push @exclude, 'auto_prolongation';
                if ($opt{has_edit_avg_cpm_without_restart_enabled}) {
                    push @exclude, ('avg_cpv', 'budget', 'finish');
                }
            }

            if ( $strategy->is_params_differs_from($camp->get_strategy(), @exclude) && $start_ts < $now_ts ) {
                return error_InvalidFormat(iget("Значение даты в поле #field# не может быть меньше текущей даты"));
            } elsif ( $campaign_start_ts &&  $start_ts < $campaign_start_ts ) {
                return error_InconsistentState(iget("Дата начала периода стратегии не может быть меньше даты начала кампании"));
            }
        }
    }
    return;
}

=head2 validate_finish

Проверка даты окончания периода

=cut

sub validate_finish {
    my ($finish, $camp, $strategy, %opt) = @_;

    if ( !defined $finish ) {
        return error_ReqField();
    } elsif ( $finish !~ /\S/ ) {
        return error_EmptyField(iget("Дата окончания периода не задана"));
    } elsif ($finish !~ /^\d{4}-\d{2}-\d{2}$/) {
        return error_InvalidFormat_IncorrectDate();
    } else {
        my ($finish_ts, $error);
        try {
            $finish_ts = ts_round_day( mysql2unix( $finish ) );
        } catch {
            $error = 1;
        };
        if ( $error || ! $finish_ts ) {
            return error_InvalidFormat_IncorrectDate();
        } else {
            return if !is_valid_date($strategy->start);
            my $start_ts = ts_round_day( mysql2unix( $strategy->start ) );
            my $now_ts = ts_round_day( time() );
            if ( $finish_ts < $start_ts ) {
                return error_InconsistentState( iget('Значение даты в поле #from# не может быть больше значения даты в поле #to#') );
            } else {
                my $days_count = scalar get_distinct_dates($strategy->start, $strategy->finish);
                if ($start_ts == $now_ts) {
                    $days_count--;
                }
                return error_InvalidFormat(iget("Для стратегий с произвольным периодом, период не может быть меньше 1 дня"))
                    if $days_count < $MIN_STRATEGY_PERIOD_DAYS_COUNT;
                return error_InvalidFormat(iget("Для стратегий с произвольным периодом, период не может быть больше %s дней", $MAX_STRATEGY_PERIOD_DAYS_COUNT))
                    if $days_count > $MAX_STRATEGY_PERIOD_DAYS_COUNT;
            }
            my $campaign_finish_ts;
            try {
                $campaign_finish_ts = ts_round_day( mysql2unix( $camp->finish_date() ) );
            };
            my @exclude;
            push @exclude, 'auto_prolongation' if $strategy->isa('Direct::Strategy::AutobudgetCpmCustomPeriodBase')
                || $strategy->isa('Direct::Strategy::AutobudgetAvgCpvCustomPeriodBase');

            if ( $strategy->is_params_differs_from($camp->get_strategy(), @exclude) && $finish_ts <= $now_ts ) {
                return error_InvalidFormat(iget("Значение даты в поле #field# не может быть меньше либо равным текущей дате"));
            } elsif ( $campaign_finish_ts &&  $finish_ts > $campaign_finish_ts ) {
                return error_InconsistentState(iget("Дата окончания периода стратегии не может быть больше даты окончания кампании"));
            }
        }
    }
    return;
}


1;

