package Direct::Validation::TargetInterests;

use Direct::Modern;

use base qw(Exporter);

use Settings;

use Yandex::I18n;

use Direct::Errors::Messages;
use Direct::Validation::Retargetings qw/validate_retargetings_bids/;
use Direct::TargetingCategories;

our @EXPORT = qw/
    validate_target_interests_for_adgroup
    /;

# Максимальное число ретаргетингов в группе
our $MAX_TARGET_INTERESTS_IN_ADGROUP = 50;

=head2 validate_target_interests_for_adgroup($checked_target_interests, $remaining_target_interests, $adgroup, $campaign)

Валидация списка ретаргетингов на группу.
Параметры:
    $checked_target_interests   -> список проверяемых (новых или изменившихся) таргетингов по интересам
    $remaining_target_interests -> список оставшихся (не изменившихся) таргетингов по интересам
    $adgroup        -> общая группа для таргетирования по интересам (Direct::Model::AdGroup)
    $campaign       -> общая кампания для таргетирования по интересам (HashRef)

=cut

sub validate_target_interests_for_adgroup {
    my ($checked_target_interests, $remaining_target_interests, $adgroup, $campaign) = @_;
    $remaining_target_interests //= [];

    my $vr_main = validate_retargetings_bids($checked_target_interests, $campaign);

    my $target_interests_limit = $MAX_TARGET_INTERESTS_IN_ADGROUP;
    my $target_interests_count = @$checked_target_interests + @$remaining_target_interests;
    my $active_target_interests_count = grep { !($_->has_is_suspended ? $_->is_suspended : 0) } (@$checked_target_interests, @$remaining_target_interests);

    if ($target_interests_count > $target_interests_limit) {
        $vr_main->add_generic(error_LimitExceeded(
                iget("Группа объявлений может содержать не более %d условий подбора аудитории (интересов)", $target_interests_limit)
            ));
    }

    # Если на этом этапе есть ошибки, то дальнейшие проверки не делаем.
    # Т.к. далее будет проверяться уникальность и прочее, и нужны как минимум корректно заполненные структуры.
    return $vr_main unless $vr_main->is_valid;

    # Проверка на корректность id-категорий нацеливания и уникальность проверяемых условий ретаргетинга
    my $target_categories = Direct::TargetingCategories->get_rmp_interests->items_by;

    my %checked_target_category_ids;
    for (@$checked_target_interests) {
        unless (exists $target_categories->{$_->target_category_id}) {
            $vr_main->add_generic(error_InvalidField(iget('Условия подбора аудитории (интерес) №%d не найдено', $_->target_category_id)));
            next;
        }
        if (!$target_categories->{$_->target_category_id}->{available}) {
            $vr_main->add_generic(error_InvalidField(iget('Условия подбора аудитории (интерес) №%d не доступно для выбора', $_->target_category_id)));
            next;
        }
        if ($checked_target_category_ids{ $_->target_category_id }++) {
            $vr_main->add_generic(error_InvalidField(iget("Условия подбора аудитории (интерес) должны быть уникальны")));
        }
    }
    return $vr_main unless $vr_main->is_valid;

    my %used_target_category_ids;

    # Добавим в хеш существующие условия ретаргетинга
    for my $target_interest (@$remaining_target_interests) {
        if ($used_target_category_ids{ $target_interest->target_category_id }++) {
            croak "target_category_id @{[$target_interest->target_category_id]} in adgroup @{[$target_interest->adgroup_id]} already duplicated";
        }
    }

    for (my $i = 0; $i < @$checked_target_interests; $i++) {
        my $target_interest = $checked_target_interests->[$i];
        my $vr = $vr_main->get_objects_results->[$i];

        if ($used_target_category_ids{ $target_interest->target_category_id }++) {
            $vr->add_generic(error_AlreadyExists(iget("Такое условие подбора аудитории (интерес) уже существует в группе")));
        }
    }

    return $vr_main;
}

1;
