package Direct::Validation::UserData;

use Direct::Modern;
use Mouse::Util::TypeConstraints;

use base qw/Exporter/;

use Yandex::I18n;
use Scalar::Util qw/blessed/;

use Direct::Validation::Errors;
use Direct::ValidationResult;

our @EXPORT_OK = qw/check_user_data/;

=head2 check_user_data($data, $constraint)

=cut

sub check_user_data {
    my ($data, $constraint, $user_var) = @_;
    _check_data_item(undef, 'root', 1, $data, $constraint, $user_var);
}

sub _check_data_item {
    my ($parent, $key, $is_exists, $value, $constraint, $user_var) = @_;

    my $validation_result = Direct::ValidationResult->new();

    if (ref($constraint) eq 'HASH') {
        if ($is_exists) {
            # Ожидаем хеш в $value
            if (ref($value) eq 'HASH') {
                while (my ($sub_key, $sub_constraint) = each %$constraint) {
                    # Выполним вложенную проверку
                    $validation_result->add($sub_key => _check_data_item($value, $sub_key, exists($value->{$sub_key}), $value->{$sub_key}, $sub_constraint, $user_var));
                }
            } else {
                $validation_result->add_generic(error_InvalidFormat(
                    iget('Переданное значение поля #field# имеет неверный формат (ожидается: #format#)'), field => "`$key`", format => 'HashRef',
                ));
            }
        } else {
            $validation_result->add_generic(error_ReqField(iget('Не передано обязательное поле #field#'), field => "`$key`"));
        }
    }

    elsif (ref($constraint) eq 'ARRAY' && !@$constraint) {
        if ($is_exists) {
            # Алиас для типа "ArrayRef"
            if (ref($value) ne 'ARRAY') {
                $validation_result->add_generic(error_InvalidFormat(
                    iget('Переданное значение поля #field# имеет неверный формат (ожидается: #format#)'), field => "`$key`", format => 'ArrayRef',
                ));
            }
        } else {
            $validation_result->add_generic(error_ReqField(iget('Не передано обязательное поле #field#'), field => "`$key`"));
        }
    }

    elsif (ref($constraint) eq 'ARRAY' && @$constraint <= 3) {
        # Цепочечная проверка:
        #   1-я проверка: на допустимость присутствия/отсутствия поля
        #   2-я проверка общая (при условии существования)
        #   3-я проверка умная: для массивов - проверка каждого элемента, для остальных - общая при условии defined
        for (my $i = 0; $i <= $#$constraint; $i++) {
            my $sub_constraint = $constraint->[$i];

            if ($i == 0) {
                # Проверка на допустимость присутствия/отсутствия
                if (!(
                    ref($sub_constraint)
                        ? (ref($sub_constraint) eq 'CODE')
                        : (!defined $sub_constraint || $sub_constraint =~ /^(?:0|1)?$/)
                )) {
                    croak "Constraint can be only coderef or boolean (incl undef)";
                }
                my $is_required = ref($sub_constraint) eq 'CODE' ? $sub_constraint->($parent, $user_var) : $sub_constraint;
                if ($is_required && !$is_exists) {
                    $validation_result->add_generic(error_ReqField(iget('Не передано обязательное поле #field#'), field => "`$key`"));
                    last;
                } elsif (!defined $is_required && $is_exists) {
                    # Поле не должно быть передано (не должно существовать)
                    $validation_result->add_generic(error_InvalidField(iget('Поля #field# не должно существовать'), field => "`$key`"));
                    last;
                }
            }
            elsif ($i == 1) {
                last if !$is_exists;

                # Общая проверка, при условии существования
                my $sub_vr = _check_data_item($parent, $key, 1, $value, $sub_constraint, $user_var);
                return $sub_vr if !$sub_vr->is_valid;
            }
            else {
                last if !defined $value;

                if (ref($value) eq 'ARRAY') {
                    # Workaround for ValidationResult constraints
                    my $nested_objects = $validation_result->nested_objects;
                    push @$nested_objects, _check_data_item($parent, $key, 1, $_, $sub_constraint, $user_var) for @$value;
                    for (my $i = 0; $i <= $#$nested_objects; $i++) { $nested_objects->[$i]->{position} = $i; }
                } else {
                    return _check_data_item($parent, $key, 1, $value, $sub_constraint, $user_var);
                }
            }
        }
    }

    elsif (ref($constraint) eq 'CODE') {
        if ($is_exists) {
            local $_ = $value;
            my $is_valid = $constraint->($value, $parent, $user_var);
            if (!$is_valid) {
                $validation_result->add_generic(error_InvalidFormat(iget('Переданное значение поля #field# имеет неверный формат'), field => "`$key`"));
            }
        } else {
            $validation_result->add_generic(error_ReqField(iget('Не передано обязательное поле #field#'), field => "`$key`"));
        }
    }

    elsif (
        (blessed($constraint) && $constraint->isa('Mouse::Meta::TypeConstraint')) ||
        (!ref($constraint) && defined $constraint && length $constraint)
    ) {
        if ($is_exists) {
            # Передан Mouse тип
            my $is_valid = Mouse::Util::TypeConstraints::find_or_create_isa_type_constraint($constraint)->check($value);
            if (!$is_valid) {
                $validation_result->add_generic(error_InvalidFormat(
                    iget('Переданное значение поля #field# имеет неверный формат (ожидается: #format#)'), field => "`$key`", format => $constraint,
                ));
            }
        } else {
            $validation_result->add_generic(error_ReqField(iget('Не передано обязательное поле #field#'), field => "`$key`"));
        }
    }

    elsif (!defined $constraint) {
        croak "No constraint passed";
    }

    else {
        croak "Unknown constraint passed".(ref($constraint) ? '' : ": $constraint");
    }

    return $validation_result;
}

1;
