package Direct::Validation::VCards;

use Direct::Modern;
use Carp;

use base qw(Exporter);

use Settings qw/
    $DISALLOW_BANNER_LETTER_RE
    $ALLOW_LETTERS
/;

use Yandex::I18n;
use Yandex::IDN qw/is_valid_email/;
use POSIX qw/floor/;
use List::MoreUtils qw/any/;

use Metro qw//;

use Direct::Validation::Errors;
use Direct::ValidationResult;


our @EXPORT_OK = qw/
    validate_vcards
    vcard_im_client_options
/;

=head2 vcard_im_client_options

ArrayRef возможных типов мессенджеров (icq, skype, jabber и т.д.) в виде строк

=cut

sub vcard_im_client_options { qw/icq skype mail_agent jabber MSN/ }

=head2 validate_vcards($vcards)

Валидация визиток.

Параметры:
    $vcards - ссылка на массив визиток [Direct::Model::VCard, ...]

Результат:
    Результат валидации (ValidationResult).
    Для каждой визитки ошибка хранятся по полям.

=cut

sub validate_vcards {
    my ($vcards) = @_;

    my $vr_main = Direct::ValidationResult->new();

    for my $vcard (@$vcards) {
        my $vcard_vr = $vr_main->next;

        #
        # name ("Название компании/ФИО")
        #
        my $name = $vcard->name;
        if (!defined $name || $name !~ /\S/) {
            $vcard_vr->add(name => error_ReqField());
        } else {
            if ($name =~ $DISALLOW_BANNER_LETTER_RE) {
                $vcard_vr->add(name => error_InvalidChars_AlphaNumPunct());
            }
            if (length($name) > 255) {
                $vcard_vr->add(name => error_MaxLength(undef, length => 255));
            }
        }

        #
        # country ("Страна")
        #
        my $country = $vcard->country;
        if (!defined $country || $country !~ /\S/) {
            $vcard_vr->add(country => error_ReqField());
        } else {
            if ($country !~ /^[${ALLOW_LETTERS}\- \(\)]+$/i) {
                $vcard_vr->add(country => error_InvalidChars_Alpha());
            }
            if (length($country) > 50) {
                $vcard_vr->add(country => error_MaxLength(undef, length => 50));
            }
        }

        #
        # city ("Город")
        #
        my $city = $vcard->city;
        if (!defined $city || $city !~ /\S/) {
            $vcard_vr->add(city => error_ReqField());
        } else {
            if ($city !~ /^[${ALLOW_LETTERS}\- \(\)]+$/i) {
                $vcard_vr->add(city => error_InvalidChars_Alpha());
            }
            if (length($city) > 55) {
                $vcard_vr->add(city => error_MaxLength(undef, length => 55));
            }
        }

        #
        # work_time ("Время работы")
        #
        my $work_time = $vcard->work_time;
        if (!defined $work_time || $work_time !~ /\S/) {
            $vcard_vr->add(work_time => error_ReqField());
        } else {
            $vcard_vr->add(work_time => validate_vcard_work_time($work_time));
        }

        #
        # phone ("Номер телефона")
        #
        my $phone = $vcard->phone;
        if (!defined $phone || $phone !~ /\S/) {
            $vcard_vr->add(phone => error_ReqField());
        } else {
            $vcard_vr->add(phone => validate_vcard_phone($phone));
        }

        #
        # contact_person ("Контактное лицо")
        #
        my $contact_person = $vcard->contact_person;
        if (defined $contact_person) {
            if ($contact_person !~ /\S/) {
                $vcard_vr->add(contact_person => error_ReqField());
            } else {
                if ($contact_person =~ $DISALLOW_BANNER_LETTER_RE) {
                    $vcard_vr->add(contact_person => error_InvalidChars_AlphaNumPunct());
                }
                if (length($contact_person) > 155) {
                    $vcard_vr->add(contact_person => error_MaxLength(undef, length => 155));
                }
            }
        }

        #
        # street ("Улица" пункта "Адрес")
        #
        my $street = $vcard->street;
        if (defined $street) {
            if ($street !~ /\S/) {
                $vcard_vr->add(street => error_ReqField());
            } else {
                if ($street =~ $DISALLOW_BANNER_LETTER_RE) {
                    $vcard_vr->add(street => error_InvalidChars_AlphaNumPunct());
                }
                if (length($street) > 55) {
                    $vcard_vr->add(street => error_MaxLength(undef, length => 55));
                }
            }
        }

        #
        # house ("Дом" пункта "Адрес")
        #
        my $house = $vcard->house;
        if (defined $house) {
            if ($house !~ /\S/) {
                $vcard_vr->add(house => error_ReqField());
            } else {
                if ($house =~ /[^\Q$ALLOW_LETTERS\E\\\/\-, ]/) {
                    $vcard_vr->add(house => error_InvalidChars_AlphaNum());
                }
                if (length($house) > 30) {
                    $vcard_vr->add(house => error_MaxLength(undef, length => 30));
                }
            }
        }

        #
        # building ("Корпус" пункта "Адрес")
        #
        my $building = $vcard->building;
        if (defined $building) {
            if ($building !~ /\S/) {
                $vcard_vr->add(building => error_ReqField());
            } else {
                if ($building =~ /[^\Q$ALLOW_LETTERS\E\\\/\-, ]/) {
                    $vcard_vr->add(building => error_InvalidChars_AlphaNum());
                }
                if (length($building) > 10) {
                    $vcard_vr->add(building => error_MaxLength(undef, length => 10));
                }
            }
        }

        #
        # apartment ("Офис" пункта "Адрес")
        #
        my $apartment = $vcard->apartment;
        if (defined $apartment) {
            if ($apartment !~ /\S/) {
                $vcard_vr->add(apartment => error_ReqField());
            } else {
                if ($apartment =~ $DISALLOW_BANNER_LETTER_RE) {
                    $vcard_vr->add(apartment => error_InvalidChars_AlphaNumPunct());
                }
                if (length($apartment) > 100) {
                    $vcard_vr->add(apartment => error_MaxLength(undef, length => 100));
                }
            }
        }

        #
        # extra_message ("Подробнее о товаре/услуге")
        #
        my $extra_message = $vcard->extra_message;
        if (defined $extra_message) {
            if ($extra_message !~ /\S/) {
                $vcard_vr->add(extra_message => error_ReqField());
            } else {
                if ($extra_message =~ $DISALLOW_BANNER_LETTER_RE) {
                    $vcard_vr->add(extra_message => error_InvalidChars_AlphaNumPunct());
                }
                if (length($extra_message) > 200) {
                    $vcard_vr->add(extra_message => error_MaxLength(undef, length => 200));
                }
            }
        }

        #
        # im_client ("Интернет-пейджер") && im_login ("Номер/имя для пейджера")
        # TODO: Подумать над заменой этих двух полей на одно поле, например contact_im, имеющий формат service:login
        #       skype:login, icq:uin, mailagent:email, и т.д.
        #       Также это позволит упростить валидацию
        #
        for (1) {
            my ($im_client, $im_login) = ($vcard->im_client, $vcard->im_login);
            last if !defined $im_client && !defined $im_login;

            if (defined $im_client && !is_im_client_valid($im_client)) {
                $vcard_vr->add(im_client => error_InvalidField(
                    iget('В поле #field# указан неподдерживаемый тип интернет-пейджера')
                ));
                last;
            }
            if (defined $im_client && !defined $im_login) {
                $vcard_vr->add(im_login => error_ReqField());
                last;
            }
            if (!defined $im_client && defined $im_login) {
                $vcard_vr->add(im_client => error_ReqField(
                    iget('В поле #field# не указан тип интернет-пейджера')
                ));
                last;
            }

            if ($im_client =~ /^icq$/i) {
                # В качестве UIN ICQ разрешаем циферки (не меньше 5, не больше 10) и дефисы.
                $vcard_vr->add(im_login => error_InvalidFormat(
                    iget('ICQ UIN в поле #field# должен содержать от 5 до 10 цифр и состоять из цифр и дефисов')
                )) unless ($im_login =~ s/\D//gr) =~ /^\d{5,10}$/ && $im_login =~ /^[\d\-]+$/;
            }
            elsif ($im_client =~ /^mail_agent$/i) {
                # Разрешаем email-адрес something@mail.ru
                $vcard_vr->add(im_login => error_InvalidFormat(
                    iget('Имя почтового ящика для Мail.Ru Агента в поле #field# должно содержать только email-адрес с доменами mail.ru, inbox.ru, bk.ru, list.ru')
                )) unless is_valid_email($im_login) && $im_login =~ /\@(?:mail|inbox|bk|list)\.ru$/i;
            }
            elsif ($im_client =~ /^jabber$/i) {
                # Разрешаем любой валидный email-адрес
                $vcard_vr->add(im_login => error_InvalidFormat(
                    iget('Jabber-идентификатор в поле #field# должен содержать только email-адрес')
                )) unless is_valid_email($im_login);
            }
            elsif ($im_client =~ /^(skype|msn)$/i) {
                # Разрешаем [a-z A-Z а-я А-Я .(точку) -(дефис) _(подчеркивание) @ 0-9]
                $vcard_vr->add(im_login => error_InvalidFormat(
                    iget('Skype-имя в поле #field# должно содержать email-адрес или имя, состоящее из букв, цифр и знаков пунктуации')
                )) unless $im_login =~ /^[$ALLOW_LETTERS\-\_\@]+$/ || is_valid_email($im_login);
            }
        }

        #
        # contact_email ("Контактный email")
        #
        my $contact_email = $vcard->contact_email;
        if (defined $contact_email) {
            if ($contact_email !~ /\S/) {
                $vcard_vr->add(contact_email => error_ReqField());
            } else {
                $vcard_vr->add(contact_email => error_InvalidFormat(
                    iget('Значение в поле #field# должно содержать только email-адрес')
                )) unless is_valid_email($contact_email, ('check_only_at_symbol' => 1));
            }
        }

        #
        # ogrn ("ОГРН/ОГРНИП")
        #
        my $ogrn = $vcard->ogrn;
        if (defined $ogrn) {
            if ($ogrn !~ /\S/) {
                $vcard_vr->add(ogrn => error_ReqField());
            } else {
                $vcard_vr->add(ogrn => validate_ogrn($ogrn));
            }
        }

        #
        # metro ("Станция метро")
        #
        my $metro = $vcard->metro;
        if (defined $metro) {
            if ($metro !~ /\S/) {
                $vcard_vr->add(metro => error_ReqField());
            } else {
                $vcard_vr->add(metro => validate_metro($metro, $vcard->city));
            }
        }

        #
        # x, y, x1, y1, x2, y2 ("Координаты на карте")
        # y -> широта (latitude), x -> долгота (longitude)
        #
        if (
            $vcard->isa('Direct::Model::VCard') &&
            $vcard->has_manual_point && $vcard->manual_point && $vcard->has_manual_bounds && $vcard->manual_bounds
        ) {
            my $manual_point_vr = Direct::ValidationResult->new();
            my $manual_bounds_vr = Direct::ValidationResult->new();

            my ($x, $y) = $vcard->manual_point =~ /^(-?\d+\.\d+),(-?\d+\.\d+)$/;
            my ($x1, $y1, $x2, $y2) = $vcard->manual_bounds =~ /^(-?\d+\.\d+),(-?\d+\.\d+),(-?\d+\.\d+),(-?\d+\.\d+)$/;

            $manual_point_vr->add(x => error_InvalidField())   if $x < -180  || $x > 180;
            $manual_point_vr->add(y => error_InvalidField())   if $y < -90   || $y > 90;
            $manual_bounds_vr->add(x1 => error_InvalidField()) if $x1 < -180 || $x1 > 180;
            $manual_bounds_vr->add(y1 => error_InvalidField()) if $y1 < -90  || $y1 > 90;
            $manual_bounds_vr->add(x2 => error_InvalidField()) if $x2 < -180 || $x2 > 180;
            $manual_bounds_vr->add(y2 => error_InvalidField()) if $y2 < -90  || $y2 > 90;

            $vcard_vr->add(manual_point => $manual_point_vr) unless $manual_point_vr->is_valid;
            $vcard_vr->add(manual_bounds => $manual_bounds_vr) unless $manual_bounds_vr->is_valid;
        }
    }

    return $vr_main;
}

=head2 validate_vcard_work_time($work_time)

Проверка правильности времени работы (work_time) в визитке.

Параметры:
    $work_time - строка с закодированным временем работы, вида "1#3#10#15#18#30;4#6#10#30#20#25"

Результат:
    Массив с ошибками (или пустой массив, если ошибок не было).

=cut

sub validate_vcard_work_time {
    my ($work_time) = @_;

    return [] unless defined $work_time;

    if ($work_time !~ /^(?:[0-6]\#[0-6]\#\d{1,2}\#\d{1,2}\#\d{1,2}\#\d{1,2};?)+$/) {
        return [error_InvalidFormat()];
    }

    my @errors;
    my %day_hash;
    my ($bad_format_flag, $cross_flag, $bad_time_flag) = (0, 0, 0);
    for (split ';', $work_time) {
        my ($d1, $d2, $h1, $m1, $h2, $m2) = split '#';
        if ($d1 =~ /^\d+$/ && $d2 =~ /^\d+$/) {
            my @days = $d1 <= $d2 ? ($d1 .. $d2) : (map { $_ % 7 } $d1 .. ($d2 + 7));
            for (@days) {
                if (!exists $day_hash{$_}) {
                    $day_hash{$_} = 1;
                } else {
                    push @errors, error_InvalidFormat(
                        iget('В поле #field# допускается указывать только один временной интервал для одного дня недели')
                    ) if !$cross_flag++;
                    last;
                }
            }
        }
        if ($h1 =~ /^\d+$/ && $m1 =~ /^\d+$/ && $h2 =~ /^\d+$/ && $m2 =~ /^\d+$/) {
            push @errors, error_InvalidFormat(
                iget('Во временном интервале в поле #field# допускается указывать часы от 0 до 23 и минуты от 0 до 59')
            ) if ($h1 > 23 || $h2 > 23 || $m1 > 59 || $m2 > 59) && !$bad_format_flag++;

            push @errors, error_InvalidFormat(
                iget('Во временном интервале в поле #field# допускается указывать минуты, кратные 15')
            ) if ($m1 % 15 || $m2 % 15) && !$bad_format_flag++;
        } else {
            push @errors, error_InvalidFormat() if !$bad_format_flag++;
        }
    }

    return \@errors;
}

=head2 validate_vcard_phone($phone)

Проверка правильности номера телефона (phone) в визитке.

Параметры:
    $phone - строка с закодированным номером телефона (с кодом страны, города, номера телефона, добавочным номером)

Результат:
    Массив с ошибками (или пустой массив, если ошибок не было).

=cut

sub validate_vcard_phone {
    my ($phone) = @_;

    return undef unless defined $phone;

    my $vr = Direct::ValidationResult->new();

    my @errors;
    my ($country_code, $city_code, $phone_short, $phone_ext) = split /\#/, $phone;

    $vr->add(country_code => error_InvalidField(iget('Не указан код страны в поле #field#')))
        if !defined $country_code || !length $country_code;

    my $isTurkey = defined $country_code && $country_code =~ /(\d+)/ && $1 == 90 ? 1 : 0;

    my $allow_empty_city_code = $isTurkey && $phone_short =~ /^444/ ? 1 : 0;

    $vr->add(city_code => error_InvalidField(iget('Не указан код города в поле #field#')))
        if (!defined $city_code || !length $city_code) && !$allow_empty_city_code;

    $vr->add(phone_short => error_InvalidField(iget('Не указан номер телефона в поле #field#')))
        if !defined $phone_short || !length $phone_short;

    return $vr unless $vr->is_valid;

    $vr->add(country_code => error_InvalidFormat(iget('Код страны в поле #field# должен содержать от 1 до 5 символов, начинаться со знака \'+\' или \'8\' и состоять из цифр')))
        if $country_code !~ /^\+?\d{1,4}$/;

    $vr->add(city_code => error_InvalidFormat(iget('Код города в поле #field# должен содержать от 1 до 5 символов, быть не равным 0 и состоять только из цифр')))
        if ($city_code !~ /^\d{1,5}$/ || $city_code eq '0') && !$allow_empty_city_code;

    $vr->add(phone_ext => error_InvalidFormat(iget('Добавочный номер в поле #field# должен содержать от 1 до 6 символов и состоять только из цифр')))
        if defined $phone_ext && length $phone_ext && $phone_ext !~ /^\d{1,6}$/;

    return $vr unless $vr->is_valid;

    # TODO: это, кажется, лишняя проверка, ее нужно перенести вверх, там, где проверяется код страны
    $vr->add(country_code => error_InvalidFormat(iget("Код страны в поле #field# должен начинаться со знака '+'")))
        if $country_code !~ /^\+/ && !(($country_code eq '8' && ($city_code == '800' || $city_code == '804')) || ($country_code eq '0' && $city_code == '800'));

    $vr->add(country_code => error_InvalidFormat(iget("В поле #field# код страны '%s' вместе с кодом города '%s' не может начинаться со знака '+'", 0+$country_code, $city_code)))
        if ($country_code eq '+8' && ( $city_code == 800 || $city_code == '804' )) || ($country_code eq '+0' && $city_code == '800');

    my $phoned = $phone_short;
    $phoned =~ s/\D//g;
    my $cityphone = ($country_code // '').($city_code // '').($phone_short // '');
    $cityphone =~ s/\D//g;

    $vr->add(phone_short => error_InvalidFormat(iget('В поле #field# указан неправильный формат номера телефона')))
        if $phone_short =~ /[^\d\- ]/ || $phoned !~ /^\d{5,9}$/ || (length($cityphone) < 8 || length($cityphone) > 17);

    return $vr->is_valid ? undef : $vr;
}

=head2 validate_ogrn($ogrn)

Проверка ОГРН/ОГРНИП значения.

    Основной государственный регистрационный номер состоит из 13 цифр. Последняя, тринадцатая, и есть контрольная.
    Делим число, состоящее из первых 12 знаков, на 11. Потом целую часть полученного результата умножаем на 11
    и сравниваем с первоначальным 12-значным числом. Разница должна совпадать с 13-й цифрой ОГРН.

    Допустим, приходит регистрационный номер 1037556120338.

    Проверяем:

    103755612033 : 11 = 9432328366,636
    9432328366 x 11 = 103755612026.
    103755612033 – 103755612026 = 7.

    Последняя цифра в заявленном ОГРН равна 8. Значит такого ОГРН не существует.
    Примеры существующих ОГРН:
    1087746113344
    1087746385320
    1055406282513
    1067760833810 (внимание, они настоящие, взятые из интернета)

    ОГРНИП - основной государственный регистрационный номер индивидуального предпринимателя.
    Состоит из 15 цифр, последняя - контрольная. От ОГРН отличается тем, что:
    1. Под номер записи в реестре выделено семь, а не пять цифр
    2. Контрольная цифра равна последней цифре остатка от деления на 13,а не на 11, предыдущего 14-значного числа
    Примеры существующих ОГРНИП:
    304540707500034
    304540220800032
    309774611900857
    310253706100022 (внимание, они настоящие, взятые из интернета)

Параметры:
    $ogrn - значение ОГРН/ОГРНИП

Результат:
    Массив с ошибками (или пустой массив, если ошибок не было).

=cut

sub validate_ogrn {
    my ($ogrn) = @_;

    return [] unless defined $ogrn;

    my $error = error_InvalidField();

    # Неправильно, если ОГРН содержит не только цифры или если длина не равна 13 или 15 цифрами.
    return [$error] if $ogrn !~ /^\d{13,15}$/ || !(length($ogrn) == 13 || length($ogrn)== 15);

    # Первая цифра ОГРН может быть только цифрами: 1, 2, 3, 5
    my $first_num = substr($ogrn, 0, 1);
    return [$error] if $first_num !~ /[1235]/;

    my $last_num = substr($ogrn, -1);

    my $mult = length($ogrn) == 13 ? 11 : 13;
    my $t1 = substr($ogrn, 0, -1);
    my $div = floor(int($t1) / $mult);
    my $t2 = $div * $mult;

    my $diff = abs($t1 - $t2);

    # Разница значений должна совпадать.
    # Иногда разница может быть больше 9, тогда надо брать только последний символ для сравнения.
    return [$error] if $last_num ne substr($diff, -1);

    return [];
}

=head2 validate_metro($metro, $city)

Проверка станции метро на существование.

Параметры:
    $metro - идентификатор станции метро
    $city  - идентификатор или название города, в котором ищем метро

Результат:
    Массив с ошибками (или пустой массив, если ошибок не было).

=cut

sub validate_metro {
    my ($metro, $city) = @_;

    return [] if $metro eq '0'; # 0 ("--не выбрана--")

    return [error_InvalidField(
        iget("В поле #city_field# не указано значение, необходимое для определения станции метро")
    )] unless defined $city;

    return [error_InvalidFormat()] unless $metro =~ /^\d+$/;

    return [] if any { $_->{region_id} == $metro } @{Metro::list_metro($city)};

    return [error_InvalidField(
        iget("В поле #field# указана станция, не существующая в выбранном городе")
    )];
}

=head2 is_im_client_valid($im_client_name)

По названию мессенджера проверяем валидно ли оно или нет, т.е. соответствует ли $IM_CLIENTS списку
undefined значение провоцирует exception
Возвращает 1 в случае успеха

=cut

sub is_im_client_valid {
    my $im_client_name = shift;
    croak "can't be undefined" unless defined $im_client_name;
    my $options = join('|', vcard_im_client_options());
    return ($im_client_name =~ /^($options)$/i) ? 1 : undef;
}

1;
