package Direct::Validation::Wallets;
use Direct::Modern;
use base qw/Exporter/;

use Settings;
use Currencies;
use Yandex::I18n;

use Direct::Validation::Errors;
use Direct::ValidationResult;

our @EXPORT_OK = qw/
    validate_autopay_sum_limits
/;

=head2 validate_autopay_sum_limits

	Проверяет ограничения на введенные размеры платежей и минимальных несгораемых сумм

=cut

sub validate_autopay_sum_limits {
	my ($wallet) = @_;

    my $vr_main = Direct::ValidationResult->new();

    if (!$wallet->has_autopay) {
    	$vr_main->add_generic(error_NotFound(iget("Автоплатеж не найден.")));
    	return $vr_main;
    }
    my $autopay = $wallet->autopay;

    my $min_autopay = Currencies::get_currency_constant($autopay->currency, 'MIN_AUTOPAY');
    my $max_autopay_card = Currencies::get_max_autopay_card($autopay->currency);
    my $max_autopay_yamoney = Currencies::get_currency_constant($autopay->currency, 'MAX_AUTOPAY_YAMONEY');
    my $max_autopay_remaining = Currencies::get_currency_constant($autopay->currency, 'MAX_AUTOPAY_REMAINING');
    my $currency_name = Currencies::get_currency_constant($autopay->currency, 'name');

    if (!defined $autopay->payment_sum) {
        $vr_main->add_generic(error_EmptyField(iget('Не указана cумма платежа')));
    }
    if (!defined $autopay->remaining_sum) {
        $vr_main->add_generic(error_EmptyField(iget('Не указана несгораемая сумма')));
    }
    if (!defined $autopay->paymethod_id) {
        $vr_main->add_generic(error_EmptyField(iget('Не выбран способ оплаты')));
    }
    if ($autopay->payment_sum < 0) {
        $vr_main->add_generic(error_InvalidField_NotPositive(iget('Сумма платежа должна быть положительным числом')));
    }
    if ($autopay->remaining_sum < 0) {
        $vr_main->add_generic(error_InvalidField_NotPositive(iget('Несгораемая сумма должна быть положительным числом')));
    }
    if ($max_autopay_card && $autopay->paymethod_type eq 'card' && $autopay->payment_sum > $max_autopay_card) {
        $vr_main->add_generic(error_AutopayLimit_CeilCard(iget('Сумма платежа не может быть больше %s %s для банковских карт', $max_autopay_card, $currency_name)));
    }
    if ($max_autopay_yamoney && $autopay->paymethod_type eq 'yandex_money' && $autopay->payment_sum > $max_autopay_yamoney) {
        $vr_main->add_generic(error_AutopayLimit_CeilYaMoney(iget("Сумма платежа не может быть больше %s %s для кошельков Яндекс.Денег", $max_autopay_yamoney, $currency_name)));
    }
    if ($min_autopay && $autopay->payment_sum < $min_autopay) {
		$vr_main->add_generic(error_AutopayLimit_Floor(iget('Сумма платежа не может быть меньше %s %s', $min_autopay, $currency_name)));	
	}
    if ($max_autopay_remaining && $autopay->remaining_sum > $max_autopay_remaining) {
        $vr_main->add_generic(error_AutopayLimit(iget('Сумма остатка не может быть больше %s %s', $max_autopay_remaining, $currency_name)));    
    }

	return $vr_main;
}

1;

