package Direct::ValidationResult;

use strict;
use warnings;
use utf8;

=head1 NAME
    
    Direct::ValidationResult - класс для хранения результата валидации массива объектов
    
=head1 SYNOPSIS

    use Direct::ValidationResult;
    use Direct::Validation::Errors qw/Generic Banner/;
    
    sub validate_title {
        
        my $title = shift;

        if ($title =~ /NOT_ALLOWED_SYMBOLS/) {
            return error_InvalidChars()
        }
        return undef;
    }
    
    sub validate_add_sitelinks {
        
        my $validation_result = new Direct::ValidationResult;
        foreach () {
            ...
        }
        
        return $validation_result;
    } 
    
    sub validate_add_banners {
        
        my $banners = shift;
    
        my $validation_result = new Direct::ValidationResult;
        foreach my $banner (@$banners) {
            
            my $banner_vr = $validation_result->next;
            
            $banner_vr->add(title => validate_title($banner->{title}));
            if (length $banner->{title} > 30) {
                $banner_vr->add(title => error_MaxLength());
            }
            # ошибка не относится к конкретному полю
            $banner_vr->add_generic(validate_geo($banner));
            
            # ValidationResult вложенный в ValidationResult
            $banner_vr->add(sitelinks => validate_add_sitelinks($banner->{sitelinks}, $banner->{href}));
        }
        
        if (@$banners > 50) {
            $validation_result->add_generic(
                error_BannersLimit(
                    iget('Превышено максимальное количество баннеров в группе')
                )
            )
        }
        
        return $validation_result;
    }
    
    my $banners = [$banner1, $banner2];
    my $banners_validation_result = validate_add_banners($banners); # ref Direct::ValidationResult
    
    unless ($banners_validation_result->is_valid) {
        
        my @response;
        # ref $object == 'Direct::ValidationResult'
        foreach my $object (@{$banners_validation_result->get_objects_results}) {
            my $source_banner = $banners->[$object->position];
            if ($object->is_valid) {
                # do some useful things
                push @response, {bid => $source_banner->{bid}};    
            } else {
                push @response, {
                    bid => $source_banner->{bid},
                    errors => $object->get_errors, # array []
                    warnings => $object->get_warnings, # array []
                };
            }
        }
        
        $banners_validation_result->get_generic_errors;  # array of Direct::Defect [error_]
        $banners_validation_result->get_generic_warnings;  # array of Direct::Defect [warning_]
        
        return @response;
    }
    
=head1 DESCRIPTION

    Класс для хранения результатов валидации объектов. Нужен для того, чтобы можно было создать и заполнить объект с результатами валидации,
    а затем интерпретировать эти результаты в вызывающем коде.

    Чтобы интерпретировать результат, можно:

    А: обойти содержимое и получить результаты для каждого исходного объекта
    Б: использовать helper-функцию для получения результата в более простой форме; например, is_valid возвращает булево значение "всё ли валидно?"
    
    Ошибки хранятся по именам полей объекта. Каждое поле может иметь более одной ошибки.
    Каждый проверяемый объект может содержать ошибки (более 1), не относящиеся ни к одному из полей объекта (->add_generic)
    Весь проверяемый набор объектов в целом может содержать ошибки (более 1) ->add_generic
    
=head1 METHODS

=cut

use Mouse;
use Carp;
use List::MoreUtils qw/part all notall any/;
use Scalar::Util qw/blessed/;

has 'position' => (is => 'rw', isa => 'Int', default => -1); 
has '_generic_errors' => (is => 'rw', isa => 'ArrayRef[Direct::Defect]', default => sub { [] });
has '_generic_warnings' => (is => 'rw', isa => 'ArrayRef[Direct::Defect]', default => sub { [] });
has 'defects_by_field' => (is => 'rw', isa => 'HashRef', default => sub { {} });
has 'nested_objects' => (is => 'rw', isa => 'ArrayRef[Direct::ValidationResult]', default => sub { [] });

=head2 next()

    Начало валидации нижеследующего объекта из массива проверяемых объектов.
    Возвращает новый объект Direct::ValidationResult, в котором необходимо сохранять результаты валидации
    (посредством вызова методов ->add ->add_generic)
    
    Результат:
        $validation_result - Direct::ValidationResult текущего объекта проверки,
            $validation_result->position вернет индекс текущего проверяемого объекта
            в общем массиве проверок.  
    
=cut

sub next {
    
    my $self = shift;
    
    my $position = scalar @{$self->nested_objects};
    my $vr;
    if (@_) {
        $vr = shift @_;
        $vr->position($position);
    } else {
        $vr = new Direct::ValidationResult(position => $position);
    }
    push @{$self->nested_objects}, $vr;
    return $vr;
}

=head2 get_nested_vr_by_index
    По заданному индексу возвращает вложеный vr, position при этом не меняется
     Результат:
        Direct::ValidationResult, соответствующий заданному индексу

=cut
    
sub get_nested_vr_by_index {
    my ( $self, $index ) = @_;
    
    return if $index > $#{$self->nested_objects} || $index < 0;
    return $self->nested_objects->[$index];
}

=head2 add($field_name => $defect)

    Добавление к текущему объекту проверки ошибки (или массива ошибок)
    
    Параметры:
        $field_name - имя поля, к которому относится ошибка
        $error - ошибка, может быть единичной ошибкой (тип Direct::Defect), массивом [Direct::Defect], объектом Direct::ValidationResult
                допустимые значения:
                - error_ReqField()
                - [error_InvalidGeo(), error_InvalidChars()]
                - $validation_result_2
                
                если к одному и тому же полю добавляется ошибка повторно (второй вызов ->add(field => ...))
                со значение $defect == Direct::ValidationResult, то будет брошено исключение, т.к. объект не знает как
                объединить текущую ошибку (сложная проверка Direct::ValidationResult) и уже имеющиеся ошибки [Direct::Defect]

=cut

sub add {

    my ($self, $field, $defect) = @_;
    
    croak 'unspecified field name' unless defined $field;
    
    return unless defined $defect;
    return if ref($defect) eq 'ARRAY' && !@$defect;
    
    if (ref $defect eq 'ARRAY') {
        croak 'all defects in array must be a Direct::Defect class' if notall { my $obj = $_; blessed($obj) && $obj->isa('Direct::Defect') } @$defect; 
    }
    
    my $field_vr = $self->defects_by_field->{$field};
    if ($field_vr && _is_instanceof_vr($defect)) {
        croak "don't know how to merge two Direct::ValidationResult";
    }
    
    unless ($field_vr) {
        if (_is_instanceof_vr($defect)) {
            $self->defects_by_field->{$field} = $defect;
            return;    
        }
        $field_vr = $self->defects_by_field->{$field} = new Direct::ValidationResult();
    }
    
    $field_vr->add_generic($defect);
    return $field_vr;
}

=head2 add_generic($defect)

    Добавление общей ошибки на текущий объект проверки. 
    Используется для случая, если ошибка относится к композиции полей или относится ко всему объекту в целом.
    
    Параметры:
        $defect - ошибка, может быть единичной ошибкой (тип Direct::Defect), массивом [Direct::Defect]

=cut  

sub add_generic {

    my ($self, $defect) = @_;

    return unless defined $defect;
    return if ref($defect) eq 'ARRAY' && !@$defect;

    my ($errors, $warnings) = part {$_->is_error ? 0 : 1} ref $defect eq 'ARRAY' ? @$defect : $defect;
    push @{$self->_generic_errors}, @$errors if $errors;
    push @{$self->_generic_warnings}, @$warnings if $warnings;
}


sub _is_instanceof_vr {
    
    my $object = shift;
    return blessed($object) && $object->isa('Direct::ValidationResult'); 
}

=head2 get_errors

    Возвращает массив ошибок на весь объект проверки (включая все вложенные объекты проверок)
    
    Результат:
        [] - массив ошибок, при отсутствии ошибок возвращается пустой массив []
                каждый элемент массива экземпляр класса Direct::Defect 
    
=cut

sub get_errors {
    my ($self) = @_;
    return $self->_get_defects('errors');
}

=head2 get_generic_errors

    Возвращает массив общих ошибок на объект проверки (НЕ включая все вложенные объекты проверок),
    добавленных через add_generic
    
    Результат:
        [] - массив ошибок, при отсутствии ошибок возвращается пустой массив []
                каждый элемент массива экземпляр класса Direct::Defect 

=cut

sub get_generic_errors {
    
    my $self = shift;
    return $self->_generic_errors;
}

=head2 get_warnings

    Возвращает массив варнингов на весь объект проверки (включая все вложенные объекты проверок)
    
    Результат:
        [] - массив варнингов, при отсутствии варнингов возвращается пустой массив []
                каждый элемент массива экземпляр класса Direct::Defect 

=cut

sub get_warnings {
    my $self = shift;
    return $self->_get_defects('warnings');
}

=head2 get_generic_warnings

    Возвращает массив общих варнингов на объект проверки (НЕ включая все вложенные объекты проверок),
    добавленных через add_generic
    
    Результат:
        [] - массив ошибок, при отсутствии ошибок возвращается пустой массив []
                каждый элемент массива экземпляр класса Direct::Defect 

=cut

sub get_generic_warnings {
    
    my $self = shift;
    return $self->_generic_warnings;
}



sub _get_defects {
    my ($self, $type, $walked_links) = @_;
    
    $walked_links ||= {};
    
    my @defects = @{$type eq 'warnings' ? $self->_generic_warnings() : $self->_generic_errors()};
    foreach my $field ($self->get_fields) {

        my $field_vr = $self->get_field_result($field);

        my $addr = Scalar::Util::refaddr($field_vr);
        croak 'a circular reference defined in validation result' if $walked_links->{$addr};
        $walked_links->{$addr} = 1;

        push @defects, @{$field_vr->_get_defects($type, $walked_links)};
    }

    foreach my $object (@{$self->get_objects_results}) {

        my $addr = Scalar::Util::refaddr($object);
        croak 'a circular reference defined in validation result' if $walked_links->{$addr};
        $walked_links->{$addr} = 1;

        push @defects, @{$object->_get_defects($type, $walked_links)};
    }

    return \@defects;
}


=head2 process_descriptions(%values)

    %values = (
        __generic => {
            'template_placeholder' => 'value',
            ..
        },
        __global => {
            'template_placeholder' => 'value',
        },
        'field_name' => {
            'template_placeholder' => 'value',
            ..
        }
    )

    Метод обрабабывает шаблоны в описаниях ошибок объектов.
    Со значениями из $values->{generic} для общих ошибок.
    $values->{$field_name} + $values->{__global} для ошибок по полю $field_name (если задано хотя бы одно из этих значений)

=cut

sub process_descriptions {
    my ($self, %values) = @_;

    $self->_process_errors_descriptions($self->_generic_errors, $values{__generic})
            if exists $values{__generic};
    my %global = exists $values{__global} ? %{$values{__global}} : ();

    foreach my $field ($self->get_fields) {
        
        my %field_values = (
            (exists $values{$field} ? %{$values{$field}} : ()),
            %global
        );
        
        my $field_vr = $self->get_field_result($field);
        $self->_process_errors_descriptions($field_vr->get_generic_errors, \%field_values);
        $self->_process_errors_descriptions($field_vr->get_generic_warnings, \%field_values);
        
        if (@{ $field_vr->get_objects_results }) {
            $field_vr->process_objects_descriptions(%values, __generic => \%field_values);
        } elsif (%{ $field_vr->defects_by_field }) {
            $field_vr->process_descriptions(%values);
        } 
    }
    if (@{$self->nested_objects}) {
        $_->process_descriptions(%values) for @{$self->nested_objects};
    }
    return;
}

=head2 process_objects_descriptions(%values)

    Для каждого поля все описания ошибок обрабатываются со значениями из $values{имя поля}

    Проходит по вложенным объектам (get_objects_results) и для каждого вызывает $object->process_descriptions(%values)
    Для всех общих ошибок вызывает TextTools::process_text_template со $values{__generic}

=cut

sub process_objects_descriptions {
    my ($self, %values) = @_;
    
    foreach my $object (@{$self->get_objects_results}) {
        $object->process_descriptions(%values);
    }
    return;
}

sub _process_errors_descriptions {
    my ($self, $errors, $values) = @_; # $self, [], {}
    
    foreach my $e (@$errors) {
        $e->process_description( $values );
    }
    return;
}

=head2 get_objects_results()

    Возвращает массив вложенных объектов проверки (созданных методом next).
    Каждый элемент массива это экземпляр класса Direct::ValidationResult
    содержащий все ошибки (и варнинги) валидации текущего объекта.
    
    Результат:
        $nested_objects - массив [] объектов класса Direct::ValidationResult    

=cut

sub get_objects_results {
    
    my $self = shift;
    return $self->nested_objects;
}

=head2 get_field_result($field)
    
    Возвращает результат проверки заданного поля.
    Для получения списка ошибок заданного поля нужно использовать get_errors/get_warnings
        get_field_result($field)->get_errors;
    
    Параметры:
        $field - имя поля (строка)
    
    Результат:
        $defect - объект класса Direct::ValidationResult,
                если к полю не были добавлены ошибоки (или варнинги), возвращается undef

=cut

sub get_field_result {
    
    my ($self, $field) = @_;
    return $self->defects_by_field->{$field};
}

=head2 get_fields()
    
    Возвращает массив полей объекта, которые содержат ошибки и/или варнинги.
    Для получения ошибок по полям нужно использовать метод get_field_result
    
    Результат
        @fields - массив строк с названиями полей
    
=cut

sub get_fields {
    
    my $self = shift;
    return sort keys %{$self->defects_by_field}; 
}

=head2 is_valid()

    Проверка валидности всего объекта в целом (включая все подобъекты) 
    
=cut

sub is_valid {
    
    my $self = shift;
    return @{$self->get_errors} == 0;
}

=head2 has_only_warnings()

    Результат проверки содержит только варнинги (т.е. валиден) 
    
=cut

sub has_only_warnings {
    
    my $self = shift;
    return $self->is_valid && @{$self->get_warnings} > 0;
}

=head2 get_error_texts

    Возвращает тексты всех ошибок (варнинги пропускаются).

    Результат:
        [] - массив строк с текстами ошибок
            при отсутствии ошибок возвращается пустой массив [] (при этом
            объект проверки может содержать варнинги)

=cut

sub get_error_texts {
    my $self = shift;
    return [map {$_->text} @{$self->get_errors}];
}

=head2 get_error_descriptions

    Возвращает текстовые описания всех ошибок (варнинги пропускаются).
    Используется для совместимости с существующими функциями валидации.

    Результат:
        [] - массив строк с текстами ошибок
            при отсутствии ошибок возвращается пустой массив [] (при этом
            объект проверки может содержать варнинги)

=cut

sub get_error_descriptions {
    my $self = shift;
    return [map {$_->description} @{$self->get_errors}];
}


=head2 get_warning_descriptions

    Возвращает текстовые описания всех предупреждений (ошибки пропускаются).

    Результат:
        [] - массив строк с текстами предупреждений
            при отсутствии предупреждений возвращается пустой массив [] (при этом
            объект проверки может содержать ошибки)

=cut

sub get_warning_descriptions {
    my $self = shift;
    return [map {$_->description} @{$self->get_warnings}];
}


=head2 get_first_error_text

    Возвращает первый встретившийся текст ошибки (варнинги пропускаются).
    
    Результат:
        $text - строка с текстовым описание ошибки
                при отсутствии ошибок (но возможном наличии варнингов) возвращается undef

=cut

sub get_first_error_text {
    my $self = shift;

    my $errors = $self->get_error_texts;
    return @$errors > 0 ? $errors->[0] : undef;
}

=head2 get_first_error_description

    Возвращает первое встретившееся описание ошибки.
    Используется для совместимости с существующими функциями валидации.

    Результат:
        $text - строка с текстовым описание ошибки
                при отсутствии ошибок (но возможном наличии варнингов) возвращается undef

=cut

sub get_first_error_description {
    my $self = shift;

    my $errors = $self->get_error_descriptions;
    return @$errors > 0 ? $errors->[0] : undef;
}

=head2 one_error_text_by_objects

    Возвращает массив текстов ошибок, по одной ошибке от каждого проверяемого объекта

    Результат:
        [] - массив строк с текстами ошибок
            при отсутствии ошибок возвращается пустой массив [] (при этом объект проверки может содержать варнинги)

=cut

sub one_error_text_by_objects {
    my $self = shift;

    my @errors;
    foreach my $object (@{$self->get_objects_results}) {
        push @errors, $object->get_first_error_text unless $object->is_valid;
    }
    return \@errors;
}

=head2 one_error_description_by_objects

    Возвращает массив описаний ошибок, по одной ошибке от каждого проверяемого объекта
    Используется для совместимости с существующими функциями валидации. 

    Результат:
        [] - массив строк с текстами ошибок
            при отсутствии ошибок возвращается пустой массив [] (при этом объект проверки может содержать варнинги)

=cut

sub one_error_description_by_objects {
    my $self = shift;

    my @errors;
    foreach my $object (@{$self->get_objects_results}) {
        push @errors, $object->get_first_error_description unless $object->is_valid;
    }
    return \@errors;
}

=head2 bind_errors_to_data

=cut
our $_id_map;

sub bind_errors_to_data {
    my ($self, $data, $id_map) = @_;

    my $errors = $self->convert_to_hash;
    local $_id_map = $id_map // {};

    return $self->_bind_data($errors, $data);
}

sub _stringify_blessed {
    my ($self, $val ) = @_;
    
    return $val unless ref $val;
    if (blessed $val) {
        return $val->id if $val->can('id');
        return $val->to_string if $val->can('to_string');
        warn 'Couldn`t stringify: ', blessed $val;
        return 'object';
    }
    
    return [map { $self->_stringify_blessed($_) } @$val ] if ref $val eq 'ARRAY';
    return { map { $_ => $self->_stringify_blessed($val->{$_}) }  keys %$val } if ref $val eq 'HASH';
}

sub _bind_data {
    my ($self, $errors, $data, $path, $object_name) = @_;
    
    return $errors if ref $errors eq 'ARRAY';
    my $errors_hash = $errors;   
    
    $path //= '/';
    my @plain_errors;
    
    foreach my $field (keys %$errors_hash) {
        
        if ($field eq 'objects_results') {
            my $i = 0;
            foreach my $item ( @{$errors_hash->{$field}}){
                my $id = $data->[$i]->{id} // $data->[$i]->{$_id_map->{$object_name} // 'id'};
                push @plain_errors, map { $_->{'_'.$object_name.'_id'} = $id; $_ } @{ $self->_bind_data($item, $data->[$i], $path.'/'.$i++, $object_name) };
            }
        }
        elsif( $field eq 'generic_errors' ) {
            push @plain_errors, map {$_->{_path} = $path; $_} @{$errors_hash->{$field}};
        }
        elsif( ref $errors_hash->{$field} eq 'ARRAY' ) {
            my @errors = map {$_->{_data} = $self->_stringify_blessed($data->{$field}); $_->{_path} = $path; $_} @{$errors_hash->{$field}};
            push @plain_errors, @errors;
        }
        else {
            push @plain_errors,  @{ $self->_bind_data($errors_hash->{$field}, $data->{$field}, $path.'/'.$field, $field) };
        }
    }
    
    return \@plain_errors;    
}


=head2 convert_to_hash

=cut

sub _extract_errors_hash {
    my ($self, $kind_of_errors) = @_;
    
    my $method = 'get_'.$kind_of_errors;
    my @hashs = map { +{%$_} } @{$self->$method};
    
    return @hashs;
}

sub _extract_generic {
    my $self = shift;
    
    my @generic_errors = (
       $self->_extract_errors_hash('generic_errors'),
       $self->_extract_errors_hash('generic_warnings'),
    );

    return \@generic_errors;
}


sub convert_to_hash {
    
    my $self = shift;

    my %hash;
    my $generic_errors = $self->_extract_generic();

    $hash{generic_errors} = $generic_errors if @$generic_errors;
    
    my %defects_by_field;
    my $count_empty = 0;
    foreach my $field ($self->get_fields) {
        ++$count_empty if $self->get_field_result($field)->is_valid && !$self->get_field_result($field)->has_only_warnings;
        $defects_by_field{$field} = $self->get_field_result($field)->convert_to_hash;
    }
    %hash = (%hash, %defects_by_field) unless $count_empty == scalar keys %defects_by_field;  
    
    my @objects_results;
    $count_empty = 0;
    foreach my $object_vr (@{$self->get_objects_results}) {
        ++$count_empty if $object_vr->is_valid && !$object_vr->has_only_warnings;
        push @objects_results, $object_vr->convert_to_hash;
    }
    $hash{objects_results} = \@objects_results unless $count_empty == @objects_results;
    
    return exists $hash{generic_errors} && scalar(keys %hash) == 1
                ? $hash{generic_errors}
                : \%hash  
}

{
    my %fields_translation = (
        adgroup_name => 'group_name',
        dynamic_condition => 'dynamic_conditions',
    );

=head2 convert_vr_for_frontend

Конвертируем ValidationResult в хеш понятный фронтенду

=cut

sub convert_vr_for_frontend {
    my ($vr, $field, $complex_fields) = @_;

    my $is_complex_field = $field && exists $complex_fields->{$field};
    my $generic_errors = [map { +{%$_} } @{$vr->get_generic_errors}];
    my $looked_errors;
    if (@{$vr->get_objects_results}) {
        my $array_errors = [];
        for (@{$vr->get_objects_results}) {
            push @$array_errors, convert_vr_for_frontend($_, $field, $complex_fields);
        }
        if (any {$_} @$array_errors) {
            $looked_errors->{array_errors} = $array_errors;
        }
        if (@$generic_errors) {
            $looked_errors->{generic_errors} = $generic_errors;
        }
    } elsif ($is_complex_field) {
        my $object_errors = {};
        for my $field ($vr->get_fields) {
            my $attr = exists $fields_translation{$field} ? $fields_translation{$field} : $field;
            $object_errors->{$attr} = convert_vr_for_frontend($vr->get_field_result($field), $field, $complex_fields);
        }
        if (keys %$object_errors) {
            $looked_errors->{object_errors} = $object_errors;
        }
        if (@$generic_errors) {
            $looked_errors->{generic_errors} = $generic_errors;
        }
    } elsif (@{$vr->get_errors}) {
        push @$looked_errors,  map { +{%$_} } @{$vr->get_errors};
    }
    return $looked_errors;
}}



__PACKAGE__->meta->make_immutable;

1;
