package Direct::VideoAdditions;

use Direct::Modern;

use Mouse;

use Settings;
use Yandex::DBShards qw/get_new_id/;

use JSON;
use List::MoreUtils qw/any part/;
use BS::ResyncQueue qw/bs_resync/;
use Yandex::ListUtils qw/xuniq/;
use BannerStorage qw/receive_video_addition_creatives generate_creatives/;
use Encode qw/decode_utf8/;

use Direct::Creatives;
use Yandex::Trace qw//;
use Yandex::DBTools;
use Yandex::TVM2;

use Direct::Model::VideoAddition;

has 'items' => (is => 'ro', isa => 'ArrayRef[Direct::Model::VideoAddition]');

around BUILDARGS => sub { my ($orig, $class) = (shift, shift); $class->$orig(@_ == 1 ? (items => $_[0]) : @_) };

=head2 get_by


=cut

sub get_by {
    my ($class, $key, $vals, $uid, %options) = @_;

    croak "only `creative_id|uid` keys are supported" unless $key =~ /^creative_id|uid$/;

    $vals = [$vals // ()] if ref($vals) ne 'ARRAY';
    return $class->new([]) if !@$vals;

    my $filter = {
        'pc.creative_type' => 'video_addition',
    };
    if ($key eq 'creative_id') {
        $filter->{'pc.creative_id'} = $vals;
    }

    if (exists $options{'text_video'}) {
        $filter->{'pc.layout_id'} = $Direct::Model::VideoAddition::COMPATIBLE_LAYOUT_IDS{'text'};
    }

    my $creatives = Direct::Creatives::_get_creatives($uid, $filter)->{creatives};
    return $class->new($creatives);
}

=head2 items_by

=cut

sub items_by
{
    my ($self, $key) = @_;
    return { map { $_->$key => $_ } @{$self->items} };
}

=head2 get_video_additions_from_bs

Получает из bannerstorage креативы

=cut

sub get_video_additions_from_bs {
    my ($class, $client_id, $bs_creatives) = @_;

    my @video_additions;
    for my $bs_creative (@{$bs_creatives}) {
        push @video_additions, Direct::Model::VideoAddition->new(
                id => $bs_creative->{creative_id},
                resource_type => 'creative',
                name => $bs_creative->{creative_name},
                preview_url => $bs_creative->{preview_url},
                live_preview_url => $bs_creative->{live_preview_url},
                client_id => $client_id,
                yacontext_categories => $bs_creative->{yacontext_categories},
                duration => $bs_creative->{duration},
                layout_id => $bs_creative->{preset_id},
                _additional_data => undef,
            );
    }
    return \@video_additions;
}

=head2 recieve_video_additions

Получает из bannerstorage видео-дополнения

=cut

sub recieve_video_additions {
    my ($class, $client_id, $creative_ids) = @_;
    my $bs_creatives = BannerStorage::receive_video_addition_creatives($client_id, $creative_ids);

    my ($ok_creatives, $missed_creatives) = part {$_->{ok} ? 0 : 1} @$bs_creatives;
    return {missed => $missed_creatives, recieved => [map {$_->{creative}} @{$ok_creatives}]};
}

=head2 save_video_additions

Получает из bannerstorage креативы и сохранение их в базе

=cut

sub save_video_additions {
    my ($class, $operator_uid, $client_id, $creatives) = @_;

    my ($push_result, $push_error) = Direct::VideoAdditions->push_creatives(
        $operator_uid,
        $client_id,
        $creatives);
    if ($push_error || any {$_ eq 'error'} map {$_ => lc($_->{status})} @{$push_result}) {
        my $error = decode_utf8($push_error ? $push_error : join( ', ', map {$_->{message} . '|'} @{$push_result}));
        die sprintf('intapi push error: operator_uid %s, client_id %s, creatives: %s - %s',
                $operator_uid,
                $client_id,
                join( ', ', map {$_->{creative_id}} @{$creatives}),
                $error);
    }
}

=head2 generate_video_additions

Генерация креатива видео-дополнения

=cut

sub generate_video_additions {
    my ($class, $client_id, $conditions) = @_;
    my @generated_video_additions;
    $_->{creative_type} = 'videoAddition' foreach @{$conditions};
    my $generate_response = BannerStorage::generate_creatives($client_id, $conditions);

    # прописываем всем полученым дополнениям категории из запроса, чтобы потом связать с баннерами
    # с теми же категориями

    my $i = 0;
    for my $condition (@$conditions) {
        for my $j (0 .. $condition->{count} - 1) {
            $generate_response->[$i]->[$j]->{yacontext_categories} = $condition->{category_ids};
        }
        $i++;
    }

    for my $creatives (@{$generate_response}) {
        push @generated_video_additions, @{$creatives};
    }

    return \@generated_video_additions;
}

=head2 enqueue_auto_resources

Постановка в очередь задания на генерацию видеоресурсов

=cut

sub enqueue_auto_resources
{
    my ($class, $client_id, $cid, $action, %options) = @_;
    my $queue = Yandex::DBQueue->new(PPC(ClientID => $client_id), 'set_auto_resources');
    $queue->insert_job({
        job_id => get_new_id('job_id'),
        ($options{UID} ? (uid => $options{UID}) : ()),
        ClientID => $client_id,
        args => {
            cid => $cid,
            action => $action,
            ($options{bid} ? (bid => $options{bid}) : ()),
            ($options{creative_id} ? (creative_id => $options{creative_id}) : ()),
        },
    });
}

=head2 push_creatives($operator_uid, $ClientID, $creatives)

Создает переданные видео дополнения через ручку intapi-java

=cut

sub push_creatives {
    my ($class, $operator_uid, $ClientID, $creatives) = @_;

    state ($log, $json);
    $log //= Yandex::Log->new(
        no_log => 1,
        use_syslog => 1,
        syslog_prefix => 'INTAPI-JAVA',
        log_file_name => "VideoAdditions.log",
    );
    $json //= JSON->new->allow_unknown(1)->allow_blessed(1)->convert_blessed(1);

    my $url_params = {operator_uid => $operator_uid, client_id => $ClientID};
    my $url = Yandex::HTTP::make_url(
        $Settings::DIRECT_INTAPI_URL . 'DisplayCanvas/upload_creatives',
        $url_params,
    );

    my @unique_creatives = xuniq { $_->{creative_id} } @$creatives;

    $log->out(['REQUEST:', 'POST', $url, $url_params, \@unique_creatives]);

    my $ticket = eval{Yandex::TVM2::get_ticket($Settings::TVM2_APP_ID{intapi})} or $log->die("Cannot get ticket for $Settings::TVM2_APP_ID{intapi}: $@");
    my $profile = Yandex::Trace::new_profile('video_additions:push_creatives', obj_num => scalar @unique_creatives);
    my $response = Yandex::HTTP::http_parallel_request(POST => { 1 => {
                url => $url,
                body => encode_json(\@unique_creatives),
            }},
        timeout => 15, log => $log,
        headers => { 'Content-Type' => 'application/json', 'Accept' => '*/*', 'X-Ya-Service-Ticket' => $ticket},
    )->{1};

    my ($result, $error);
    if ($response->{is_success}) {
        utf8::decode($response->{content});
        $result = $json->decode($response->{content})->{uploadResults} if $response->{content};
    }
    else {
        $error = $response->{content} || ('Status: ' . $response->{headers}->{Status} . ' (' . $response->{headers}->{Reason} . ')');
    }

    $log->out(['RESPONSE:', $result // $error, ( $error ? ( $response->{headers}->{Reason}) : () ) ]);

    return $result, $error;
}

=head2 resend_to_bs

Для указанного клиента переотправить все баннеры с видео-дополнениями в БК
Параметры:
    $client_id

=cut

sub resend_to_bs
{
    my ($class, $client_id, $priority) = @_;
    $priority //= $BS::ResyncQueue::PRIORITY_DEFAULT;
    my $rows = get_all_sql(PPC(ClientID => $client_id), [
        "SELECT cid, bid, ? as priority
        FROM banners_performance bp
        JOIN perf_creatives perfc ON perfc.creative_id = bp.creative_id AND perfc.creative_type = 'video_addition'",
        where => {
            'perfc.ClientID' => $client_id,
            'bp.statusModerate' => 'Yes',
        },
    ], $priority);
    bs_resync($rows);
}

1;

