package Direct::Wallets;

use Direct::Modern;

use Mouse;
use List::MoreUtils qw/none any/;

use Settings;
use PrimitivesIds;
use RBACElementary;
use Client;
use Campaign::Types;
use Wallet;

use Yandex::I18n;
use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::HashUtils;
use Yandex::ListUtils qw/xsort/;
use Yandex::Balance::Simple qw/balance_simple_list_payment_methods
                               balance_simple_unbind_card/;

use Direct::Model::Wallet;
use Direct::Model::Wallet::Manager;
use Direct::Wallets::Payment;

has 'items' => (is => 'ro', isa => 'ArrayRef[Direct::Model::Wallet]');

=head2 get_by

По заданному критерию возвращает список кошельков (моделей Direct::Model::Wallet).

    my $feeds = Direct::Wallets->get_by(campaign_id => $wallet_cid, %opts);

    На входе:
        class -> Direct::Wallets
        key - ключ для поиска, доступно пока только campaign_id
        vals - значение для поиска
        %opts:
            filter - HashRef; произвольный фильтр в формате Yandex::DBTools
            no_additional - не подгружать дополнительные данные:
                                * статус последней транзакции автоплатежа
    На выходе:
        объект Direct::Wallets с полями:
            items - ArrayRef(Direct::Model::Wallet) с данными о кошельках.

=cut
sub get_by {
    my ($class, $key, $vals, %opts) = @_;

    croak "only `campaign_id`, `client_id` keys are supported" unless $key =~ /^(?:campaign|client)_id$/;

    $vals = [$vals // ()] if ref($vals) ne 'ARRAY';
    return $class->new(items => []) if !@$vals;

    my @select_columns = (
        Direct::Model::Wallet->get_db_columns('wallet_campaigns' => 'wc', prefix => 'wc_'),
        Direct::Model::Wallet->get_db_columns('campaigns' => 'c', prefix => 'wc_'),
        Direct::Model::Wallet->get_db_columns('camp_options' => 'co', prefix => 'wc_'),
        Direct::Model::Autopay->get_db_columns('autopay_settings' => 'a', prefix => 'a_'),
    );

    my %shard_selector = (campaign_id => 'cid', client_id => 'ClientID');
    my $rows = get_all_sql(PPC($shard_selector{$key} => $vals), [
        "SELECT", join(", ", @select_columns),
        "FROM campaigns c
            JOIN camp_options co ON co.cid = c.cid
            JOIN wallet_campaigns wc ON wc.wallet_cid = c.cid
            LEFT JOIN autopay_settings a ON a.wallet_cid = wc.wallet_cid
        ",
        WHERE => [
            %{$opts{filter} // {}},
            'c.type' => 'wallet',
            'c.'.$shard_selector{$key} => SHARD_IDS,
        ],
    ]);

    my $wallets = Direct::Model::Wallet->from_db_hash_multi($rows, prefix => 'wc_');
    if (!$opts{no_additional}) {
        my $autopays = Direct::Model::Autopay->from_db_hash_multi([grep {defined $_->{a_paymethod_id}} @$rows], prefix => 'a_');
        $class->_enrich_with_last_transaction($autopays);

        my %autopays_hash = map {$_->id => $_} @$autopays;
        foreach (@$wallets) {
            if (defined $autopays_hash{$_->id}) {
                $_->autopay($autopays_hash{$_->id});
            }
        }
    }

    return $class->new(items => $wallets);
}

=head2 items_by($self, $key)

    Группирует загруженные общие счета по ключу $key и возвращает их в виде хеша.

    Поддерживаются ключи campaign_id и client_id.
    Если группировка по client_id, то в значениях хеша возвращаются массивы общих счетов,
    т.к. у клиента их может быть несколько.

=cut
sub items_by {
    my ($self, $key) = @_;

    croak "only `campaign_id`, `client_id` keys are supported" unless $key =~ /^(?:campaign|client)_id$/;

    if ($key eq 'client_id') {
        my %result;

        push @{$result{$_->client_id}}, $_ for @{$self->items};
        return \%result;
    } elsif ($key eq 'campaign_id') {
        return {
            map { $_->id => $_ } @{$self->items}
        };
    }
}

sub _enrich_with_last_transaction {
    my ($class, $autopays) = @_;

    foreach my $autopay (@$autopays) {
        next if $autopay->tries_num == 0;
        # Если trues_num отрицательный, значит транзакции вовсе не создаются из-за баланса.
        if ($autopay->tries_num < 0) {
            $autopay->last_transaction_status('Error');
            next;
        }
        my $last_autopay_transaction = Direct::Wallets::Payment::get_last_wallet_transaction($autopay->id);
        $autopay->last_transaction_status($last_autopay_transaction->{balance_status_code});
    }

}

=head2 get_list_payment_methods

    Узнать все доступные методы оплаты для пользователя
    На входе:
        class -> Direct::Wallets
        uid - паспортный UID
        ip - IP пользователя
        rbac - RBAC-объект
        %opts:
            this_user_only - не надо представителей смотреть, только данного клиента
    На выходе:
        HashRef с полями:
            cards - ArrayRef(HashRef) с данными о картах.
            yandex_money - ArrayRef(HashRef) с данными о Яндекс.Кошельках.

=cut

sub get_list_payment_methods {
    my ($class, $rbac, $uid, $ip, %opts) = @_;

    my (@cards, @yandex_money);

    my $client_id = get_clientid(uid => $uid);

    my $rep_uids = ($opts{this_user_only}) ? [$uid] : rbac_get_client_uids_by_clientid($client_id);

    my $currency = get_client_currencies($client_id)->{work_currency};

    foreach my $rep_uid (@$rep_uids) {
        my $payment_methods_result = balance_simple_list_payment_methods({uid => $rep_uid, user_ip => $ip});
        return {error => iget("Не удалось получить список привязанных способов оплаты")} unless $payment_methods_result->{status} eq 'success' &&
                                                                                                defined $payment_methods_result->{payment_methods};
        my $payment_methods = $payment_methods_result->{payment_methods};
        foreach my $payment_method_key (keys %$payment_methods) {
            my $payment_method = {};
            $payment_method->{paymethod_id} = $payment_method_key;
            $payment_method->{paymethod_type} = $payment_methods->{$payment_method_key}->{type};
            $payment_method->{allowed_set_autopay} = ($uid == $rep_uid) ? 1 : 0;
            $payment_method->{min_payment_sum} = Currencies::get_currency_constant($currency, "MIN_AUTOPAY");
            $payment_method->{max_remaining_sum} = Currencies::get_currency_constant($currency, "MAX_AUTOPAY_REMAINING");

            hash_copy $payment_method, $payment_methods->{$payment_method_key}, qw/number/;

            if ($payment_methods->{$payment_method_key}->{type} eq 'card') {
                $payment_method->{max_payment_sum} = Currencies::get_max_autopay_card($currency);
                $payment_method->{binding_ts} = $payment_methods->{$payment_method_key}->{binding_ts};
                push @cards, $payment_method;
            }
            if ($payment_methods->{$payment_method_key}->{type} eq 'yandex_money') {
                $payment_method->{max_payment_sum} = Currencies::get_currency_constant($currency, "MAX_AUTOPAY_YAMONEY");
                push @yandex_money, $payment_method;
            }
        }
    }

    return {cards => [ xsort {$_->{number}} @cards], yandex_money => \@yandex_money};

}

=head2 unbind_card

    Отвязка карты пользователя. Если на карту был настроен автоплатеж, он отключается, данные о нем стираются.
    На входе:
        class -> Direct::Wallets
        uid   -> uid пользователя
        payment_id -> ID карты
        ip    -> IP пользователя
        session_id -> Сессия клиента
    На выходе:
        1 - произошли проблемы
        0 - отвязка прошла успешно


=cut

sub unbind_card {
    my ($class, $uid, $paymethod_id, $ip, $session_id, $host) = @_;

    my @select_columns = Direct::Model::Autopay->get_db_columns_list('autopay_settings');
    my $rows = get_all_sql(PPC(uid => $uid), [
        "SELECT", join(", ", map { sql_quote_identifier($_) } @select_columns),
        "FROM autopay_settings",
        WHERE => {payer_uid => $uid, paymethod_id => $paymethod_id}
    ]);

    my $unbind_card_result = balance_simple_unbind_card({card => $paymethod_id, user_ip => $ip, session_id => $session_id, host => $host});
    return 1 if $unbind_card_result->{status} ne 'success';

    my $autopays = Direct::Model::Autopay->from_db_hash_multi($rows);

    my $wallets = Direct::Wallets->get_by(campaign_id => [map {$_->id} @$autopays])->items;
    $_->autopay_mode('none') foreach (@$wallets);

    my $manager = Direct::Model::Wallet::Manager->new(items => $wallets);

    do_in_transaction {
        $manager->delete_autopay();
        $manager->update();
    };

    return 0;
}

=head2 can_client_use_autopay

    Имеет ли клиент возможность устанавливать автоплатеж.
    На входе:
        class -> Direct::Wallets
        rbac  -> RBAC2::Extended
        uid   -> uid пользователя (можно представителя)
    На выходе:
        0 - не может
        1 - может

=cut

sub can_client_use_autopay {
    my ($class, $rbac, $uid) = @_;

    my $chief_uid = rbac_get_chief_rep_of_client_rep($uid);
    my $client_id = get_clientid(uid => $chief_uid);
    my $client_data = get_client_data($client_id, [qw/country_region_id work_currency/]);

    # Проверяем валюту клиента
    return 0 if (none {$client_data->{work_currency} eq $_} @Direct::Wallets::Payment::AUTOPAY_CURRENCIES);

    # Проверяем страну клиента
    return 0 if (none {$client_data->{country_region_id} eq $_} ($geo_regions::RUS));

    my $campaigns = get_all_sql(PPC(uid => $chief_uid),
            ['SELECT cid, AgencyID, statusModerate FROM campaigns', where => {uid => $chief_uid, type => get_camp_kind_types('web_edit_base')}]);

    # Проверяем пункт "не менявшие канал обслуживания по программе лояльности"
    return 0 if any {$_->{AgencyID}} @$campaigns;

    # Проверяем пункт "прямые клиенты", т.е. агентствам нельзя
    return 0 if (rbac_who_is($rbac, $chief_uid) eq 'agency');

    return 1;
}

=head2 is_user_paymethod_id

    Проверка является ли карта/кошелек (paymethod_id) собственностью пользователя или это не его?
    Требуется при отвязки карты и сохранений настроек автопополнения.
    На входе:
        class -> Direct::Wallets
        rbac  -> RBAC2::Extended
        uid   -> uid пользователя (можно представителя)
        paymethod_type -> тип метода платежа
        paymethod_id -> ID метода платежа
        ip    -> ip пользователя
    На выходе:
        0 - не его
        1 - его

=cut
sub is_user_paymethod_id {
    my ($class, $rbac, $uid, $paymethod_type, $paymethod_id, $ip) = @_;

    return 0 if any {! defined $_} ($rbac, $uid, $ip, $paymethod_type, $paymethod_id);
    my $available_paymethod_types = $class->get_list_payment_methods($rbac, $uid, $ip, this_user_only=>1);

    return 1 if any {$_->{paymethod_type} eq $paymethod_type && $_->{paymethod_id} eq $paymethod_id} map {@$_ } values %$available_paymethod_types;

    return 0;
}

=head2 get_available_autopay_paymethod_types

    Возвращает список видов платежей, который доступны пользователю с данной валютой
    На входе:
        class -> Direct::Wallets
        currency -> валюта
    На выходе:
        ArrayRef[Str] - массив строк с названиями видов платежей

=cut

sub get_available_autopay_paymethod_types {
    my ($class, $currency) = @_;
    my %availability = (RUB => ['card', 'yandex_money']);
    return $availability{$currency} || [];
}

=head2 get_actual_client_wallet

    Для клиента, соответствующего переданному DirectContext, возвращает инстанцированный объект кошелька и рабочую валюту
    На входе:
        class -> Direct::Wallets
        context -> инстанцированный DirectContext
    На выходе:
        wallet - инстанцированный Direct::Model::Wallet
        work_currency - код рабочей валюты клиента
=cut

sub get_actual_client_wallet {
    my ($class, $context) = @_;

    my $client_currencies = Client::get_client_currencies($context->client_client_id);
    my $client_wallets = Wallet::get_wallets_by_uids([{c => $context, client_currency => $client_currencies->{work_currency}}]);
    return (undef, $client_currencies->{work_currency}) unless $client_wallets->[0] && $client_wallets->[0]->{wallet_cid};

    my $wallet = $class->get_by(campaign_id => $client_wallets->[0]->{wallet_cid})->items->[0];
    $wallet->autopay->currency($client_currencies->{work_currency}) if $wallet->has_autopay;
    $wallet->allow_pay($client_wallets->[0]->{wallet_camp}->{allow_pay});

    return ($wallet, $client_currencies->{work_currency});
}


1;
