package Direct::Wallets::Payment;

=head1 NAME

    Direct::Wallets::Payment

=head1 DESCRIPTION

    Модуль с функциями для пополнения кампании (общего счета) прямыми платежами (банковские карты, Я.Деньги)

=cut

use Direct::Modern;

use List::MoreUtils qw/any/;
use JSON;

use Yandex::HashUtils;
use Yandex::Balance qw / balance_create_request_2 balance_pay_request balance_check_request_payment /;
use Yandex::Balance::Simple qw/ balance_simple_create_basket
                                balance_simple_pay_basket
                                balance_simple_check_basket /;

use Yandex::Validate qw/is_valid_float is_valid_id/;
use Yandex::DBTools;
use Yandex::DBShards;
use Yandex::Log::Messages;

use Settings;
use Tools qw/encode_json_and_compress/;
use Currencies;
use Client;
use Campaign ();
use PrimitivesIds;
use Notification;
use RBAC2::Extended;

use Primitives;

use base qw/Exporter/;

our @EXPORT_OK = qw/
    get_last_wallet_transaction
    is_successful_transaction
    is_retriable_transaction
    is_not_enough_funds_error
    is_expired_card_error
    /;

=head1 VARS

=head2 $TLOG

    лог транзакций, типа Yandex::Log

=cut

our $TLOG;

=head2 @AUTOPAY_CURRENCIES

    список валют клиентов, доступных к автопополнению

=cut
our @AUTOPAY_CURRENCIES = qw/RUB/;

=head2 @RETRY_PAUSE_HOURS

    сколько часов нужно подождать после неудачной транзакции
    первый элемент в списке - первая повторная попытка оплаты

=cut
my @RETRY_PAUSE_HOURS = (6, 12, 24, 48);

=head2 $MAX_FAILED_TRIES

    максимальное допустимое количество неуспешных транзакций подряд (без вмешательства пользователя)

=cut
my $MAX_FAILED_TRIES = scalar(@RETRY_PAUSE_HOURS) + 1;

=head2 @RETRIABLE_BALANCE_STATUSES

    статусы ошибок транзаций от баланса, при которых через время можно посторить попытку оплаты

=cut

my @RETRIABLE_BALANCE_STATUSES = qw/not_enough_funds 
                                    payment_timeout
                                    payment_refused
                                    technical_error
                                    payment_gateway_technical_error
                                    timedout_no_success
                                    unknown_error
                                    declined_by_issuer
                                    authorization_reject/;

=head2 $BALANCE_SIMPLE_API_TIMEOUT

    таймаут запросов к Simple API баданса, секунд

=cut

our $BALANCE_SIMPLE_API_TIMEOUT //= 10;

=head2 $BALANCE_PAYMENT_TIMEOUT

    таймаут платежа (по истечению указанного времени (секунд), платеж завершается ошибкой)

=cut

our $BALANCE_PAYMENT_TIMEOUT //= 120;

=head2 $TRANSACTIONS_SELECT_SQL_PART

    Часть sql-запроса для выборки транзакций, до where

=cut

my $TRANSACTIONS_SELECT_SQL_PART  = "select id, wallet_cid, payer_uid, trust_payment_id, request_id,
                                            status, balance_status, balance_status_code, type, total_balance_tid,
                                            create_time, UNIX_TIMESTAMP() - UNIX_TIMESTAMP(create_time) as seconds_after_create
                                       from wallet_payment_transactions";

=head2 $BALANCE_OPERATOR_UID
=cut

my $BALANCE_OPERATOR_UID = 0;

=head1 SUBS

=head2 get_last_wallet_transaction

    Возвращает последнюю выполненную транзакцию по кампании, определенного типа (если указан)

=cut

sub get_last_wallet_transaction {
    my ($wallet_cid) = @_;

    croak "wallet_cid is not defined" unless $wallet_cid;

    return get_one_line_sql(PPC(cid => $wallet_cid), [$TRANSACTIONS_SELECT_SQL_PART,
                                                where => {wallet_cid => SHARD_IDS},
                                            'order by id desc 
                                                limit 1']);
}

=head2 get_previous_wallet_transaction

    Возвращает транзакцию, предыдущую к указанной, того же типа

=cut

sub get_previous_wallet_transaction {
    my $t = shift;

    for my $t_field (qw/id wallet_cid type/) {
        croak "$t_field is required transaction field: " . to_json($t) unless $t->{$t_field};
    }

    return get_one_line_sql(PPC(cid => $t->{wallet_cid}), [$TRANSACTIONS_SELECT_SQL_PART,
                                                where => {wallet_cid => SHARD_IDS,
                                                          type => $t->{type},
                                                          id__lt => $t->{id}},
                                            'order by id desc 
                                                limit 1']);

}

=head2 get_incompleted_transactions

    Возвращает массив незавершенных транзакций, по которым требуется проверка статуса

=cut


sub get_incompleted_transactions {
    my ($shard, $only_cids) = @_;

    return get_all_sql(PPC(shard => $shard), [$TRANSACTIONS_SELECT_SQL_PART,
                                              where => {status__not_in => [qw/Done Error/],
                                                        ($only_cids && @$only_cids ? (wallet_cid => $only_cids) : ())} ]);
}

=head2 is_completed_transaction

    завершена ли транзакция

=cut

sub is_completed_transaction {
    my $t = shift;

    return any { $t->{status} eq $_ } qw/Done Error/;
}

=head2 is_successful_transaction

    успешна ли транзакция

=cut

sub is_successful_transaction {
    my $t = shift;

    return $t->{status} eq 'Done' ? 1 : 0;
}

=head2 is_retriable_transaction 

    ошибка в транзакции допускает повторные транзакции по кампании, 

=cut

sub is_retriable_transaction {
    my $t = shift;

    return $t->{status} eq 'Error' &&
           (($t->{balance_status_code} && any { $t->{balance_status_code} eq $_ } @RETRIABLE_BALANCE_STATUSES)
           || $t->{balance_status} eq 'payment_timeout'
           || $t->{balance_status} eq 'no_payment');
}

=head2 is_not_enough_funds_error

    транзакция завершилась ошибкой "недостаточно средств на карте или кошельке"

=cut

sub is_not_enough_funds_error {
    my $t = shift;

    return $t->{status} eq 'Error' && $t->{balance_status_code} eq 'not_enough_funds' ? 1 : 0;   
}

=head2 is_expired_card_error

    транзакция завершилась ошибкой "срок действия карты истек"

=cut

sub is_expired_card_error {
    my $t = shift;

    return $t->{status} eq 'Error' && $t->{balance_status_code} eq 'expired_card' ? 1 : 0;   
}

=head2 is_time_pause_ok

    прошло ли достаточно времени с момента последней неуспешной транзакции для повторной попытки оплаты
    входные параметры:
        $t - последняя транзакция
        $tries_num - сколько неуспешных попыток уже было произведено

=cut

sub is_time_pause_ok {
    my ($t, $tries_num) = @_;

    return 0 unless $tries_num;
    
    my $pause_hours = $RETRY_PAUSE_HOURS[$tries_num-1];
    return 0 unless defined $pause_hours;

    if ($t->{seconds_after_create} >= $pause_hours * 3600) {
        return 1;
    }
    return 0;
}

=head2 get_wallet_autopay_cids

    возвращает массив wallet_cid-ов, у которых включено автопополнение

=cut

sub get_wallet_autopay_cids($;$) {
    my ($shard, $only_cids) = @_;

    return get_one_column_sql(PPC(shard => $shard), ["select wallet_cid 
                                                        from wallet_campaigns",
                                                       where => { autopay_mode__ne => 'none',
                                                                  ($only_cids && @$only_cids ? (wallet_cid => $only_cids) : ()) 
                                                                }]);
}

=head2 get_wallet_autopay_options

    возвращает настройки автоплатежа по кошельку

=cut

sub get_wallet_autopay_options {
    my $wallet_cid = shift;

    return get_one_line_sql(PPC(cid => $wallet_cid), ["select wc.wallet_cid,
                                                              wc.autopay_mode,
                                                              ap.payer_uid,
                                                              ap.paymethod_type,
                                                              ap.paymethod_id,
                                                              ap.remaining_sum,
                                                              ap.payment_sum,
                                                              ap.tries_num,
                                                              ap.person_id,
                                                              wc.total_sum,
                                                              wc.total_balance_tid,
                                                              IFNULL(c.currency, 'YND_FIXED') currency,
                                                              c.uid camp_owner_uid,
                                                              c.ProductID,
                                                              u.ClientID
                                                         from wallet_campaigns wc
                                                         join campaigns c ON c.cid = wc.wallet_cid
                                                         join users u ON u.uid = c.uid
                                                    left join autopay_settings ap ON ap.wallet_cid = wc.wallet_cid",
                                                        where => {'wc.wallet_cid' => SHARD_IDS,
                                                                  _TEXT => 'EXISTS (SELECT 1 FROM campaigns tc where tc.uid = c.uid and tc.wallet_cid = c.cid limit 1)'},
                                                       'limit 1']);
}

=head2 get_wallet_total_sum_spent

    возращает суммарные открутки по кошельку и кампаниям под ним

=cut

sub get_wallet_total_sum_spent {
    my ($wallet_cid, $wallet_uid) = @_;
    unless ($wallet_cid && $wallet_uid) {
        croak sprintf("wallet_cid and wallet_uid should be defined: wallet_cid = %s, wallet_uid = %s", $wallet_cid, $wallet_uid);
    }

    return get_one_field_sql(PPC(uid => $wallet_uid), ["select sum(sum_spent) 
                                                          from campaigns",
                                                         where => {uid => $wallet_uid,
                                                                   _OR => {
                                                                        cid => $wallet_cid,
                                                                        wallet_cid => $wallet_cid
                                                                    }
                                                                  }]);
}

=head2 get_wallet_total_sum_rest_approx

    возращает суммарные остатки по кошельку и кампаниям под ним (для проверки корректности суммы из wallet_campaigns.total_sum)

=cut

sub get_wallet_total_sum_rest_approx {
    my ($wallet_cid, $wallet_uid) = @_;
    unless ($wallet_cid && $wallet_uid) {
        croak sprintf("wallet_cid and wallet_uid should be defined: wallet_cid = %s, wallet_uid = %s", $wallet_cid, $wallet_uid);
    }

    return get_one_field_sql(PPC(uid => $wallet_uid), ["select sum(sum-sum_spent) 
                                                          from campaigns",
                                                         where => {uid => $wallet_uid,
                                                                   _OR => {
                                                                        cid => $wallet_cid,
                                                                        wallet_cid => $wallet_cid
                                                                    }
                                                                  }]);
}

=head2 make_payment_transaction

    инициирует платеж для пополнения кошелька
    входные параметры:
        wallet_cid - кампания-кошелек для пополнения
        payer_uid - от имени кого совершать платеж (должен совпадать с владельцем метода оплаты)
        paymethod_id - метод оплаты (id выданный биллингом)
        sum - сумма платежа
        currency - валюта платежа
        type - тип платежа (сейчас поддерживается только auto)
        total_balance_tid - id нотификации баланса принятое на кошелек по состоянию на сейчас
        %O - именованные параметры
            log_params - доп. параметры для логгирования в привязке к транзакции, {}
            log - объект Yandex::Log, необязательный

=cut

sub make_payment_transaction {
    my ($wallet_cid, $payer_uid, $client_id, $product_id, $paymethod_id, $person_id, $sum, $currency, $type, $total_balance_tid, %O ) = @_;

    $O{log}->out("Making payment transaction") if $O{log};

    croak "wallet_cid is not valid id" unless is_valid_id($wallet_cid);
    croak "payer_uid is not valid id" unless is_valid_id($payer_uid);
    croak "paymethod_id is not defined" unless $paymethod_id;
    croak "sum should be valid float > 0.00" unless is_valid_float($sum) && $sum > 0;
    croak "currency is not defined" unless $currency;
    croak "payment type is not correctly defined" unless $type && any { $type eq $_ } qw/auto auto2/;

    $total_balance_tid //= 0;

if ($type eq 'auto') {

    my $create_basket_options = {
        uid => $payer_uid,
        paymethod_id => $paymethod_id,
        currency => $currency, # валюта, в которой с метода оплаты клиента спишутся деньги
        payment_timeout => $BALANCE_PAYMENT_TIMEOUT,
        user_ip => "127.0.0.1",
        orders => [{ service_order_id => $wallet_cid,
                     qty => $sum, # нужно указывать в валюте кампании(!), currency на эту сумму не действует
                    }],
    };
    my $create_basket_response = balance_simple_create_basket( $create_basket_options, timeout => $BALANCE_SIMPLE_API_TIMEOUT );
    unless ($create_basket_response->{status} 
            && $create_basket_response->{status} eq 'success'
            && $create_basket_response->{trust_payment_id}) {
        if ($type eq 'auto') {
            do_sql(PPC(cid => $wallet_cid), ['update autopay_settings 
                                                 set tries_num = -1',
                                               where => {wallet_cid => SHARD_IDS} ]);
            add_notification(rbac(), 'autopay_error', {cid => $wallet_cid,
                                                      client_uid => get_uid(cid => $wallet_cid),
                                                      letter_type => 'other_error'},
                                                     {mail_fio_tt_name => 'fio'});
            
        }
        croak("Can't create transaction in balance: " . to_json($create_basket_response));
    }

    my $create_basket_log_data = ['create_basket', {wallet_cid => $wallet_cid,
                                                    payer_uid => $payer_uid,
                                                    paymethod_id => $paymethod_id,
                                                    sum => $sum,
                                                    currency => $currency,
                                                    type => $type,
                                                    trust_payment_id => $create_basket_response->{trust_payment_id},
                                                    total_balance_tid => $total_balance_tid,
                                                    $O{log_params} ? (extra_params => $O{log_params}) : (),
                                                    }
                               ];

    my $t = { wallet_cid => $wallet_cid,
              payer_uid => $payer_uid,
              trust_payment_id => $create_basket_response->{trust_payment_id},
              type => $type,
              status => 'Processing',
              total_balance_tid => $total_balance_tid,
            };
    $t->{id} = do_insert_into_table(PPC(cid => $wallet_cid), 'wallet_payment_transactions', {%$t,
                                                                                             params => encode_json_and_compress($create_basket_log_data),
                                                                                            });

    do_sql(PPC(cid => $wallet_cid), ['update autopay_settings
                                         set tries_num = tries_num+1',
                                       where => {wallet_cid => SHARD_IDS}]);

    my $transaction_log_metadata = sprintf("wallet_cid=%s,payer_uid=%s,trust_payment_id=%s", 
                                            $wallet_cid, $payer_uid, $create_basket_response->{trust_payment_id});
    transaction_log()->out(sprintf("%s\t%s", $transaction_log_metadata, to_json($create_basket_log_data)));
    $O{log}->out($create_basket_log_data) if $O{log};

    my $pay_basket_options = {
        uid => $t->{payer_uid},
        trust_payment_id => $t->{trust_payment_id},
        user_ip => "127.0.0.1",
    };
    my $pay_basket_response = balance_simple_pay_basket( $pay_basket_options, timeout => $BALANCE_SIMPLE_API_TIMEOUT);
    my $pay_basket_log_data = ['pay_basket', {request => $pay_basket_options, 
                                              response => hash_cut($pay_basket_response, qw/status status_code/) }];
    transaction_log()->out(sprintf("%s\t%s", $transaction_log_metadata, to_json($pay_basket_log_data)));
    $O{log}->out($pay_basket_log_data) if $O{log};

    process_transaction_status($t,
                               hash_cut($pay_basket_response, qw/status status_code/),
                               log => $O{log});
}

    if ($type eq 'auto2') {

        my $product = product_info(ProductID => $product_id);

        my $create_request_options = {
            ServiceID => $product->{EngineID},
            ServiceOrderID => $wallet_cid,
            Qty => $sum
        };

         my $request_params = {
            AdjustQty => 0
        };

        eval {
            my $create_request_response = balance_create_request_2($payer_uid, $client_id, [$create_request_options], $request_params);

            my $RequestID = $create_request_response->[0]->{RequestID};

            my $t = {
                wallet_cid => $wallet_cid,
                payer_uid => $payer_uid,
                request_id => $RequestID,
                type => $type,
                status => 'Processing',
                total_balance_tid => $total_balance_tid,
            };


            my $create_request_log_data = ['create_request', {
                %$t,
                paymethod_id => $paymethod_id,
                sum => $sum,
                ($O{log_params} ? (extra_params => $O{log_params}) : ())
            }];

            my $transaction_log_metadata = sprintf("wallet_cid=%s,payer_uid=%s,request_id=%s", 
                                                    $wallet_cid, $payer_uid, $RequestID);
            transaction_log()->out(sprintf("%s\t%s", $transaction_log_metadata, to_json($create_request_log_data)));
            $O{log}->out($create_request_log_data) if $O{log};

            my $pay_request_options = {
                RequestID => $RequestID,
                PaymentMethodID => $paymethod_id,
                Currency => $currency,
                PersonID => $person_id,
                PaymentMode => "recurring"
            };

            # resp_code в ответе Balance.PayRequest - https://wiki.yandex-team.ru/Trust/Payments/RC/, resp_desc - произвольное описание
            my $balance_pay_request_response = balance_pay_request($payer_uid, $pay_request_options);

            $t->{id} = do_insert_into_table(PPC(cid => $wallet_cid), 'wallet_payment_transactions',
                {%$t, params => encode_json_and_compress($create_request_log_data)});

            do_sql(PPC(cid => $wallet_cid), ['update autopay_settings
                                                 set tries_num = tries_num+1',
                                               where => {wallet_cid => SHARD_IDS} ]);

            my $pay_request_status = $balance_pay_request_response->[0]->{resp_code};

            my $pay_request_log_data = ['pay_request', {request => $pay_request_options, 
                                                      response => $pay_request_status }];
            transaction_log()->out(sprintf("%s\t%s", $transaction_log_metadata, to_json($pay_request_log_data)));
            $O{log}->out($pay_request_log_data) if $O{log};

            process_transaction_status($t,
                                       { status => $pay_request_status, status_code => $pay_request_status },
                                       log => $O{log});
        };

        if ($@) {
            my $error = $@;

            do_sql(PPC(cid => $wallet_cid), ['update autopay_settings 
                                                 set tries_num = -1',
                                               where => {wallet_cid => SHARD_IDS} ]);
            add_notification(undef, 'autopay_error', {
                cid => $wallet_cid,
                client_uid => get_uid(cid => $wallet_cid),
                letter_type => 'other_error'
            }, { mail_fio_tt_name => 'fio' });

            croak("Can't create transaction in balance: " . $error);
        }
    }
}

=head2 process_transaction_status

    Обрабатывает статусы транзакции, полученные от Баланса, и сохраняет их в БД
    входные параметры:
        t - транзакция, с обязательными ключами:
            wallet_cid - кампания-кошелек по которой была транзакция
            trust_payment_id - id транзакции/платежа от баланса
            type - тип платежа
            status - текущий статус транзакции
        balance_resp - ответ баланса, со статусами: {status => ..., status_code => ...}
        %O - именованные параметры
            log - объект Yandex::Log, необязательный


=cut

sub process_transaction_status($$;%) {
    my ($t, $balance_resp, %O) = @_;

    $O{log}->out("Processing transaction statuses") if $O{log};

    my $status;
if ($t->{type} eq 'auto') {
    croak "balance_resp->{status} should be defined" unless $balance_resp->{status};

    for my $t_field (qw/wallet_cid trust_payment_id type status/) {
        croak "$t_field is required transaction field: " . to_json($t) unless $t->{$t_field};
    }

    if ($balance_resp->{status} eq 'wait_for_notification') {
        $status = 'Processing';
    } elsif ($balance_resp->{status} eq 'success') {
        $status = 'Done';
    } else{
        $status = 'Error';
    }
}

    if ($t->{type} eq 'auto2') {
        if (!$balance_resp->{status}) {
            $status = 'Processing';
        } elsif ($balance_resp->{status} eq 'success') {
            $status = 'Done';
        } else{
            $status = 'Error';
        }
    }

    my %statuses_to_update = (status => $status,
                              balance_status => $balance_resp->{status} // '',
                              balance_status_code => $balance_resp->{status_code} // '', );

    my $status_changed = 0;
    for (keys %statuses_to_update) {
        $status_changed = 1 if $statuses_to_update{$_} ne ($t->{$_} // '');
    }
    
    if ($status_changed) {
        hash_merge $t, \%statuses_to_update;
        $O{log}->out('Updating transaction statuses: ' . to_json($t)) if $O{log};
        do_update_table(PPC(cid => $t->{wallet_cid}), 'wallet_payment_transactions',
                                                 \%statuses_to_update,
                                                 where => {wallet_cid => SHARD_IDS,
                                                            ($t->{type} eq 'auto2' ?
                                                                ("request_id" => $t->{request_id}, "trust_payment_id__is_null" => 1)
                                                              : ("trust_payment_id" => $t->{trust_payment_id}, "request_id__is_null" => 1) ),
                                                        });
        if (is_successful_transaction($t)) {
            do_sql(PPC(cid => $t->{wallet_cid}), ['update autopay_settings 
                                                      set tries_num = 0',
                                                    where => {wallet_cid => SHARD_IDS} ]);
        }
        notifications_on_transaction_status_change($t);
    } else {
        $O{log}->out('Transaction statuses not changed: ' . to_json($t)) if $O{log};
    }
}

=head2 notifications_on_transaction_status_change

    шлет необходимые уведомления при изменении статусов транзакции

=cut

sub notifications_on_transaction_status_change($) {
    my $t = shift;

        my $letter_type = _get_transaction_letter_type($t);
        if ($letter_type) {
            my $autopay_opts = get_wallet_autopay_options($t->{wallet_cid});
            my $send_letter = 0;
            if ($autopay_opts->{tries_num} == 0) {
                # вероятно пользователь уже нажал кнопку Возобновить, и в письме про остановку автопополнения смысла нет
            } elsif ($autopay_opts->{tries_num} == 1) {
                # первая попытка после успешной транзакии, или после возобновления автопополнения пользователем
                $send_letter = 1;
            } else {
                my $prev_t = get_previous_wallet_transaction($t);
                if (!$prev_t || $letter_type ne _get_transaction_letter_type($prev_t)) {
                    $send_letter = 1;
                }
            }

            add_notification(rbac(), 'autopay_error', {cid => $t->{wallet_cid},
                                                       client_uid => $autopay_opts->{camp_owner_uid},
                                                       letter_type => $letter_type},
                                                      {mail_fio_tt_name => 'fio'}) if $send_letter;
        }
}

=head2 _get_autopay_transaction_letter_type

    по транзакции возвращает письмо какого типа по ней можно отправить

=cut

sub _get_transaction_letter_type {
    my $t = shift;

    if (is_completed_transaction($t) 
            && !is_successful_transaction($t)) {
        if (is_not_enough_funds_error($t)) {
            # недостаточно средств
            return 'not_enough_funds_error';
        } elsif (is_expired_card_error($t)) {
            # срок действия карты истек
            return 'expired_card_error';
        } else {
            # другие ошибки
            return 'other_error';
        }
    }
    return '';
}

=head2 get_and_process_transaction_status

    Запрашивает в балансе, и сохраняет в БД статус транзакции
    входные пераметры:
        $t - транзакция, {}
        %O - именованные параметры
            log - объект Yandex::Log, необязательный

=cut

sub get_and_process_transaction_status($;%) {
    my ($t, %O) = @_;

    $O{log}->out("Getting transaction statuses from balance") if $O{log};


if ($t->{type} eq 'auto') {
    my $check_basket_options = {
        uid => $t->{payer_uid},
        trust_payment_id => $t->{trust_payment_id},
        user_ip => "127.0.0.1",
    };

    my $check_basket_response = balance_simple_check_basket( $check_basket_options, timeout => $BALANCE_SIMPLE_API_TIMEOUT );
    my $transaction_log_metadata = sprintf("wallet_cid=%s,payer_uid=%s,trust_payment_id=%s", @{$t}{qw/wallet_cid payer_uid trust_payment_id/});
    my $check_basket_log_data = ['check_basket', {request => $check_basket_options, 
                                                  response => hash_cut($check_basket_response, qw/status status_code/) }];
    transaction_log()->out(sprintf("%s\t%s", $transaction_log_metadata, to_json($check_basket_log_data)));
    $O{log}->out($check_basket_log_data) if $O{log};

    process_transaction_status($t,
                               hash_cut($check_basket_response, qw/status status_code/),
                               log => $O{log});
}


    if ($t->{type} eq 'auto2') {

        my $product = product_info(cid => $t->{wallet_cid});
        my $ServiceID = $product->{EngineID};

        my $check_request_payment_response = balance_check_request_payment($t->{payer_uid}, $t->{request_id}, $ServiceID);

        my $resp_code = $check_request_payment_response->[0]->{resp_code};

        my $transaction_log_metadata = sprintf("wallet_cid=%s,payer_uid=%s,request_id=%s", @{$t}{qw/wallet_cid payer_uid request_id/});
        my $check_request_payment_log_data = ['check_request_payment', {request => [$t->{payer_uid}, $t->{request_id}, $ServiceID],
                                                      response => $resp_code }];
        transaction_log()->out(sprintf("%s\t%s", $transaction_log_metadata, to_json($check_request_payment_log_data)));
        $O{log}->out($check_request_payment_log_data) if $O{log};

        process_transaction_status($t,
                                   { status => $resp_code, status_code => $resp_code },
                                   log => $O{log});
    }
}

=head2 check_and_process_wallet_autopay

    проверяет, выполняется ли условие автоплатежа по кампании, и в случае необходимости - инициирует платеж
    входные параметры:
        wallet_cid - кампания-кошелек, которую надо проверить
        %O - именованные параметры
            log - объект Yandex::Log, обязательный
            force_create_camp_balance - отправлять кампанию-кошелек в баланс перед каждой оплатой
            dont_wait_notification - не ждать нотификацию после успешной транзакции
    выходные параметры (списком):
        0|1 - была ли инициирована оплата
        'Error' - непредвиденная ошибка, если такая имела место быть

=cut

sub check_and_process_wallet_autopay($;%) {
    my ($wallet_cid, %O) = @_;

    $O{log}->out("Processing wallet autopay settings");
    my $opts = get_wallet_autopay_options($wallet_cid);
    $O{log}->out($opts);

    my $new_autopay_allowed = Client::ClientFeatures::has_new_autopay_type($opts->{ClientID});

    my $autopay_type;
    if (!$opts->{person_id} || !$new_autopay_allowed || ($opts->{paymethod_type} eq 'yandex_money')) {
        $autopay_type = 'auto';
    } else {
        $autopay_type = 'auto2';
    }

    $O{log}->out("Got person_id ". ($opts->{person_id} // "undef") . " and allowed new autopay: $new_autopay_allowed; result autopay type - $autopay_type");

    unless ($opts 
            && $opts->{autopay_mode} eq 'min_balance'
            && any { $opts->{currency} eq $_ } @AUTOPAY_CURRENCIES) {
        $O{log}->out("Autopay for this wallet is not possible");
        return 0;
    }

    if ($opts->{tries_num} < 0) {
        $O{log}->out("tries_num < 0, wait for user actions");
        return 0;
    }

    if ($opts->{total_sum} < $Currencies::EPSILON && !$opts->{total_balance_tid} &&
        get_wallet_total_sum_rest_approx($wallet_cid, $opts->{camp_owner_uid}) > 0) {
        $O{log}->out("total_sum=0, total_balance_tid=0, sum_rest_under_wallet>0, wait wallet notification");
        return 0;
    }

    if (Campaign::check_method_pay($wallet_cid) eq 'with_block') {
        $O{log}->out("Available camp method pay: with_block, wait for camps under wallet moderation");
        return 0;
    }

    my $t = get_last_wallet_transaction($wallet_cid);

    if ($t) {
        $O{log}->out("Last transaction: " . to_json($t));
        if (!is_completed_transaction($t)) {
            $O{log}->out("Last transaction not completed");
            return 0;
        }

        if (!is_successful_transaction($t)) {
            if ($opts->{tries_num}) { # если tries_num == 0, то последнюю транзакцию проверяем только на завершенность
                if ($opts->{tries_num} >= $MAX_FAILED_TRIES) {
                    $O{log}->out("Limit for failed transactions reached");
                    return 0;
                }
                if (!is_retriable_transaction($t)) {
                    $O{log}->out("Last failed transaction is not retriable (wait for user actions)");
                    return 0;
                }
                if (!is_time_pause_ok($t, $opts->{tries_num})) {
                    $O{log}->out("Not enough time elapsed for next payment retry");
                    return 0;
                }
            }
        } elsif ($opts->{total_balance_tid} <= $t->{total_balance_tid} && !$O{dont_wait_notification}) {
            $O{log}->out("total_balance_tid not changed after successful transaction, wait wallet notification");
            return 0;
        }
    }

    $opts->{total_sum_spent} =  eval { get_wallet_total_sum_spent($wallet_cid, $opts->{camp_owner_uid}) };
    $O{log}->die($@) if $@;

    my $client_nds = get_client_NDS($opts->{ClientID});
    $opts->{remaining_sum_with_nds} = add_nds($opts->{remaining_sum}, $client_nds);
    $opts->{payment_sum_with_nds} = add_nds($opts->{payment_sum}, $client_nds);

    $O{log}->out(hash_cut($opts, qw/total_sum total_sum_spent/));
    if (($opts->{total_sum} - $opts->{total_sum_spent}) - $opts->{remaining_sum_with_nds} <= $Currencies::EPSILON) {
        # если есть подозрение что кошелек не уходил в баланс, или не уходил с актуальным статусом модерации - отправляем его в баланс
        if ($opts->{total_sum} == 0 || $O{force_create_camp_balance}) {
            my $error;
            my $camp_balance_response = eval { Campaign::create_campaigns_balance( rbac(), $BALANCE_OPERATOR_UID, [$wallet_cid] ) };
            if ($@) {
                $error = $@;
            } elsif ($camp_balance_response->{error} || !$camp_balance_response->{balance_res}) {
                $error = $camp_balance_response;
            }
            if ($error) {
                $O{log}->out("Error while sending wallet to balance: ", $error);
                return 0, 'Error';
            }
        }

        # инициируем пополнение
        eval {
            make_payment_transaction($wallet_cid, 
                                     $opts->{payer_uid}, 
                                     $opts->{ClientID},
                                     $opts->{ProductID},
                                     $opts->{paymethod_id},
                                     $opts->{person_id},
                                     $opts->{payment_sum_with_nds},
                                     $opts->{currency},
                                     $autopay_type,
                                     $opts->{total_balance_tid},
                                     log => $O{log},
                                     log_params => $opts, );
        };
        if ($@) {
            $O{log}->out("Error while making payment transaction: ", $@);
            return 0, 'Error';
        }
        return 1;
    }
    return 0;
}

=head2 transaction_log

    возвращает объект Yandex::Log для лога транзакций
    при необходимости - инициализирует

=cut

sub transaction_log {
    $TLOG //= new Yandex::Log::Messages();
    $TLOG->msg_prefix("payment_transactions");
    return $TLOG;
}

=head2 rbac

    возвращает объекст RBAC

=cut

sub rbac {
    state $rbac //= RBAC2::Extended->get_singleton(1);
    return $rbac;
}

1;
