package Direct::XLS::Validation;

use Direct::Modern;
use Mouse;
use Yandex::I18n;

use TextTools;

with 'Direct::Role::ValidationResultConsumer';

=head2 decorate_error
    Вызывается как метод класса. Принимает на вход hashref с данными и текст ошибки.
    При наличии в данных group_name, bid или line_number добавляет соответствующие префиксы к тексту ошибки.

    Возвращает полученный текст.
=cut

sub decorate_error {
    my ($class, $data, $err_msg) = @_;

    my @prefix;
    push @prefix, iget('Группа объявлений "%s":', $data->{group_name}) if ($data->{group_name} // '') =~ /\S/;
    push @prefix, iget('Объявление M-%s:', $data->{bid}) if $data->{bid};
    push @prefix, iget('Строка %s:', $data->{line_number}) if $data->{line_number};
    
    return join ' ', @prefix, $err_msg;
}

=head2 get_errors_and_warnings_for_xls
    Получает на вход структуру данных, для которой выполнялась валидация посредством текщего инстанса Direct::XLS::Validation.
    Обходит имеющийся в инстансе ValidationResult и формирует плоские списки ошибок и предупреждений,
    дополняя тексты номерами строк, объявлений и именами групп из полученой структуры.

=cut

sub get_errors_and_warnings_for_xls {
    my ($self, $data) = @_;
    
    my $vr = $self->validation_result;
    
    my (@errors, @warnings);
    # Общие ошибки на все группы
    push @errors, map { +{priority => 0, error => $_->description} } @{$vr->get_generic_errors};

    # Ошибки по каждой группе
    for (my $i = 0; $i < @$data; $i++) {
        my $adgroup = $data->[$i];
        my $vr_adgroup = $vr->nested_objects->[$i];
        my ($vr_banners, $vr_keywords, $vr_retargetings) = delete @{$vr_adgroup->defects_by_field}{qw/banners keywords retargetings/};

        my (@adgroup_prefix_name, @adgroup_prefix_line);
        push @adgroup_prefix_name, iget('Группа объявлений "%s":', $adgroup->{group_name}) if ($adgroup->{group_name} // '') =~ /\S/;
        push @adgroup_prefix_line, iget('Строка %s:', $adgroup->{line_number});

        my @adgroup_prefix = (@adgroup_prefix_name, @adgroup_prefix_line);

        # Ошибки на группу
        push @errors, map { +{
            priority => 10,
            line_number => $adgroup->{line_number},
            error => join(" ", @adgroup_prefix_name, @adgroup_prefix_line, $_),
        } } @{$vr_adgroup->get_error_descriptions};

        # Ошибки на баннеры
        if (!$vr_banners->is_valid) {
            my @banner_lines = map { $_->{line_number} } @{$adgroup->{banners}};
            my $banner_line_numbers = @banner_lines > 1
                ? iget('Строки %s:', array_to_compact_str(@banner_lines))
                : iget('Строка %s:', array_to_compact_str(@banner_lines));

            # Общие на все баннеры
            push @errors, map { +{
                priority => 10,
                line_number => $adgroup->{line_number},
                error => join(' ', @adgroup_prefix_name, $banner_line_numbers, $_->description),
            } } @{$vr_banners->get_generic_errors};

            # По каждому баннеру
            for (my $j = 0; $j < @{$adgroup->{banners}}; $j++) {
                my $banner = $adgroup->{banners}->[$j];
                my $vr_banner = $vr_banners->nested_objects->[$j];

                my @banner_prefix;
                push @banner_prefix, iget('Объявление M-%s:', $banner->{bid}) if $banner->{bid};
                push @banner_prefix, iget('Строка %s:', $banner->{line_number});

                push @errors, map { +{
                    priority => 10,
                    line_number => $banner->{line_number},
                    error => join(" ", @banner_prefix, $_),
                } } @{$vr_banner->get_error_descriptions};
            }
        }

        # Ошибки на ключевые фразы
        if (!$vr_keywords->is_valid) {
            # Общие на все фразы
            # Пропустим ошибку про превышение длины фраз
            push @errors, map { +{
                priority => 10,
                line_number => $adgroup->{line_number},
                error => join(' ', @adgroup_prefix_name, $_->description),
            } } grep { $_->name ne 'ReachLimit' } @{$vr_keywords->get_generic_errors};

            for (my $j = 0; $j < @{$adgroup->{phrases}}; $j++) {
                my $keyword = $adgroup->{phrases}->[$j];
                my $vr_keyword = $vr_keywords->nested_objects->[$j];

                push @errors, map { +{
                    priority => 10,
                    line_number => $keyword->{line_number},
                    error => join(' ', @adgroup_prefix_name, iget('Строка %s:', $keyword->{line_number}), $_),
                } } @{$vr_keyword->get_error_descriptions};
            }
        }

        # Ошибки на ретаргетинг
        if (!$vr_retargetings->is_valid) {
            # Общие на весь ретаргетинг
            push @errors, map { +{
                priority => 10,
                line_number => $adgroup->{line_number},
                error => join(" ", @adgroup_prefix_name, $_->description),
            } } @{$vr_retargetings->get_generic_errors};

            for (my $j = 0; $j < @{$adgroup->{retargetings}}; $j++) {
                my $retargeting = $adgroup->{retargetings}->[$j];
                my $vr_retargeting = $vr_retargetings->nested_objects->[$j];

                push @errors, map { +{
                    priority => 10,
                    line_number => $retargeting->{line_number},
                    error => join(' ', @adgroup_prefix_name, iget('Строка %s:', $retargeting->{line_number}), $_),
                } } @{$vr_retargeting->get_error_descriptions};
            }
        }
    }
 
    return {
        errors => [
            map { $_->{error} . ($_->{error} !~ /\.$/ ? '.' : '') }
            sort { $a->{priority} <=> $b->{priority} or $a->{line_number} <=> $b->{line_number} } @errors
        ],
        warnings => [
            map { $_->{warning} . ($_->{warning} !~ /\.$/ ? '.' : '') }
            sort { $a->{line_number} <=> $b->{line_number} } @warnings
        ],
    };
}

sub _validate { return shift }

1;
