=head1 NAME

Direct::YT::account_score_yt;

=head1 DESCRIPTION

Функции для расчёта показателя качества аккаунта.
Код (был) рассчитан на исполнение на кластере YT, поэтому не должен зависеть от среды.

Реализация и константы должны совпадать с YQL-версией:
https://a.yandex-team.ru/arc/trunk/arcadia/direct/jobs/src/main/resources/accountscore/CalcAccountScoreFactors.yql
https://a.yandex-team.ru/arc/trunk/arcadia/direct/jobs/src/main/java/ru/yandex/direct/jobs/accountscore/CalcFactorsJob.java?rev=5017803#L45


=cut

package Direct::YT::account_score_yt;

use strict;
use warnings;

use utf8;

use parent 'Exporter';

our @EXPORT_OK = qw/

    calc_factors_index
    calc_score_by_factors
    $STOP_DAYS_STAT_LENGTH

/;

use List::Util qw/sum/;

# Максимальный рейтинг.
our $MAX_SCORE = 10;

# За сколько дней считать данные о днях простоя.
# NB! должно совпадать со значением в Java: https://a.yandex-team.ru/arc/trunk/arcadia/direct/jobs/src/main/java/ru/yandex/direct/jobs/accountscore/CalcFactorsJob.java?rev=5017803#L45
our $STOP_DAYS_STAT_LENGTH = 30;
# Веса факторов
our %FACTORS_DATA = ( broadmatch_percent   => {coef => 0.4, verdict_name => "broadmatch"}, # Дополнительные релевантные фразы
                     stop_days_count      => {coef => 3.0, verdict_name => "stop_days"}, # Число дней простоя
                     avg_banners_count    => {coef => 1.0, verdict_name => "avg_phrases"}, # Среднее количество объявлений в кампаниях
                     avg_phrases_count    => {coef => 1.4, verdict_name => "avg_phrases"}, # Количество ключевых фраз на баннер
                     sitelinks_percent    => {coef => 1.3, verdict_name => "sitelinks"}, # Дополнительные ссылки
                     images_percent       => {coef => 0.5, verdict_name => "images"}, # Картинка на РСЯ
                     ctx_price_coef_percent => {coef => 1.3, verdict_name => "ctx_price_coef"}, # Доля бюджета на РСЯ
                     separate_place_count => {coef => 0.6, verdict_name => "separate_place"}, # Отдельные кампании на РСЯ
                     vcards_count         => {coef => 0.5, verdict_name => "vcards"}, # Визитка
                     );
# в FACTORS_DATA verdict_name не случайно дважды имеет значение avg_phrases (у avg_banners_count и avg_phrases_count),
# это склейка флагов на выдачу пользователям вердиктов. На два вердикта надо написать только один текст.

=head2 calc_factors_index

    Посчитать индексы факторов. Требуется для подсчета рейтинга и для подсчета вердиктов

=cut

sub calc_factors_index {
    my $factors = shift;
    my $res = {};

    # Число дней простоя
    $res->{stop_days_count} = $factors->{stop_days_fine};
    # DIRECT-34272 - Исправить фактор "дни простоя"
    # Если данные были собраны ДО выпуска изменений, то считаем по-старому.
    if (!defined $factors->{stop_days_fine} && defined $factors->{shows_days_count} ) {
       # Если нет ни одного дня показа, то считаем, что у него максимальный штраф, то есть 21 день простоя.
        my $stop_days = $STOP_DAYS_STAT_LENGTH - $factors->{shows_days_count};
        $res->{stop_days_count} = ($stop_days <= 21 && $factors->{shows_days_count} > 0) ? $stop_days / 21 : 0;
    }
    # Среднее количество объявлений в кампаниях
    $res->{avg_banners_count} = ($factors->{avg_banners_count} <= 2) ? 1 : 0;

    # Количество ключевых фраз на баннер
    $res->{avg_phrases_count} = ($factors->{avg_phrases_count} > 20) ? 1 : ($factors->{avg_phrases_count} > 10) ? 0.8 : 0;

    # Визитка
    $res->{vcards_count} = ($factors->{vcards_count}) ? 0 : 1;

    # Дополнительные релевантные фразы
    $res->{broadmatch_percent} = ($factors->{broadmatch_percent} >= 20) ? 0 : 1;

    # Дополнительные ссылки
    $res->{sitelinks_percent} = ($factors->{sitelinks_percent} >= 20) ? 0 : 1;

    # Картинка на РСЯ
    $res->{images_percent} = ($factors->{images_percent} >= 20) ? 0 : 1;

    # Доля бюджета на РСЯ
    $res->{ctx_price_coef_percent} = ($factors->{ctx_price_coef_percent} > 0 && $factors->{ctx_price_coef_percent} < 16) ? 0.5 : 0;

    # Отдельные кампании на РСЯ
    $res->{separate_place_count} = ($factors->{separate_place_count}) ? 0 : 1;

    return $res;
}

=head2 calc_score_by_factors

    Посчитать рейтинг по индексам факторов
    на входе:
        хеш с факторами (после calc_factors_index)
    на выходе:
        число, означающее рейтинг

=cut
sub calc_score_by_factors {
    my $factors = shift;

    return $MAX_SCORE - sum (map {$factors->{$_} * $FACTORS_DATA{$_}->{coef}} keys %FACTORS_DATA);

}

1;
