package Direct::YT::monitor_stats::base;

=head1 NAME

    Direct::YT::monitor_stats::base - базовый класс для всех статистик

=head1 DESCRIPTION

    разбираться лучше всего на примере Direct::YT::monitor_stats::banners

=cut
    

use strict;
use warnings;


=head2 new()

=cut
sub new {
    my ($class) = @_;
    my $self = bless { 
        _stat => {
            sum => {},
            avg => {},
            count_distinct => {},
        },
    }, $class;
    return $self;
}


=head1 Кастомизация

=head2 prefix()

    префикс для всех производимых метрик, по-умолчанию генерируется из имени пакета,
    можно переопределять в наследниках

=cut 
sub prefix {
    my ($self) = @_;
    (my $prefix = ref($self)) =~ s/^Direct::YT::monitor_stats:://;
    return $prefix eq '' ? "" : "$prefix.";
}


=head2 metrics()

    Функцию нужно переопределить так, чтобы возвращалась ссылка на хэш:
    имя_метрики => описание

=cut
sub metrics {
    return {};
}

=head2 used_fields()

    Функцию нужно переопределить так, чтобы возвращалась ссылка на хэш,
    где ключи таблицы, а значения - список полей, нужных для работы
    Допустимые таблицы: campaigns, vcards, phrases, banners, bids, hierarchical_multipliers

=cut
sub used_fields {
    return {};
}



=head1 Hook-и для обработки объектов

    В дочерних классах нужно переопределить один или несколько методов для обработки приходящих объектов
    в объектах точно есть поля, которые указаны в used_fields, но помимо них могут быть и поля, нужные другим обработчикам
    один и тот же объект передаётся всем обработчикам, поэтому менять его нельзя

    порядок вызова обработчиков:
    on_vcard*, (on_adgroup, on_banner*, on_phrase*)*, on_campaign

=head2 on_adgroup($adgroup)

    Вызывается для каждой собранной целиком adgroup-ы,
    Помимо полей из used_fields, добавляются поля
    campaign - ссылка на объект кампания
    banners - ссылка на массив из всех баннеров группы
    phrases - ссылка на массив из всех фраз группы
    hierarchical_multipliers - ссылка на хэш из множителей ставок, ключ тип, значение - структура из соответствующей таблички

=head2 on_vcard($vcard)

    вызывается для каждой визитки, принадлежащей кампании (не обязательно привязанной)
    дополнительное поле - campaign, ссылка на кампанию

=head2 on_banner($banner)

    дополнительное поле - adgroup, ссылка на группу объявлений

=head2 on_phrase($phrase)

    дополнительное поле - adgroup, ссылка на группу объявлений

=head2 on_campaign($camp)

    дополнительное поле - hierarchical_multipliers - ссылка на хэш из множителей ставок, ключ тип, значение - структура из соответствующей таблички

=head2 state($camp)

    Иногда возникает потребность, при обработке кампании сохранить некоторую информацию,
    чтобы использовать при обработке кампании (или беннера, фразы, ...)

    Информацию можно сохрянать в хэше, возвращаемом state
    Пример:
    sub on_adgroup {
        my ($self, $adgroup) = @_;
        # считаем количество adgroup в кампании
        $self->state($adgroup->{campaign})->{adgroups_cnt}++;  
    sub on_campaign {
        my ($self, $camp) = @_;
        $self->avg(adgroups_per_camp => $self->state($camp)->{adgroups_cnt})

=cut
sub state {
    my ($self, $obj) = @_;
    return $obj->{_state}->{ref $self} ||= {};
}



=head1 Аггрегирующие функции а-ля SQL

=head2 sum(metric1 => cnt1, ...)

    Аналог sql-ных SUM и COUNT

=cut
sub sum {
    my ($self, @kvs) = @_;
    while(my ($k, $v) = splice @kvs, 0, 2) {
        $self->{_stat}->{sum}->{$k} += $v;
    }
}


=head2 avg(metric1 => cnt1, ...)

    Аналог sql-ного AVG

=cut
sub avg {
    my ($self, @kvs) = @_;
    while(my ($k, $v) = splice @kvs, 0, 2) {
        my $d = $self->{_stat}->{avg}->{$k} ||= [0, 0];
        $d->[0] += $v;
        $d->[1] += 1;
    }
}


=head2 count_distinct(metric1 => uniq1, ...)

    Аналог sql-ного COUNT(DISTINCT uniq1)

=cut
sub count_distinct {
    my ($self, @kvs) = @_;
    while(my ($k, $v) = splice @kvs, 0, 2) {
        $self->{_stat}->{count_distinct}->{$k}->{$v} = undef;
    }
}


=head1 Внутренние функции, которые скорее всего не нужно переопределять

=head2 stat_cnt()

    вернуть воличество закешированных метрик,
    используется для определения времени сброса данных (flush)

=cut
sub stat_cnt {
    my ($self) = @_;
    my $cnt = 0;

    $cnt += keys %{$self->{_stat}->{sum}};

    $cnt += keys %{$self->{_stat}->{avg}};

    while(my ($k, $v) = each %{$self->{_stat}->{count_distinct}}) {
        $cnt += keys %$v;
    }

    return $cnt;
}


=head2 flush()

    сбросить накопленные метрики, обнулить кеш

=cut
sub flush {
    my ($self, $stream) = @_;

    my $prefix = $self->prefix;
    
    while(my ($k, $v) = each %{$self->{_stat}->{sum}}) {
        $stream->yield({metric => "$prefix$k", sum => $v});
    }
    $self->{_stat}->{sum} = {};

    while(my ($k, $v) = each %{$self->{_stat}->{avg}}) {
        $stream->yield({metric => "$prefix$k", avg_sum => $v->[0], avg_cnt => $v->[1]});
    }
    $self->{_stat}->{avg} = {};

    while(my ($k, $v) = each %{$self->{_stat}->{count_distinct}}) {
        $stream->yield({metric => "$prefix$k", uniq => $_}) for keys %$v;
    }
    $self->{_stat}->{count_distinct} = {};
}

1;
