package Direct::YT::monitor_stats::hierarchical_multipliers;

=head1 NAME

    Direct::YT::monitor_stats::hierarchical_multipliers - метрики по использованию корректировок ставок

    https://st.yandex-team.ru/DIRECT-44298
    Для ретаргетинга и соцдема нужно сделать ежедневный замер по числу активных кампаний и активных групп с коэффициентами в разрезах по:
        типу кампании (самостоятельная/сервисируемая/агентская/все)
        типу корректировки (демография ретаргетинг)
        процентам увеличения (1-50, 51-100, 101-150, 151-200, 201-250, 251-300) - здесь считаем количество корректировок, а не кампаний


hierarchical_multipliers.$type.${group_or_camp}_${serv}
hierarchical_multipliers.$type.${range_name}_${serv}

=cut

use strict;
use warnings;
use utf8;
use feature 'state';

use parent 'Direct::YT::monitor_stats::base';

my @ranges = (
    {from => -100, to => -100},
    {from => -99, to => -49},
    {from => -50, to => -40},
    {from => -39, to => -30},
    {from => -29, to => -20},
    {from => -19, to => -10},
    {from => -9, to => -1},
    {from => 0, to => 0},
    {from => 1, to => 50},
    {from => 51, to => 100},
    {from => 101, to => 200},
    {from => 201, to => 300},
    {from => 301, to => 600},
    {from => 601, to => 900},
    {from => 901, to => 1200},
);

my %serv_types = (
    all => "все",
    manager => "менеджерские",
    agency => "агентские",
    self => "самоходные",
);

my %multiplier_types = (
    mobile_multiplier => {
        name => 'мобильные',
        range_from => -50,
        range_to => 1200,
        values => sub { $_[0]{multiplier_pct} },
    },
    demography_multiplier => {
        name => 'соц. дем.',
        range_from => 1,
        range_to => 300,
        values => sub { map { $_->{multiplier_pct} } @{$_[0]{values}} },
    },
    retargeting_multiplier => {
        name => 'ретаргетинговые',
        range_from => 1,
        range_to => 300,
        values => sub { map { $_->{multiplier_pct} } @{$_[0]{values}} },
    },
);

sub prefix {
    return 'hierarchical_multipliers.';
}

sub metrics {
    my %ret;
    for my $mult_type (keys %multiplier_types) {
        my $name = $multiplier_types{$mult_type}{name};
        for my $serv (keys %serv_types) {
            my $serv_desc = $serv_types{$serv};
            $ret{"${mult_type}.groups_$serv"} = "активных групп где включены $name корректировки ($serv_desc)";
            $ret{"${mult_type}.camps_$serv"} = "активных кампаний где включены $name корректировки ($serv_desc)";

            for my $range (@ranges) {
                $ret{"${mult_type}.groups_range_$range->{from}_$range->{to}_$serv"} = "$name корректировки на группу от $range->{from} до $range->{to} ($serv_desc)";
                $ret{"${mult_type}.camps_range_$range->{from}_$range->{to}_$serv"} = "$name корректировки на кампанию от $range->{from} до $range->{to} ($serv_desc)";
            }
        }
    }
    return \%ret;
}

sub used_fields {
    return {
        campaigns => [qw/statusActive AgencyUID ManagerUID mobile_multiplier_pct sum sum_spent wallet_sum wallet_sum_spent/],
        phrases => [qw/statusModerate mobile_multiplier_pct/],
        banners => [qw/statusActive/],
        hierarchical_multipliers => [qw/is_enabled multiplier_pct type values/],
    };
}

sub on_adgroup {
    my ($self, $adgroup) = @_;
    my $camp = $adgroup->{campaign};
    my $cid = $camp->{cid};
    my $pid = $adgroup->{pid};

    return unless ($camp->{sum}||0) + ($camp->{wallet_sum}||0) > ($camp->{sum_spent}||0) + ($camp->{wallet_sum_spent}||0)
        && $camp->{statusActive} eq 'Yes'
        && $adgroup->{statusModerate} eq 'Yes'
        && (grep {$_->{statusActive} eq 'Yes'} @{$adgroup->{banners}});

    $self->state($camp)->{is_good} = 1;

    my $serv_type = $camp->{AgencyUID} ? 'agency' : $camp->{ManagerUID} ? 'manager' : 'self';

    for my $mult_type (keys %{$adgroup->{hierarchical_multipliers}}) {
        next unless exists $multiplier_types{$mult_type};
        my $prefix = "${mult_type}.groups_";
        my @values = grep { defined $_ } $multiplier_types{$mult_type}{values}->($adgroup->{hierarchical_multipliers}{$mult_type});
        for my $serv ('all', $serv_type) {
            my $seen;
            for my $group_pct (@values) {
                $seen = 1;
                if (my $range = _range($group_pct)) {
                    $self->sum("${prefix}range_$range->{from}_$range->{to}_$serv" => 1);
                }
            }
            if ($seen) {
                $self->sum("${prefix}$serv" => 1);
            }
        }
    }
}

sub on_campaign {
    my ($self, $camp) = @_;
    return unless $self->state($camp)->{is_good};

    my $serv_type = $camp->{AgencyUID} ? 'agency' : $camp->{ManagerUID} ? 'manager' : 'self';

    for my $mult_type (keys %{$camp->{hierarchical_multipliers}}) {
        next unless exists $multiplier_types{$mult_type};
        my $prefix = "${mult_type}.camps_";
        my @values = grep { defined $_ } $multiplier_types{$mult_type}{values}->($camp->{hierarchical_multipliers}{$mult_type});
        for my $serv ('all', $serv_type) {
            my $seen;
            for my $group_pct (@values) {
                $seen = 1;
                if (my $range = _range($group_pct)) {
                    $self->sum("${prefix}range_$range->{from}_$range->{to}_$serv" => 1);
                }
            }
            if ($seen) {
                $self->sum("${prefix}$serv" => 1);
            }
        }
    }
}

sub _range {
    my ($pct) = @_;
    state $pct2range;
    if (!$pct2range) {
        for my $range (@ranges) {
            $pct2range->{$_} = $range for ($range->{from}+100) .. ($range->{to}+100);
        }
    }
    return defined $pct && $pct2range->{$pct} ? $pct2range->{$pct} : undef;
}

1;
